#define TIMER_MS 100 // основной (наименьший) таймер программы

// сколько циклов таймера игнорировать изменение ключа зажигания, двигателя
#define STATE_COUNTER_MAX 100 // *100 ms = 10 s

// граница между разряженным и заряженным аккумулятором: 0...1024 = 0...5 V
#define U_TRES 510 // 510/1024*5 = 2,49 вольт

// входящие байты для переключения режима (исходящие: 0 = OFF, 1 = SLEEP)
#define BYTE_OFF 20
#define BYTE_SLEEP 10

// всякие задержки
#define PSU_BEFORE_ON_DELAY_MS 5000 // перед включение блока питания
#define PC_BEFORE_ON_DELAY_MS 5000 // перед включением компьютера
#define PC_DELAY_DURING_OFF_MS 4000 // жать кнопку питания для полного отключения компьютера
#define PC_DELAY_DURING_ON_MS 1000 // жать кнопку питания для включения компьютера
#define PC_DELAY_DURING_SLEEP_MS 1000 // жать кнопку питания, чтобы компьютер заснул
#define PC_DELAY_AFTER_OFF_MS 600000 // 10 минут; ожидание корректного отключения компьютера, потом отключаем блок питания
#define PC_AFTER_ON_DELAY_MS 10000 // сколько ждать после включения компьютера, чтобы появился сигнал pinMB (иначе считаем, что компьютер не включился, и пробуем снова)

#include <Metro.h>
Metro timer; // основной таймер (100 мс)
Metro timer_off; // таймер при выключении компьютера (10 минут)

// input
const int pinU = 0; // analog
const int pinKey = 2; // digital
const int pinMB = 5; // digital
const int pinEngine = 6; // digital
const int pinPowerModeInput = 8; // digital

// output
const int pinPSU = 3; // digital, 0 = off, 1 = on
const int pinPower = 4; // digital, default 0
const int pinPowerModeOutput = 13;//7; // digital

struct {
	// входы
	int u_input;
	int key_input;
	int mb_input;
	int engine_input;
	int power_mode_input;
	
	// бизнес-состояния
	int u_state;
#define POWER_MODE_OFF 0
#define POWER_MODE_SLEEP 1
	int power_mode;
#define KEY_MODE_OFF 0
#define KEY_MODE_ACC 1
#define KEY_MODE_ON 2
	int key_mode;
	int mb_state;
	int psu_state;
#define PC_STATE_OFF 0
#define PC_STATE_ON 1
#define PC_STATE_SLEEP 2
	int pc_state;
} instant_state, prev_instant_state, stable_state;

void setup() {
	// инициализация переменных
	instant_state.u_input = 0;
	instant_state.key_input = 0;
	instant_state.mb_input = 0;
	instant_state.engine_input = 0;
	instant_state.power_mode_input = 0;
	instant_state.u_state = 0;
	instant_state.power_mode = POWER_MODE_OFF;
	instant_state.key_mode = KEY_MODE_OFF;
	instant_state.mb_state = 0;
	instant_state.psu_state = 0;
	instant_state.pc_state = PC_STATE_OFF;
	prev_instant_state = instant_state;
	stable_state = instant_state;
	
	// outputs
	pinMode(pinPSU, OUTPUT);
	digitalWrite(pinPSU, LOW);
	pinMode(pinPower, OUTPUT);
	digitalWrite(pinPower, LOW);
	pinMode(pinPowerModeOutput, OUTPUT);
	digitalWrite(pinPowerModeOutput, LOW);

	// inputs
	// pinU = analog
	pinMode(pinKey, INPUT);
	pinMode(pinMB, INPUT);
	pinMode(pinEngine, INPUT);
	pinMode(pinPowerModeInput, INPUT);
	
	Serial.begin(9600); // pins 0, 1
		
	timer = Metro();
	timer.interval(TIMER_MS);
	timer.reset();
}

int incomingByte = -1;
void loop() {
	if (timer.check()) {
		// **** Читаем входы ****
		prev_instant_state = instant_state;
		update_instant_state();
		
		// ...и ком-порт
		incomingByte = -1;
		while (Serial.available() > 0) {
			incomingByte = Serial.read();
		}
		switch (incomingByte) {
			case BYTE_OFF:
				stable_state.power_mode = POWER_MODE_OFF;
				break;
			case BYTE_SLEEP:
				stable_state.power_mode = POWER_MODE_SLEEP;
				break;
		}
		// если дребезг не учитывать, то так:
		stable_state.u_state = instant_state.u_input;
		stable_state.mb_state = instant_state.mb_input;
		if (stable_state.mb_state) { // по идее эта проверка лишняя, но пусть будет
		} else {
			stable_state.pc_state = PC_STATE_OFF;
		}
		if (instant_state.power_mode_input != prev_instant_state.power_mode_input) {
			stable_state.power_mode = instant_state.power_mode_input;
		}
		// а вот с ключом несколько сложнее: 2 входа, и надо ждать
		analyze_key();
		
		// **** Пишем выходы (и ком-порт) ****
		// сообщаем текущий Power_mode
		digitalWrite(pinPowerModeOutput, stable_state.power_mode);
		Serial.write(stable_state.power_mode);
		
		switch (stable_state.key_mode) {
			case KEY_MODE_ON: // работаем
				make_pc_on();
				break;
			case KEY_MODE_ACC: // смотрим на аккумулятор
				if (stable_state.u_state) { // есть заряд — работаем
					make_pc_on();
				} else { // нет заряда — отключаем
					make_pc_off();
				}
				break;
			case KEY_MODE_OFF: // смотрим на режим
				if (stable_state.power_mode) { // режим сна: смотрим на аккумулятор
					if (stable_state.u_state) { // есть заряд — спим
						make_pc_sleep();
					} else { // нет заряда — отключаем
						make_pc_off();
					}
				} else { // режим полного отключения: отключаем всё
					make_pc_off();
				}
				break;
		}
	}
}

int state_counter = 0;
void analyze_key(void) {
	// распознаём текущее мгновенное состояние
	if (instant_state.key_input) { // ACC либо ON
		if (instant_state.engine_input) {
			instant_state.key_mode = KEY_MODE_ON;
		} else {
			instant_state.key_mode = KEY_MODE_ACC;
		}
	} else { // LOCK либо START, но START нас не интересует, так что всегда LOCK
		instant_state.key_mode = KEY_MODE_OFF;
	}
	
	// Если все 3 равны, ничего не делаем
	if ((instant_state.key_mode == prev_instant_state.key_mode) && (instant_state.key_mode == stable_state.key_mode)) {
		state_counter = 0;
		return;
	}
	
	// Если мгновенные состояния равны, увеличиваем счётчик, иначе сбрасываем и выходим
	if (instant_state.key_mode == prev_instant_state.key_mode) {
		state_counter++;
	} else {
		state_counter = 0;
		return;
	}
	
	// не пора ли переключать состояние?
	if (state_counter > STATE_COUNTER_MAX) {
		stable_state.key_mode = instant_state.key_mode;
		state_counter = 0;
	}
	
	return;
}

void make_pc_off(void) {
	switch (stable_state.pc_state) {
		case PC_STATE_OFF: // ничего не делаем
			break;
		case PC_STATE_ON:
			// нажимаем кнопку питания
			digitalWrite(pinPower, 1);
			delay(PC_DELAY_DURING_OFF_MS);
			digitalWrite(pinPower, 0);
			// ждём, чтобы вход mb_input = 0, но не более 10 минут
			timer_off.interval(PC_DELAY_AFTER_OFF_MS);
			timer_off.reset();
			timer.reset();
			while (stable_state.mb_state && !timer_off.check()) {
				while(!timer.check()) {
					;
				};
				update_instant_state();
				// тут можно было бы как-то аккуратнее проверять (учитывать дребезг, а также обрабатывать остальные входы), но в принципе и так сойдёт
				stable_state.mb_state = instant_state.mb_input;
			};
			// отключаем БП
			turn_psu_off();
			break;
		case PC_STATE_SLEEP: // отключаем БП
			turn_psu_off();
			break;
	}
	stable_state.pc_state = PC_STATE_OFF;
	stable_state.power_mode = POWER_MODE_OFF; // чтобы БП не включился, если напряжение возрастёт — всё равно компьютер отключён
	timer.reset();
}

void make_pc_sleep(void) {
	switch (stable_state.pc_state) {
		case PC_STATE_OFF: // это возможно, если при подаче питания на Ардуину вход pinPowerModeInput был POWER_MODE_SLEEP. Тогда переключаем режим питания
			stable_state.power_mode = POWER_MODE_OFF;
			break;
		case PC_STATE_ON:
			turn_pc_sleep();
			break;
		case PC_STATE_SLEEP: // ничего не делаем
			break;
	}
	timer.reset();
}

void make_pc_on(void) {
	if (stable_state.psu_state) { // БП уже включён
		switch (stable_state.pc_state) { // смотрим состояние компьютера
			case PC_STATE_ON: // ничего не делаем
				break;
			case PC_STATE_OFF: // включаем
				delay(PC_BEFORE_ON_DELAY_MS);
				turn_pc_on();
				break;
			case PC_STATE_SLEEP: // пробуждаем
				turn_pc_on();
				break;
		}
	} else { // БП отключён — включаем всё
		delay(PSU_BEFORE_ON_DELAY_MS);
		turn_psu_on();
		delay(PC_BEFORE_ON_DELAY_MS);
		turn_pc_on();
	}
	timer.reset();
}

void turn_psu_on(void) {
	digitalWrite(pinPSU, 1);
	stable_state.psu_state = 1;
}

void turn_psu_off(void) {
	digitalWrite(pinPSU, 0);
	stable_state.psu_state = 0;
}


void turn_pc_on(void) {
	digitalWrite(pinPower, 1);
	delay(PC_DELAY_DURING_ON_MS);
	digitalWrite(pinPower, 0);
	delay(PC_AFTER_ON_DELAY_MS);
	stable_state.pc_state = PC_STATE_ON;
}

void turn_pc_sleep(void) {
	digitalWrite(pinPower, 1);
	delay(PC_DELAY_DURING_SLEEP_MS);
	digitalWrite(pinPower, 0);
	stable_state.pc_state = PC_STATE_SLEEP;
}

void update_instant_state(void) {
	instant_state.u_input = (analogRead(pinU)>U_TRES)?1:0;
	instant_state.key_input = digitalRead(pinKey);
	instant_state.mb_input = digitalRead(pinMB);
	instant_state.engine_input = digitalRead(pinEngine);
	instant_state.power_mode_input = digitalRead(pinPowerModeInput)?POWER_MODE_SLEEP:POWER_MODE_OFF;
	return;
}
