#include <IRremote.h>
#include <OneWire.h>
#include <DallasTemperature.h>
#include <Wire.h>
#include <MeetAndroid.h>
#include <Keypad.h>
#include <avr/wdt.h>   

#define HEATER1 21
#define HEATER2 22                                                                                                                                                                               

#define AMP_ON 16
#define AUDIO_MUTE 17
#define ONE_WIRE_BUS 25
const int RECV_PIN = 26;
#define AUDIO_ON 44

#define ENCODER_PIN1 19
#define ENCODER_PIN2 18  
#define MUTE_PIN 20

/*
1 (49) - 1
2 (50) - 2
3 (51) - 3
4 (52) - 4
5 (53) - 5
6 (54) - 6
7 (55) - RDS
8 (56) - TA
9 (57) - LEFT
A (65) - SRC
B (66) - DOWN
C (67) - BAND
D (68) - MAN
E (69) - RIGHT
F (70) - AUDIO
G (71) - UP
*/
char keyMap[4][4] = {
  {'7','2','5','6'},
  {'8','9','A','B'},
  {'C','D','E','F'},
  {'1','3','4','G'}
};
byte keyRowPins[4] = {23,24,43,40}; 
byte keyColPins[4] = {42,38,41,39};

#define TDA7318_I2C_ADDRESS 0x44
#define TDA_SW1 0x58
#define TDA_SW2 0x59
#define TDA_SW3 0x52
#define TDA_SW4 0x5B
// LM pins
#define LM_CE 14 //LM7001 PIN3
#define LM_CL 13 //LM7001 PIN4
#define LM_DA 12 //LM7001 PIN5
// LM delay in microsec
#define LM_DELAY 2

#define RADIO_MIN_FREQUENCY 880 
#define RADIO_MAX_FREQUENCY 1080

#define RADIO_SOURCE 2
#define MUSIC_SOURCE 3

#define PCF8576_I2C_ADDRESS 0x38
#define I2C_LCD_SEND_BUFFER_SIZE 17 

DeviceAddress intTempSensor = { 0x28, 0x44, 0x0A, 0xD8, 0x02, 0x00, 0x00, 0x58 };
DeviceAddress extTempSensor = { 0x28, 0xA8, 0xE4, 0x7D, 0x02, 0x00, 0x00, 0x5C };
DeviceAddress hlTempSensor = { 0x28, 0x7C, 0xDF, 0xD7, 0x02, 0x00, 0x00, 0x02 };
DeviceAddress hrTempSensor = { 0x28, 0xB6, 0x1A, 0x7E, 0x02, 0x00, 0x00, 0x98 };

byte volMap[] = {0x3F,0x3D,0x3B,0x39,0x37,0x35,0x33,0x31,   
                  0x2F,0x2D,0x2B,0x29,0x27,0x25,0x23,0x21,   
                  0x1F,0x1D,0x1B,0x19,0x17,0x15,0x13,0x11,   
                  0x0F,0x0D,0x0B,0x09,0x07,0x05,0x03,0x00};
                          
byte lfAttMap[] = {0x9F,0x9D,0x9B,0x99,0x97,0x95,0x93,0x91,
                   0x8F,0x8D,0x8B,0x89,0x87,0x85,0x83,0x80};                          

byte rfAttMap[] = {0xBF,0xBD,0xBB,0xB9,0xB7,0xB5,0xB3,0xB1,
                   0xAF,0xAD,0xAB,0xA9,0xA7,0xA5,0xA3,0xA0}; 

byte lrAttMap[] = {0xDF,0xDD,0xDB,0xD9,0xD7,0xD5,0xD3,0xD1,
                   0xCF,0xCD,0xCB,0xC9,0xC7,0xC5,0xC3,0xC0}; 

byte rrAttMap[] = {0xFF,0xFD,0xFB,0xF9,0xF7,0xF5,0xF3,0xF1,
                   0xEF,0xED,0xEB,0xE9,0xE7,0xE5,0xE3,0xE0}; 
                            
byte bassMap[] = {0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x6F,
                  0x6E,0x6D,0x6C,0x6B,0x6A,0x69,0x68}; 

byte trebleMap[] = {0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x7F,
                    0x7E,0x7D,0x7C,0x7B,0x7A,0x79,0x78}; 

byte lcdCharMap[91] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
                        0,0,0,0,0,0,0,0,0,0,0,1,0,0,2,0,0,3,4,5,6,7,8,9,10,11,12,0,0,0,
                        0,0,0,0,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38
                      };

char lcdCharDict1[10][39] = { { 0,33,33,17,33,17,17,33,17,17,33,17,17,33,17,17,17,17,33,17,33,17,17,33,17,33,33,17,33,17,33,17,33,17,33,33,33,17,17 },
                              { 0,0x9,0x8,0x4,0x70,0x4,0x4,0x68,0x4,0x4,0x62,0x4,0x4,0x6A,0x4,0x4,0x4,0x4,0xA,0x4,0x68,0x4,0x4,0x14,0x4,0x70,0x64,0x4,0x4A,0x4,0x4E,0x4,0x3,0x4,0x10,0x64,0x14,0x4,0x4 },
                              { 0,34,34,33,0,33,33,34,33,33,0,33,33,34,33,33,33,33,34,33,34,33,33,34,34,34,34,33,34,33,34,33,0,33,34,34,34,33,33 },
                              { 0,0x40,0x40,0x62,0x0,0x4A,0x6A,0x48,0x2A,0x2A,0x0,0x6A,0x6A,0x4C,0x6B,0x2,0x63,0xA,0x4C,0x2A,0x4C,0x3,0x60,0x4C,0xC,0x8C,0x8C,0x62,0x4C,0x66,0x4C,0x2A,0x0,0x60,0x2C,0x2C,0xA0,0x68,0x12 },
                              { 0,0,0,34,0,34,34,0,34,34,0,34,34,0,0,34,0,34,0,34,0,0,34,0,0,0,0,34,0,34,0,34,0,34,0,0,0,34,34 },
                              { 0,0x0,0x0,0xC,0x0,0x44,0x40,0x0,0x48,0x4C,0x0,0x4C,0x48,0x0,0x0,0xC,0x0,0x4C,0x0,0xC,0x0,0x0,0x4,0x0,0x0,0x0,0x0,0xC,0x0,0xC,0x0,0x48,0x0,0xC,0x0,0x0,0x0,0x48,0x20 },
                              { 0,0,0x0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                              { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0 },
                              { 0,0,0,0x0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                              { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0 }
                           };

char lcdCharDict2[10][39]  = { { 0,31,31,16,31,16,31,31,31,16,31,16,31,16,31,16,31,16,16,16,16,32,16,15,16,16,15,16,16,15,15,31,32,16,16,15,15,31,31 },
                               { 0,0x1,0x1,0x4,0xE,0x4,0xD,0xD,0x5,0x4,0xC,0x4,0xD,0x4,0xD,0x4,0xC,0x4,0x4,0x4,0x4,0xE0,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0xC0,0x4,0x4,0x4,0x4,0xD,0x2 },
                               { 0,32,32,31,0,31,32,32,32,31,32,31,32,31,32,32,32,31,31,31,31,0,31,16,32,31,16,31,31,16,16,32,0,31,31,16,31,32,32 },
                               { 0,0x48,0x8,0xC,0x0,0x9,0xA8,0x9,0xA9,0x5,0x80,0xD,0xA9,0xD,0xE0,0xA1,0xE0,0x1,0x1,0x5,0xD,0x0,0xC,0x4,0x21,0xE,0x4,0xC,0x9,0x4,0x4,0xA9,0x0,0xC,0x2,0x4,0x2,0x29,0xA4 },
                               { 0,0,0,32,0,32,0,0,0,32,0,32,0,32,0,0,0,32,32,32,32,0,32,31,0,32,31,32,32,31,31,0,0,32,32,31,32,0,0 },
                               { 0,0x0,0x0,0xA1,0x0,0xA8,0x0,0x0,0x0,0xA9,0x0,0xA9,0x0,0x89,0x0,0x0,0x0,0xA9,0x89,0xA1,0x9,0x0,0x20,0x2,0x0,0x11,0xC,0xA1,0x89,0xC,0x9,0x0,0x0,0x21,0x5,0xC,0x14,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,0,0,32,0,0,32,32,0,0,0,0,32,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x9,0x0,0x0,0x11,0x0,0x0,0xA1,0x89,0x0,0x0,0x0,0x0,0x5,0x0,0x0,0x0 },
                               { 0,0,0,0x0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0 }
                             }; 
         
char lcdCharDict3[10][39]  = { { 0,30,30,13,13,29,13,13,13,13,13,13,13,13,13,30,13,30,30,13,13,30,13,30,30,13,13,13,29,13,29,13,30,13,14,13,14,13,14 },
                               { 0,0x49,0x41,0x4,0x4,0x1,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x14,0x4,0x55,0x51,0x4,0x4,0x1C,0x4,0xA1,0x4,0x4,0x4,0x4,0x1,0x4,0x1,0x4,0x18,0x4,0x4,0x4,0x4,0x4,0x4 },
                               { 0,0,0,29,29,30,29,29,30,30,29,29,29,29,29,31,29,31,31,30,29,0,29,31,31,29,29,29,30,29,30,30,0,29,30,14,30,29,30 },
                               { 0,0x0,0x0,0x1,0x1,0x55,0x1,0x1,0x55,0x55,0x1,0x1,0x1,0x1,0x1,0x60,0x1,0x60,0x60,0x54,0x1,0x0,0x1,0x60,0x60,0x1,0x1,0x1,0x51,0x1,0x71,0x55,0x0,0x1,0x80,0x4,0xA2,0x1,0x94 },
                               { 0,0,0,30,30,31,30,30,31,31,30,30,30,30,30,0,30,0,0,31,30,0,30,0,0,30,30,30,31,30,31,31,0,30,31,29,0,30,0 },
                               { 0,0x0,0x0,0x14,0x80,0x20,0x55,0x41,0x40,0x60,0x10,0x55,0x55,0x51,0x5C,0x0,0x1C,0x0,0x0,0x60,0x41,0x0,0x4,0x0,0x0,0x82,0x22,0x14,0x60,0x34,0x60,0x40,0x0,0x4,0x60,0x1,0x0,0x45,0x0 },
                               { 0,0,0,31,0,0,0,31,0,0,0,31,31,31,0,0,0,0,0,0,31,0,31,0,0,31,31,31,0,31,0,0,0,31,0,30,0,31,0 },
                               { 0,0x0,0x0,0x60,0x0,0x0,0x0,0x40,0x0,0x0,0x0,0x60,0x40,0x60,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x20,0x0,0x0,0x60,0x60,0x60,0x0,0x60,0x0,0x0,0x0,0x60,0x0,0x20,0x0,0x40,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,31,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x0,0x0 }
                             };                   

char lcdCharDict4[10][39]  = { { 0,28,28,12,28,12,12,28,12,12,28,12,12,28,12,12,12,12,28,12,28,12,12,28,12,28,28,12,28,12,28,12,28,12,28,28,28,12,12 },
                               { 0,0x9,0x8,0x4,0x70,0x4,0x4,0x68,0x4,0x4,0x62,0x4,0x4,0x6A,0x4,0x4,0x4,0x4,0xA,0x4,0x68,0x4,0x4,0x14,0x4,0x70,0x64,0x4,0x4A,0x4,0x4E,0x4,0x3,0x4,0x10,0x64,0x14,0x4,0x4 },
                               { 0,29,29,28,0,28,28,29,28,28,0,28,28,29,28,28,28,28,29,28,29,28,28,29,29,29,29,28,29,28,29,28,0,28,29,29,29,28,28 },
                               { 0,0x40,0x40,0x62,0x0,0x4A,0x6A,0x48,0x2A,0x2A,0x0,0x6A,0x6A,0x4C,0x6B,0x2,0x63,0xA,0x4C,0x2A,0x4C,0x3,0x60,0x4C,0xC,0x8C,0x8C,0x62,0x4C,0x66,0x4C,0x2A,0x0,0x60,0x2C,0x2C,0xA0,0x68,0x12 },
                               { 0,0,0,29,0,29,29,0,29,29,0,29,29,0,0,29,0,29,0,29,0,0,29,0,0,0,0,29,0,29,0,29,0,29,0,0,0,29,29 },
                               { 0,0x0,0x0,0xC,0x0,0x44,0x40,0x0,0x48,0x4C,0x0,0x4C,0x48,0x0,0x0,0xC,0x0,0x4C,0x0,0xC,0x0,0x0,0x4,0x0,0x0,0x0,0x0,0xC,0x0,0xC,0x0,0x48,0x0,0xC,0x0,0x0,0x0,0x48,0x20 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0 }                             
                             };   
    
char lcdCharDict5[10][39]  = { { 0,26,26,11,26,11,26,26,26,11,26,11,26,11,26,11,26,11,11,11,11,27,11,10,11,11,10,11,11,10,10,26,27,11,11,10,10,26,26 },
                               { 0,0x1,0x1,0x4,0xE,0x4,0xD,0xD,0x5,0x4,0xC,0x4,0xD,0x4,0xD,0x4,0xC,0x4,0x4,0x4,0x4,0xE0,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x5,0xC0,0x4,0x4,0x4,0x4,0xD,0x2 },
                               { 0,27,27,26,0,26,27,27,27,26,27,26,27,26,27,27,27,26,26,26,26,0,26,11,27,26,11,26,26,11,11,27,0,26,26,11,26,27,27 },
                               { 0,0x48,0x8,0xC,0x0,0x9,0xA8,0x9,0xA9,0x5,0x80,0xD,0xA9,0xD,0xE0,0xA1,0xE0,0x1,0x1,0x5,0xD,0x0,0xC,0x4,0x21,0xE,0x4,0xC,0x9,0x4,0x4,0xA9,0x0,0xC,0x2,0x4,0x2,0x29,0xA4 },
                               { 0,0,0,27,0,27,0,0,0,27,0,27,0,27,0,0,0,27,27,27,27,0,27,26,0,27,26,27,27,26,26,0,0,27,27,26,27,0,0 },
                               { 0,0x0,0x0,0xA1,0x0,0xA8,0x0,0x0,0x0,0xA9,0x0,0xA9,0x0,0x89,0x0,0x0,0x0,0xA9,0x89,0xA1,0x9,0x0,0x20,0x2,0x0,0x11,0xC,0xA1,0x89,0xC,0x9,0x0,0x0,0x21,0x5,0xC,0x14,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,0,0,27,0,0,27,27,0,0,0,0,27,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x9,0x0,0x0,0x11,0x0,0x0,0xA1,0x89,0x0,0x0,0x0,0x0,0x5,0x0,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0 }                             
                             };
                             
char lcdCharDict6[10][39]  = { { 0,25,25,8,8,24,8,8,8,8,8,8,8,8,8,25,8,25,25,8,8,25,8,25,25,8,8,8,24,8,24,8,25,8,9,8,9,8,9 },
                               { 0,0x49,0x41,0x4,0x4,0x1,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x4,0x14,0x4,0x55,0x51,0x4,0x4,0x1C,0x4,0xA1,0x4,0x4,0x4,0x4,0x1,0x4,0x1,0x4,0x18,0x4,0x4,0x4,0x4,0x4,0x4 },
                               { 0,0,0,24,24,25,24,24,25,25,24,24,24,24,24,26,24,26,26,25,24,0,24,26,26,24,24,24,25,24,25,25,0,24,25,9,25,24,25 },
                               { 0,0x0,0x0,0x1,0x1,0x55,0x1,0x1,0x55,0x55,0x1,0x1,0x1,0x1,0x1,0x60,0x1,0x60,0x60,0x54,0x1,0x0,0x1,0x60,0x60,0x1,0x1,0x1,0x51,0x1,0x71,0x55,0x0,0x1,0x80,0x4,0xA2,0x1,0x94 },
                               { 0,0,0,25,25,26,25,25,26,26,25,25,25,25,25,0,25,0,0,26,25,0,25,0,0,25,25,25,26,25,26,26,0,25,26,24,0,25,0 },
                               { 0,0x0,0x0,0x14,0x80,0x20,0x55,0x41,0x40,0x60,0x10,0x55,0x55,0x51,0x5C,0x0,0x1C,0x0,0x0,0x60,0x41,0x0,0x4,0x0,0x0,0x82,0x22,0x14,0x60,0x34,0x60,0x40,0x0,0x4,0x60,0x1,0x0,0x45,0x0 },
                               { 0,0,0,26,0,0,0,26,0,0,0,26,26,26,0,0,0,0,0,0,26,0,26,0,0,26,26,26,0,26,0,0,0,26,0,25,0,26,0 },
                               { 0,0x0,0x0,0x60,0x0,0x0,0x0,0x40,0x0,0x0,0x0,0x60,0x40,0x60,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x20,0x0,0x0,0x60,0x60,0x60,0x0,0x60,0x0,0x0,0x0,0x60,0x0,0x20,0x0,0x40,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,26,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x0,0x0 }
                             };                             

char lcdCharDict7[10][39]  = { { 0,23,23,7,23,7,7,23,7,7,23,7,7,23,7,7,7,7,23,7,23,7,7,23,7,23,23,7,23,7,23,7,23,7,23,23,23,7,7 },
                               { 0,0x9,0x8,0x4,0x70,0x4,0x4,0x68,0x4,0x4,0x62,0x4,0x4,0x6A,0x4,0x4,0x4,0x4,0xA,0x4,0x68,0x4,0x4,0x14,0x4,0x70,0x64,0x4,0x4A,0x4,0x4E,0x4,0x3,0x4,0x10,0x64,0x14,0x4,0x4 },
                               { 0,24,24,23,0,23,23,24,23,23,0,23,23,24,23,23,23,23,24,23,24,23,23,24,24,24,24,23,24,23,24,23,0,23,24,24,24,23,23 },
                               { 0,0x40,0x40,0x62,0x0,0x4A,0x6A,0x48,0x2A,0x2A,0x0,0x6A,0x6A,0x4C,0x6B,0x2,0x63,0xA,0x4C,0x2A,0x4C,0x3,0x60,0x4C,0xC,0x8C,0x8C,0x62,0x4C,0x66,0x4C,0x2A,0x0,0x60,0x2C,0x2C,0xA0,0x68,0x12 },
                               { 0,0,0,24,0,24,24,0,24,24,0,24,24,0,0,24,0,24,0,24,0,0,24,0,0,0,0,24,0,24,0,24,0,24,0,0,0,24,24 },
                               { 0,0x0,0x0,0xC,0x0,0x44,0x40,0x0,0x48,0x4C,0x0,0x4C,0x48,0x0,0x0,0xC,0x0,0x4C,0x0,0xC,0x0,0x0,0x4,0x0,0x0,0x0,0x0,0xC,0x0,0xC,0x0,0x48,0x0,0xC,0x0,0x0,0x0,0x48,0x20 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x0,0x0 }
                             };
                             
char lcdCharDict8[10][39]  = { { 0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,5,5,5,5,5,22,5,5,6,5,5,5,5,5,5,5,22,5,5,5,5,5,5 },
                               { 0,0x8,0x8,0x60,0x70,0x48,0x68,0x68,0x28,0x28,0x60,0x68,0x68,0x68,0x68,0x4,0x60,0x8,0x8,0x28,0x68,0xE0,0x60,0x14,0x4,0x70,0x64,0x60,0x48,0x64,0x4C,0x28,0xC0,0x60,0x10,0x64,0x14,0x68,0x10 },
                               { 0,22,22,6,0,6,22,22,22,6,22,6,22,6,22,22,22,6,6,6,6,0,6,6,22,6,6,6,6,6,6,22,0,6,6,6,22,22,22 },
                               { 0,0x48,0x8,0x4,0x0,0x4,0xA8,0x9,0xA9,0x4,0x80,0x4,0xA9,0x4,0xE0,0xA1,0xE0,0x4,0x4,0x4,0x4,0x0,0x4,0x4,0x21,0x4,0x4,0x4,0x4,0x4,0x4,0xA9,0x0,0x4,0x4,0x4,0x14,0x29,0xA4 },
                               { 0,0,0,22,0,22,0,0,0,22,0,22,0,22,0,0,0,22,22,22,22,0,22,22,0,22,22,22,22,22,22,0,0,22,22,22,0,0,0 },
                               { 0,0x0,0x0,0xA1,0x0,0xA8,0x0,0x0,0x0,0xA9,0x0,0xA9,0x0,0x89,0x0,0x0,0x0,0xA9,0x89,0xA1,0x9,0x0,0x20,0x9,0x0,0x11,0x11,0xA1,0x89,0xA1,0x89,0x0,0x0,0x21,0x5,0x5,0x0,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x0,0x0 },
                               { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 },
                               { 0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x60,0x0,0x0,0x0 }
                             }; 

byte currentVolume = 16;
byte currentHeaterOff1 = 0;
byte currentHeaterOn1 = 0;
byte currentHeaterOff2 = 0;
byte currentHeaterOn2 = 0;
unsigned long heaterTimer1 = 150000;
unsigned long heaterTimer2 = 0;
boolean muteButtonState;
boolean prevMuteButtonState;

OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature sensors(&oneWire);

IRrecv irrecv(RECV_PIN);
decode_results results;

MeetAndroid meetAndroid;

Keypad keypad = Keypad(makeKeymap(keyMap), keyRowPins, keyColPins, 4, 4);

HardwareSerial Uart = HardwareSerial();

void setup() {
  wdt_disable();
  Uart.begin(38400); 
 // Serial.begin(9600); 

  setCallbacks();
  initTempSensors();
  Wire.begin();
  initTda();
  initLM();
  initHeaters();
  initIR();
  initLCD();

  pinMode(AMP_ON, OUTPUT);
  pinMode(AUDIO_ON, OUTPUT);
  pinMode(AUDIO_MUTE, OUTPUT);
  pinMode(MUTE_PIN, INPUT_PULLUP);
  
  pinMode(ENCODER_PIN1, INPUT_PULLUP);
  pinMode(ENCODER_PIN2, INPUT_PULLUP);
  
  attachInterrupt(6, updateEncoderRotation, FALLING);
  
  wdt_enable(WDTO_8S);
}

void loop() {
  receiveKeys();
  meetAndroid.receive();
  receiveIR();
  checkHeaters();
  wdt_reset();
}

void setCallbacks() {
  meetAndroid.registerFunction(getInternalTemperature, 'A'); //Internal temperature: 1
  meetAndroid.registerFunction(setAudioBalance, 'B'); //Audio balance: 0-15
  meetAndroid.registerFunction(setLcdInfo, 'C'); //LCD: P,MP3,0001 - Player; A,VOL,32 - Audio
  meetAndroid.registerFunction(setHeaterOnTemperature1, 'G'); //Heater temp 1 ON: 25-ON, 0-OFF
  meetAndroid.registerFunction(setHeaterOnTemperature2, 'H'); //Heater temp 2 ON: 25-ON, 0-OFF
  meetAndroid.registerFunction(getExternalTemperature, 'E'); //External temperature: 1
  meetAndroid.registerFunction(setAudioBass, 'J'); //Audio bass: 0-14
  meetAndroid.registerFunction(setAudioTreble, 'K'); //Audio treble: 0-14
  meetAndroid.registerFunction(setAudioRearLeftVolume, 'L'); //Audio rear left volume: 0-15
  meetAndroid.registerFunction(setAudioMute, 'M'); //Audio mute: 1-ON, 0-OFF
  meetAndroid.registerFunction(setHeaterOffTemperature1, 'N'); //Heater temp 1 OFF: 25-OFF, 0-OFF
  meetAndroid.registerFunction(setHeaterOffTemperature2, 'O'); //Heater temp 2 OFF: 25-OFF, 0-OFF
  meetAndroid.registerFunction(setRadioFrequency, 'R'); //Radio frequency: (880-1080)0/1(OFF/ON)(1-999) - Preset
  meetAndroid.registerFunction(setAudioSource, 'S'); //Audio source: 2-radio, 3-music   
  meetAndroid.registerFunction(setAudioVolume, 'V'); //Audio volume: 0-31
  meetAndroid.registerFunction(setAudioRearRightVolume, 'X'); //Audio rear right volume: 0-15
  meetAndroid.registerFunction(getHeaterTemperature1, 'Y'); //Heater 1
  meetAndroid.registerFunction(getHeaterTemperature2, 'Z'); //Heater 2  
} 

void initTempSensors() {
  sensors.begin();
  // set the resolution to 9 bit
  sensors.setResolution(intTempSensor, 9);
  sensors.setResolution(extTempSensor, 9);
  sensors.setResolution(hlTempSensor, 9);    
  sensors.setResolution(hrTempSensor, 9);
}

void initTda() {
  sendAudioMute(1);
  sendAudioOn(1);
  delay(3000);
  sendAmpOn(1);
  Wire.begin();
  sendAudioVolume(16);
  sendAudioLFAttenuator(15);
  sendAudioRFAttenuator(15);
  sendAudioLRAttenuator(15);
  sendAudioRRAttenuator(15);
  sendAudioSwitch(3);  
  sendAudioBass(7);
  sendAudioTreble(7);
  sendAudioMute(0);
}

void initLM() {
  pinMode(LM_CE, OUTPUT);
  pinMode(LM_CL, OUTPUT);
  pinMode(LM_DA, OUTPUT);  
}

void initHeaters() {
  pinMode(HEATER1, OUTPUT);
  pinMode(HEATER2, OUTPUT);
}

void initIR() {
  irrecv.enableIRIn();
  irrecv.blink13(true);  
}

void receiveIR() {
  if (irrecv.decode(&results)) {
    char buf[50];
    sprintf(buf, "IR:%u", results.value);
    meetAndroid.send(buf);
    irrecv.resume(); 
  }  
}  
  
void updateEncoderRotation() {
  if (digitalRead(ENCODER_PIN1)) {
    meetAndroid.send("EN:1");
  } else {
    meetAndroid.send("EN:0");
  }
}
  
void receiveKeys() {
  char key = keypad.getKey();

  if (key != NO_KEY){
    char buf[5];
    sprintf(buf, "KE:%u", key);
    meetAndroid.send(buf);
    //Serial.println(buf);
  }
  
  muteButtonState = digitalRead(MUTE_PIN);
  if (muteButtonState == LOW && muteButtonState != prevMuteButtonState) {
    delay(50);
    if (muteButtonState == digitalRead(MUTE_PIN)) {
      meetAndroid.send("MK:1");
    }
  }
  prevMuteButtonState = muteButtonState;
}  
  
void checkHeaters() {
  if (currentHeaterOff1 == 0 && currentHeaterOn1 == 0) {
    //Serial.println("HEATER1: OFF");
    digitalWrite(HEATER1, LOW);    
  }
  else {  
    if (heaterTimer1 == 300000) {
      heaterTimer1 = 0;
      
      sensors.requestTemperaturesByAddress(hlTempSensor);
      int tempL = sensors.getTempC(hlTempSensor);
      if (tempL != -127) {
        if (tempL >= currentHeaterOff1) {
       //   Serial.println("HEATER1: OFF");
          digitalWrite(HEATER1, LOW); 
        } 
        if (tempL <= currentHeaterOn1) {
         // Serial.println("HEATER1: ON");
          digitalWrite(HEATER1, HIGH); 
        } 
      }  
    }
    heaterTimer1++;
  }  
  if (currentHeaterOff2 == 0 && currentHeaterOn2 == 0) {
    //Serial.println("HEATER2: OFF");
    digitalWrite(HEATER2, LOW);    
  }
  else {  
    if (heaterTimer2 == 300000) {
      heaterTimer2 = 0;
      
      sensors.requestTemperaturesByAddress(hrTempSensor);
      int tempR = sensors.getTempC(hrTempSensor);
      if (tempR != -127) {
        if (tempR >= currentHeaterOff2) {
         // Serial.println("HEATER2: OFF");
          digitalWrite(HEATER2, LOW); 
        } 
        if (tempR <= currentHeaterOn2) {
          //Serial.println("HEATER2: ON");
          digitalWrite(HEATER2, HIGH); 
        }  
      } 
      
    } 
    heaterTimer2++;
  }    
}  
  
void writeI2c(byte address, byte value) {
  Wire.beginTransmission(address); 
  Wire.send(value);
  Wire.endTransmission();  
}

void sendAudioMute(byte value) {
  digitalWrite(AUDIO_MUTE, value == 1 ? HIGH : LOW);
}

void sendAmpOn(byte value) {
  digitalWrite(AMP_ON, value == 1 ? HIGH : LOW);
}

void sendAudioOn(byte value) {
  digitalWrite(AUDIO_ON, value == 1 ? HIGH : LOW);
}

void sendAudioVolume(byte value) {
  if (value > 31) return;
  currentVolume = value;
  writeI2c(TDA7318_I2C_ADDRESS, volMap[value]);   
} 

void sendAudioLFAttenuator(byte value) {   
  if (value > 15) return;
  writeI2c(TDA7318_I2C_ADDRESS, lfAttMap[value]);   
}

void sendAudioRFAttenuator(byte value) {   
  if (value > 15) return;
  writeI2c(TDA7318_I2C_ADDRESS, rfAttMap[value]);   
}

void sendAudioLRAttenuator(byte value) {   
  if (value > 15) return;
  writeI2c(TDA7318_I2C_ADDRESS, lrAttMap[value]);   
}

void sendAudioRRAttenuator(byte value) {   
  if (value > 15) return;
  writeI2c(TDA7318_I2C_ADDRESS, rrAttMap[value]);   
}

void sendAudioBass(byte value) {   
  if (value > 14) return;   
  writeI2c(TDA7318_I2C_ADDRESS, bassMap[value]);   
}

void sendAudioTreble(byte value) {   
  if (value > 14) return;   
  writeI2c(TDA7318_I2C_ADDRESS, trebleMap[value]);   
}

void sendAudioSwitch(byte value) {
  switch (value) {
    case 1:
      writeI2c(TDA7318_I2C_ADDRESS, TDA_SW1);   
      break;
    case 2:
      writeI2c(TDA7318_I2C_ADDRESS, TDA_SW2);   
      break;
    case 3:
      writeI2c(TDA7318_I2C_ADDRESS, TDA_SW3);   
      break;
    case 4:
      writeI2c(TDA7318_I2C_ADDRESS, TDA_SW4);   
      break;      
  }  
} 

void sendRadioFrequency(int frequency, boolean power) {
  frequency += 107;
  digitalWrite(LM_CE, HIGH);
  writeLM(byte(frequency));
  writeLM(byte(frequency >> 8));
  writeLM((power == true) ? 135 : 71);
  digitalWrite(LM_CE, LOW);
}

void writeLM(byte bytes) {
  int d; 
  int d1;
  delayMicroseconds(LM_DELAY);
  for (int x=0; x <= 7; x++) {
  // Set DA
    d = bytes >> 1;
    d1 = d << 1;
    digitalWrite(LM_DA, bytes == d1 ? LOW : HIGH);
    // Make CL  
    digitalWrite(LM_CL, HIGH);
    delayMicroseconds(LM_DELAY);
    digitalWrite(LM_CL, LOW);
    delayMicroseconds(LM_DELAY);  
    bytes = bytes >> 1;    
  }
  delayMicroseconds(LM_DELAY);
} 

void getInternalTemperature(byte flag, byte numOfValues) {
  sensors.requestTemperaturesByAddress(intTempSensor);
  float tempC = sensors.getTempC(intTempSensor);
  if (tempC == -127.00) {
    tempC = 0;
  }  
  char buf[50];
  sprintf(buf, "IT:%d", (int)tempC);
  meetAndroid.send(buf);
}

void getExternalTemperature(byte flag, byte numOfValues) {
  sensors.requestTemperaturesByAddress(extTempSensor);
  float tempC = sensors.getTempC(extTempSensor);
  if (tempC == -127.00) {
    tempC = 0;
  }  
  char buf[50];
  sprintf(buf, "ET:%d", (int)tempC);
  meetAndroid.send(buf);
}

void setAudioBalance(byte flag, byte numOfValues) {
  int value = meetAndroid.getInt();
  sendAudioLFAttenuator(value);
  sendAudioRFAttenuator(15 - value);
}

void setHeaterOnTemperature1(byte flag, byte numOfValues) {
  currentHeaterOn1 = meetAndroid.getInt();    
}

void setHeaterOffTemperature1(byte flag, byte numOfValues) {
  currentHeaterOff1 = meetAndroid.getInt();    
}

void setHeaterOnTemperature2(byte flag, byte numOfValues) {
  currentHeaterOn2 = meetAndroid.getInt();    
}

void setHeaterOffTemperature2(byte flag, byte numOfValues) {
  currentHeaterOff2 = meetAndroid.getInt();    
}

void setAudioBass(byte flag, byte numOfValues) {
  sendAudioBass(meetAndroid.getInt());
}

void setAudioTreble(byte flag, byte numOfValues) {
  sendAudioTreble(meetAndroid.getInt());
}

void setAudioRearLeftVolume(byte flag, byte numOfValues) {
  sendAudioLRAttenuator(meetAndroid.getInt());
}

void setAudioMute(byte flag, byte numOfValues) {
  sendAudioMute(meetAndroid.getInt());
}

void setRadioFrequency(byte flag, byte numOfValues) {
  char data[10];
  meetAndroid.getString(data);

  String content = "";
  int i = 0;
  while(i < 10) {
    content.concat(data[i]);
    i++;
  }
  String frequencyStr = content.substring(0, 4);
  char frequencyArray[5];
  frequencyStr.toCharArray(frequencyArray, 5);
  int frequency = atoi(frequencyArray);

  if (content.substring(5, 6) == '0') {
    sendRadioFrequency(frequency, false);
    sendAudioSwitch(MUSIC_SOURCE);
  }
  else if (frequency >= RADIO_MIN_FREQUENCY && frequency <= RADIO_MAX_FREQUENCY){
    sendRadioFrequency(frequency, true);
    sendAudioSwitch(RADIO_SOURCE);
  }
  if (frequencyArray[0] == '0') {
    frequencyArray[0] = ' ';
  }
  if (data[7] == '0') {
    data[7] = ' ';
  }
  if (data[8] == '0') {
    data[8] = ' ';
  }

  char lcdData[8] = { frequencyArray[0], frequencyArray[1], frequencyArray[2], frequencyArray[3], ' ',  data[7], data[8], data[9] };
  sendDataToLCD('R', lcdData);
}

void setAudioSource(byte flag, byte numOfValues) {
  sendAudioSwitch(meetAndroid.getInt());
}

void setAudioVolume(byte flag, byte numOfValues) {
  sendAudioVolume(meetAndroid.getInt());
}

void setAudioRearRightVolume(byte flag, byte numOfValues) {
  sendAudioRRAttenuator(meetAndroid.getInt());
}

void getHeaterTemperature1(byte flag, byte numOfValues) {
  sensors.requestTemperaturesByAddress(hlTempSensor);
  float tempC = sensors.getTempC(hlTempSensor);
  if (tempC == -127.00) {
    tempC = 0;
  }  
  char buf[50];
  sprintf(buf, "LT:%d", (int)tempC);
  meetAndroid.send(buf);
}

void getHeaterTemperature2(byte flag, byte numOfValues) {
  sensors.requestTemperaturesByAddress(hrTempSensor);
  float tempC = sensors.getTempC(hrTempSensor);
  if (tempC == -127.00) {
    tempC = 0;
  }  
  char buf[50];
  sprintf(buf, "RT:%d", (int)tempC);
  meetAndroid.send(buf);
}

void initLCD() {
  unsigned char data[I2C_LCD_SEND_BUFFER_SIZE];
  unsigned char data2[I2C_LCD_SEND_BUFFER_SIZE];
  
  getInitialLCDData(data, data2);

  transmitDataToLCD(data, data2); 
  
  char lcdData[8] = {'L', 'O', 'A', 'D', 'I', 'N', 'G', ' '};
  sendDataToLCD(' ', lcdData);
}

void getInitialLCDData(unsigned char *data, unsigned char *data2) {
  byte i = 0;

  data[0] = 0xE0;
  data[1] = 0xCB;
  data[2] = 0xF0;
  for (i = 3; i < I2C_LCD_SEND_BUFFER_SIZE; i++) {
    data[i] = 0x00; 
  }  

  data2[0] = 0xE0;
  data2[1] = 0xCB;
  data2[2] = 0xF0;
  data2[3] = 0x2;
  for (i = 4; i < I2C_LCD_SEND_BUFFER_SIZE; i++) {
    data2[i] = 0x00; 
  }  
}

void transmitDataToLCD(unsigned char *data, unsigned char *data2) {
  Wire.beginTransmission(PCF8576_I2C_ADDRESS);
  Wire.write(data, I2C_LCD_SEND_BUFFER_SIZE);
  Wire.endTransmission();

  Wire.beginTransmission(PCF8576_I2C_ADDRESS);
  Wire.write(data2, I2C_LCD_SEND_BUFFER_SIZE);
  Wire.endTransmission();
}

void sendDataToLCD(unsigned char mode, char lcdData[8]) {
  unsigned char data[I2C_LCD_SEND_BUFFER_SIZE];
  unsigned char data2[I2C_LCD_SEND_BUFFER_SIZE];
  
  getInitialLCDData(data, data2);
  
  if (mode == 'R') {
    data2[5] += 0x80;
    data2[11] += 0x2;  
  }  
  else if (mode == 'P') {
    data2[6] += 0x2;
  }  
  byte i; 
  byte charNumber;

  for (i = 0; i < 8; i++) {
    charNumber = lcdCharMap[lcdData[i]];
    if (charNumber > 0) {
      switch (i) {
        case 0:
          getCharLCDData(charNumber, lcdCharDict1, data, data2);
          break;
        case 1:
          getCharLCDData(charNumber, lcdCharDict2, data, data2);
          break;  
        case 2:
          getCharLCDData(charNumber, lcdCharDict3, data, data2);
          break;
        case 3:
          getCharLCDData(charNumber, lcdCharDict4, data, data2);
          break;  
        case 4:
          getCharLCDData(charNumber, lcdCharDict5, data, data2);
          break;
        case 5:
          getCharLCDData(charNumber, lcdCharDict6, data, data2);
          break;  
        case 6:
          getCharLCDData(charNumber, lcdCharDict7, data, data2);
          break;
        case 7:
          getCharLCDData(charNumber, lcdCharDict8, data, data2);
          break;  
      }
    }
  } 
  transmitDataToLCD(data, data2); 
}  

void getCharLCDData(byte charNumber, char lcdCharDict[10][39], unsigned char *data, unsigned char *data2) {
  byte pos;
  byte bytePos;
  
  for (pos = 0; pos < 5; pos ++) { 
    bytePos = lcdCharDict[pos * 2][charNumber];

    if (bytePos > 0) {
      if (bytePos > I2C_LCD_SEND_BUFFER_SIZE) {   
        data2[bytePos - I2C_LCD_SEND_BUFFER_SIZE - 1] += lcdCharDict[2 * pos + 1][charNumber];
      }
      else {
        data[bytePos - 1] += lcdCharDict[2 * pos + 1][charNumber];
      }
    } 
  }
}

void setLcdInfo(byte flag, byte numOfValues) {
  char data[10];
  meetAndroid.getString(data);
  //P,MP3,0001 - Player; A,VOL , 32 - Audio
  if (data[0] == 'P') {
    char lcdData[8] = { data[2], data[3], data[4], ' ', data[6], data[7], data[8], data[9] };
    sendDataToLCD(data[0], lcdData); 
  }
  else if (data[0] == 'A' || data[0] == 'M') {
    if (data[8] == '0') {
      data[8] = ' ';
    }  
    char lcdData[8] = { data[2], data[3], data[4], data[5], ' ', data[7], data[8], data[9] };
    sendDataToLCD(data[0], lcdData); 
  }
}
