/*
* SACD Decoder plugin
* Copyright (c) 2011 Maxim V.Anisiutkin <maxim.anisiutkin@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with FFmpeg; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <intrin.h>
#include <xmmintrin.h>
#include "dsdpcm_converter.h"

#ifndef _DSDPCM_CONVERTER_REAL_H_INCLUDED
#define _DSDPCM_CONVERTER_REAL_H_INCLUDED

typedef float real_t;

#define SSE_ALIGN 16
#define SSE_ASIZE(l, s) ((l + (SSE_ALIGN / s - 1)) & ~(SSE_ALIGN / s - 1))
#define NORM ((real_t)1 / (real_t)((uint32_t)1 << 31))

typedef real_t ctable_r[256];

class dsdpcm_fir_r {
	ctable_r* fir_ctables;
	int       fir_order;
	int       fir_length;
	int       fir_size;
	int       channels;
	int       decimation;
	uint8_t*  fir_buffer[DSDPCM_MAX_CHANNELS];
	int       fir_index;
public:
	dsdpcm_fir_r() {
		memset(fir_buffer, 0, sizeof(fir_buffer));
	}
	~dsdpcm_fir_r() {
		free();
	}
	void init(ctable_r* fir_ctables, int fir_length, int channels, int decimation);
	void free();
	int get_decimation() {
		return decimation;
	}
	float get_delay();
	int run(uint8_t* dsd_data, real_t* pcm_data, int dsd_samples);
	int run_sse(uint8_t* dsd_data, real_t* pcm_data, int dsd_samples);
};

class pcmpcm_fir_r {
	real_t* fir_coefs;
	int     fir_order;
	int     fir_length;
	int     fir_size;
	int     channels;
	int     decimation;
	real_t* fir_buffer[DSDPCM_MAX_CHANNELS];
	int     fir_index;
public:
	pcmpcm_fir_r() {
		memset(fir_buffer, 0, sizeof(fir_buffer));
	}
	~pcmpcm_fir_r() {
		free();
	}
	void init(real_t* fir_coefs, int fir_length, int channels, int decimation);
	void free();
	int get_decimation() {
		return decimation;
	}
	float get_delay();
	int run(real_t* pcm_data, real_t* out_data, int pcm_samples);
	int run_sse(real_t* pcm_data, real_t* out_data, int pcm_samples);
};

class dsdpcm_converter_r : public dsdpcm_conv_impl_t {
	__declspec(align(SSE_ALIGN)) static real_t dsd_fir1_8_ctables[SSE_ASIZE(CTABLES(DSDFIR1_8_LENGTH), sizeof(real_t))][256];
	__declspec(align(SSE_ALIGN)) static real_t dsd_fir1_16_ctables[SSE_ASIZE(CTABLES(DSDFIR1_16_LENGTH), sizeof(real_t))][256];
	__declspec(align(SSE_ALIGN)) static real_t dsd_fir1_64_ctables[SSE_ASIZE(CTABLES(DSDFIR1_64_LENGTH), sizeof(real_t))][256];
	__declspec(align(SSE_ALIGN)) static real_t pcm_fir2_2_coefs[SSE_ASIZE(PCMFIR2_2_LENGTH, sizeof(real_t))];
	__declspec(align(SSE_ALIGN)) static real_t pcm_fir3_2_coefs[SSE_ASIZE(PCMFIR3_2_LENGTH, sizeof(real_t))];

	float delay;

	dsdpcm_fir_r dsd_fir1;
	pcmpcm_fir_r pcm_fir2a;
	pcmpcm_fir_r pcm_fir2b;
	pcmpcm_fir_r pcm_fir3;
	
	real_t gain;
	bool sse2_enabled;

	real_t pcm_temp1[DSDPCM_MAX_CHANNELS * (DSDxFs8 / 75)];
	real_t pcm_temp2[DSDPCM_MAX_CHANNELS * (DSDxFs4 / 75)];
	real_t pcm_temp3[DSDPCM_MAX_CHANNELS * (DSDxFs4 / 75)];
	real_t pcm_tempo[DSDPCM_MAX_CHANNELS * (DSDxFs8 / 75)];

	bool conv_called;
	
public:
	dsdpcm_converter_r(conv_type_t conv_type);
	~dsdpcm_converter_r();
	int init(int channels, int dsd_samplerate, int pcm_samplerate);
	float get_delay();
	bool is_convert_called();
	int convert(uint8_t* dsd_data, int32_t* pcm_data, int dsd_samples);
	int convert(uint8_t* dsd_data, float* pcm_data, int dsd_samples);
	void set_gain(float dB_gain);
private:
	int set_ctables(int32_t* fir_coefs, int fir_length, ctable_r* fir_ctables);
	void set_coefs(const int32_t* int_coefs, int fir_length, real_t* real_coefs);
	void preinit();
	int convert_internal(uint8_t* dsd_data, real_t* pcm_data, int dsd_samples);
	int convert_internal_sse(uint8_t* dsd_data, real_t* pcm_data, int dsd_samples);
};

#endif
