/*
* SACD Decoder plugin
* Copyright (c) 2011 Maxim V.Anisiutkin <maxim.anisiutkin@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with FFmpeg; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#ifndef _DSDPCM_CONV_IMPL_H_INCLUDED
#define _DSDPCM_CONV_IMPL_H_INCLUDED

#include <stdint.h>

#define DSDxFs1   (44100 * 1)
#define DSDxFs2   (44100 * 2)
#define DSDxFs4   (44100 * 4)
#define DSDxFs8   (44100 * 8)
#define DSDxFs64  (44100 * 64)
#define DSDxFs128 (44100 * 128)

#define CTABLES(fir_length) ((fir_length + 7) / 8)

#define DSDPCM_MAX_CHANNELS  8
#define DSDPCM_MAX_SAMPLES   (DSDxFs128 / 75 / 8 * DSDPCM_MAX_CHANNELS)
#define DSDPCM_GAIN_0        8388608

#define DSDFIR1_8_LENGTH  80
#define DSDFIR1_16_LENGTH 160
#define PCMFIR2_2_LENGTH  27
#define PCMFIR3_2_LENGTH  151
#define PCMFIR_OFFSET     0x7fffffff
#define PCMFIR_SCALE      31

typedef double real_t;

typedef int32_t ctable_i[256];
typedef real_t  ctable_r[256];

typedef int32_t sample_i[DSDPCM_MAX_CHANNELS];
typedef real_t  sample_r[DSDPCM_MAX_CHANNELS];

class fir_buffer_i {
	sample_i* buffer;
	int       length;
	int       index;
public:
	fir_buffer_i() {
		buffer = NULL;
		length = 0;
		index = 0;
	};
	~fir_buffer_i() {
		if (buffer) {
			delete[] buffer;
		}
	};
	int get_length() {
		return length;
	};
	int get_index() {
		return index;
	};
	void set_index(int index) {
		this->index = index;
	};
	sample_i& operator[](int i) {
		return buffer[i];
	}
	sample_i* init(int length, int32_t value = (int32_t)0) {
		if (buffer) {
			delete[] buffer;
		}
		buffer = new sample_i[2 * length];
		if (buffer) {
			for (int sample = 0; sample < 2 * length; sample++) {
				for (int ch = 0; ch < DSDPCM_MAX_CHANNELS; ch++) {
					buffer[sample][ch] = value;
				}
			}
			this->length = length;
		}
		else {
			this->length = 0;
		}
		index = 0;
		return buffer;
	};
};

class fir_buffer_r {
	sample_r* buffer;
	int       length;
	int       index;
public:
	fir_buffer_r() {
		buffer = NULL;
		length = 0;
		index = 0;
	};
	~fir_buffer_r() {
		if (buffer) {
			delete[] buffer;
		}
	};
	int get_length() {
		return length;
	};
	int get_index() {
		return index;
	};
	void set_index(int index) {
		this->index = index;
	};
	sample_r& operator[](int i) {
		return buffer[i];
	}
	sample_r* init(int length, real_t value = (real_t)0) {
		if (buffer) {
			delete[] buffer;
		}
		buffer = new sample_r[2 * length];
		if (buffer) {
			for (int sample = 0; sample < 2 * length; sample++) {
				for (int ch = 0; ch < DSDPCM_MAX_CHANNELS; ch++) {
					buffer[sample][ch] = value;
				}
			}
			this->length = length;
		}
		else {
			this->length = 0;
		}
		index = 0;
		return buffer;
	};
};

class dsdpcm_fir_i {
	fir_buffer_i fir_buffer;
	int          fir_length;
	int          channels;
	int          decimation;
	ctable_i*    fir_ctables;
public:
	dsdpcm_fir_i() {
		fir_length = 0;
		channels = 0;
	}
	void init(ctable_i* fir_ctables, int fir_length, int channels, int decimation);
	int  run(uint8_t* dsd_data, int32_t* pcm_data, int dsd_samples);
};

class dsdpcm_fir_r {
	fir_buffer_r fir_buffer;
	int          fir_length;
	int          channels;
	int          decimation;
	ctable_r*    fir_ctables;
public:
	dsdpcm_fir_r() {
		fir_length = 0;
		channels = 0;
	}
	void init(ctable_r* fir_ctables, int fir_length, int channels, int decimation);
	int  run(uint8_t* dsd_data, real_t* pcm_data, int dsd_samples);
};

class pcmpcm_fir_i {
	int32_t* fir_coefs;
	int offset;
	int scale;
public:
	void init(int32_t* fir_coefs, int fir_length, int channels, int decimation, int offset = PCMFIR_OFFSET, int scale = PCMFIR_SCALE);
	int  run(int32_t* pcm_data, int32_t* out_data, int pcm_samples);
};

class pcmpcm_fir_r {
	int32_t* fir_coefs;
	int offset;
	int scale;
public:
	void init(int32_t* fir_coefs, int fir_length, int channels, int decimation, int offset = PCMFIR_OFFSET, int scale = PCMFIR_SCALE);
	int  run(real_t* pcm_data, real_t* out_data, int pcm_samples);
};

enum conv_mode_t {DSD64_44100, DSD64_88200, DSD64_176400, DSD64_352800, DSD128_44100, DSD128_88200, DSD128_176400, DSD128_352800};

class dsdpcm_conv_impl_t {
protected:
	conv_mode_t conv_mode;
	int         channels;
	int         dsd_samplerate;
	int         pcm_samplerate;
	float       gain_adjust;
public:
	dsdpcm_conv_impl_t() {
		channels = 0;
		dsd_samplerate = 0;
		pcm_samplerate = 0;
	}
	virtual int init(int channels, int dsd_samplerate, int pcm_samplerate) = 0;
	virtual int convert(uint8_t* dsd_data, int32_t* pcm_data, int dsd_samples) = 0;
	virtual int convert(uint8_t* dsd_data, float* pcm_data, int dsd_samples) = 0;
	virtual void set_gain(float dB_gain) = 0;
};

class dsdpcm_conv_impl_i : public dsdpcm_conv_impl_t {
	static ctable_i dsd_fir1_8_ctables[CTABLES(DSDFIR1_8_LENGTH)];
	static ctable_i dsd_fir1_16_ctables[CTABLES(DSDFIR1_16_LENGTH)];
	dsdpcm_fir_i dsd_fir1;
	pcmpcm_fir_i pcm_fir2a;
	pcmpcm_fir_i pcm_fir2b;
	pcmpcm_fir_i pcm_fir3;
	int32_t pcm_temp1[DSDPCM_MAX_SAMPLES];
	int32_t pcm_temp2[DSDPCM_MAX_SAMPLES];
	int32_t pcm_temp3[DSDPCM_MAX_SAMPLES];
	int32_t pcm_temp4[DSDPCM_MAX_SAMPLES];
	int32_t gain0;
	int32_t gain;
	int64_t out_minval;
	int64_t out_maxval;
public:
	dsdpcm_conv_impl_i() : dsdpcm_conv_impl_t() {
	}
	int init(int channels, int dsd_samplerate, int pcm_samplerate);
	int convert(uint8_t* dsd_data, int32_t* pcm_data, int dsd_samples);
	int convert(uint8_t* dsd_data, float* pcm_data, int dsd_samples);
	void set_gain(float dB_gain);
private:
	void set_ctables(int32_t* fir_coefs, int fir_length, ctable_i* fir_ctables);
	void preinit();
	int convert_internal(uint8_t* dsd_data, int32_t* pcm_data, int dsd_samples);
	int fracmul(int32_t* pcm_data, int32_t* out_data, int pcm_samples);
};

class dsdpcm_conv_impl_r : public dsdpcm_conv_impl_t {
	static ctable_r dsd_fir1_8_ctables[CTABLES(DSDFIR1_8_LENGTH)];
	static ctable_r dsd_fir1_16_ctables[CTABLES(DSDFIR1_16_LENGTH)];
	dsdpcm_fir_r dsd_fir1;
	pcmpcm_fir_r pcm_fir2a;
	pcmpcm_fir_r pcm_fir2b;
	pcmpcm_fir_r pcm_fir3;
	real_t pcm_temp1[DSDPCM_MAX_SAMPLES];
	real_t pcm_temp2[DSDPCM_MAX_SAMPLES];
	real_t pcm_temp3[DSDPCM_MAX_SAMPLES];
	real_t pcm_temp4[DSDPCM_MAX_SAMPLES];
	real_t gain0;
	real_t gain;
	real_t out_minval;
	real_t out_maxval;
public:
	dsdpcm_conv_impl_r() : dsdpcm_conv_impl_t() {
	}
	int init(int channels, int dsd_samplerate, int pcm_samplerate);
	int convert(uint8_t* dsd_data, int32_t* pcm_data, int dsd_samples);
	int convert(uint8_t* dsd_data, float* pcm_data, int dsd_samples);
	void set_gain(float dB_gain);
private:
	void set_ctables(int32_t* fir_coefs, int fir_length, ctable_r* fir_ctables);
	void preinit();
	int convert_internal(uint8_t* dsd_data, real_t* pcm_data, int dsd_samples);
};

#endif
