/*
 * radio.cpp - FM radio GUI
 *
 * Copyright (C) 2010-2011 Serge Begger
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "radio.h"
#include "ui_radio.h"

Radio::Radio(QWidget *parent) :
    abstractmmplugin(parent),
    ui(new Ui::Radio)
{
    ui->setupUi(this);

    modulename = "radio-hd";
    changeStyle(themes::instance()->getStyleSheet());
    connect(core,SIGNAL(play()),this,SLOT(play()));
    connect(core,SIGNAL(pause()),this,SLOT(pause()));
    connect(core,SIGNAL(prev()),this,SLOT(prev()));
    connect(core,SIGNAL(next()),this,SLOT(next()));
    connect(core,SIGNAL(endTextEdit(QString)),this,SLOT(endTextEdit(QString)));

    setThemeIcon();
    qsSlider = new gslider(ui->signalFrame);
    qsSlider->setMinimumSize(QSize(40, 325));
    qsSlider->setMaxValue(100);
    qsSlider->setMinValue(0);
    qsSlider->show();
    qsSlider->setValue(0);

    freqPanel = new numPad(this,true,QRect(0,115,225,315));
    freqPanel->helpLabel->setText("FREQ: 87.9 - 108 Mhz");
    connect(freqPanel, SIGNAL(putFreq()), this, SLOT(updateUiFreq()));
    connect(freqPanel, SIGNAL(endPutFreq()), this, SLOT(setManFreq()));

    timeloop = 10;

    min_freq = 87.90;
    max_freq = 108.00;
    currFreq = min_freq;

    QString querytext = "create table radiostation ( numstation int,"
            "namestation varchar(100), frequency float)";
    maindb = new qcpcdb(querytext,"radiostation",3);

    //Load settings
    mySet = new QSettings("qcarpc", modulename,this);
    mySet->setPath(QSettings::IniFormat,QSettings::UserScope,"qcarpc");
    currVolume = mySet->value("volume").toInt();
    currVolume = currVolume == 0 ? 20: currVolume;

    hd_tuner = new hd_unit(this);
    connect(hd_tuner,SIGNAL(showMessage(QString)),this,SIGNAL(setMessageText(QString)));

    currStation = mySet->value("currentStation").toInt();
    if (currStation == 0) currStation = 1;
    if (currFreq == 0) currFreq = min_freq;

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(updateGUI()));

    scanTimer = new QTimer(this);
    connect(scanTimer, SIGNAL(timeout()), this, SLOT(startScan()));

    seekTimer = new QTimer(this);
    connect(seekTimer, SIGNAL(timeout()), this, SLOT(autoSeekStation()));

    emit isStarted(modulename);
}

Radio::~Radio()
{
    mm_core->stop();
    emit setMessageText("");
    delete hd_tuner;
    delete ui;
}

void Radio::init() {

    mm_core->stop();

    QString inDevPath = mySet->value("indevpath").toString();

    inDevPath = inDevPath.isEmpty() ? "default": inDevPath;

    if(!hd_tuner->deviceOpen) {
        ui->typeSignalLabel->setText("TUNER NOT CONNECTED");
        return;
    }

    if(!timer->isActive()){
        timer->start (100);
    }

    if (!mm_core->playLink(modulename, qcarpc::analog_input,inDevPath,0))
        return;

    currFreq = returnFreq();
    setTunerFreq();

    mm_core->setVolume(currVolume);
}

void Radio::setThemeIcon() {
    ui->setFreqButton->setIcon(QIcon(themes::instance()->getIconFileName("button_numpad")));
    ui->startScanButton->setIcon(QIcon(themes::instance()->getIconFileName("button_search")));
    ui->eqButton->setIcon(QIcon(themes::instance()->getIconFileName("button_eq")));
    ui->seekUpButton->setIcon(QIcon(themes::instance()->getIconFileName("button_vol_up")));
    ui->seekDownButton->setIcon(QIcon(themes::instance()->getIconFileName("button_vol_down")));
}

void Radio::on_startScanButton_clicked()
{
    myMessage message;
    message.exec(3,"Search station?");
    if (!message.result)
        return;

    if(scanTimer->isActive())
    {
        scanTimer->stop();
        mm_core->setMute(false);
        return;
    }

    mm_core->setMute(true);
    ui->typeSignalLabel->setText("");
    if(maindb->model->rowCount()>0) {
        maindb->model->removeRows(0,maindb->model->rowCount());
        maindb->model->submitAll();
    }

    emit setMessageText("Search station....");

    currFreq = min_freq;
    currStation = 1;

    if(!scanTimer->isActive()){
        scanTimer->start(50);
    }
}

void Radio::on_seekDownButton_pressed()
{
    range = -1;
    startTimer();
}

void Radio::on_seekUpButton_pressed()
{
    range = 1;
    startTimer();
}

void Radio::on_setFreqButton_pressed()
{
    bool frPanVis = freqPanel->isVisible();

    if (!frPanVis)
        freqPanel->onShow();
    else
        freqPanel->offShow();
}

void Radio::setManFreq()
{
    float tempFreq = freqPanel->freq.toFloat();
    if (tempFreq >= min_freq && tempFreq <= max_freq) {
        currFreq = tempFreq;
        setTunerFreq();
    }
}

void Radio::startTimer()
{
    QTimer::singleShot(300, this, SLOT(endTimer()));
}

void Radio::endTimer()
{
    if(seekTimer->isActive())  seekTimer->stop();
    if(scanTimer->isActive())  scanTimer->stop();
    if (ui->seekUpButton->isDown() || ui->seekDownButton->isDown()){
        seekTimer->start(100);
    } else {
        if(scanTimer->isActive()){
            scanTimer->stop();
            return;
        }
        seekStation();
    }
}

void Radio::startScan()
{    
    scanTimer->blockSignals(true);
    currFreq = currFreq + 0.05;
    if(currFreq >max_freq)
    {
        mm_core->setMute(false);
        currFreq = max_freq;
        scanTimer->stop();
        if(maindb->model->rowCount() > 0) {
            currStation = 1;
            currFreq = returnFreq();
            setTunerFreq();
        }
        updateGUI();
        return;
    }
    hd_tuner->setFreq(currFreq);
    QEventLoop loop; QTimer::singleShot(timeloop, &loop, SLOT(quit())); loop.exec();
    int signal = hd_tuner->getSNR();
    if (signal > 100) {
        QTimer::singleShot(400, &loop, SLOT(quit())); loop.exec();
        signal = hd_tuner->getSNR();
        if (signal > 130) {
            QTimer::singleShot(800, &loop, SLOT(quit())); loop.exec();
            signal = hd_tuner->getSNR();
            if (signal > 140) {
                if (hd_tuner->getRSSI() > 7) {
                    saveCurrStation();
                    currStation++;
                    emit setMessageText("Add new station -- "+QString::number(currStation-1));
                }
            }
        }
    }
    scanTimer->blockSignals(false);
}

void Radio::seekStation()
{
    currFreq = currFreq + 0.05 * range;
    if (currFreq > max_freq) {
        currFreq = max_freq;
    } else {
        if (currFreq < min_freq) {
            currFreq = min_freq;
        }
    }
    setTunerFreq();
}

void Radio::setTunerFreq()
{
    mm_core->setMute(true);
    QEventLoop loop; QTimer::singleShot(timeloop, &loop, SLOT(quit())); loop.exec();
    hd_tuner->setFreq(currFreq);
    mySet->setValue("currentStation",currStation);
    QTimer::singleShot(timeloop, &loop, SLOT(quit())); loop.exec();
    mm_core->setMute(false);
}

void Radio::autoSeekStation()
{
    mm_core->setMute(true);
    float step = 0.05;
    currFreq = currFreq + step * range;
    if (currFreq > max_freq) {
        currFreq = max_freq;
    } else {
        if (currFreq < min_freq) {
            currFreq = min_freq;
        }
    }
    hd_tuner->setFreq(currFreq);
    QEventLoop loop; QTimer::singleShot(timeloop, &loop, SLOT(quit())); loop.exec();
    int signal = hd_tuner->getSNR();
    if (signal > 100) {
        QTimer::singleShot(400, &loop, SLOT(quit())); loop.exec();
        signal = hd_tuner->getSNR();
        if (signal > 130) {
            QTimer::singleShot(800, &loop, SLOT(quit())); loop.exec();
            signal = hd_tuner->getSNR();
            if (signal > 140) {
                if (hd_tuner->getRSSI() > 7) {
                    seekTimer->stop();
                    mm_core->setMute(false);
                }
            }
        }
    }
}

void Radio::updateGUI()
{
    if(!freqPanel->isVisible())
        ui->freqencyLcd->setText(QString::number(currFreq,'f',2));

    qsSlider->setValue(hd_tuner->getSNR()/2);
    //update stereo
    bool is_stereo = hd_tuner->isStereo();
    if (ui->typeSignalLabel->text() != "NO TUNER")
        ui->typeSignalLabel->setText(is_stereo ? "STEREO" : "MONO");
}

void Radio::updateUiFreq()
{
    ui->freqencyLcd->setText(QString::number(freqPanel->freq.toFloat(),'f',2));
}

float Radio::returnFreq()
{
    float tempFreq = currFreq;

    tempFreq = maindb->getDbValue(currStation-1,2).toFloat();
    QString stationName = maindb->getDbValue(currStation-1,1).toString();
    emit setMessageText("-- "+QString::number(currStation)+" -- "+stationName,0,stationName);

    if (tempFreq == 0)
        tempFreq = min_freq;

    return tempFreq;
}

void Radio::saveCurrStation(QString nameStation)
{
    int rows = maindb->model->rowCount();
    if (rows >= currStation) {
        rows = currStation-1;
        if (nameStation.isEmpty()) {
            nameStation = maindb->getDbValue(currStation-1,1).toString();
        }
    } else {
        maindb->model->insertRows(rows, 1);
        nameStation = "noname";
    }

    maindb->model->setData(maindb->model->index(rows, 0), currStation);
    maindb->model->setData(maindb->model->index(rows, 1), nameStation);
    maindb->model->setData(maindb->model->index(rows, 2), currFreq);
    maindb->model->submitAll();
}

void Radio::endTextEdit(QString text) {
    saveCurrStation(text);
    QString stationName = maindb->getDbValue(currStation-1,1).toString();
    emit setMessageText("-- "+QString::number(currStation)+" -- "+stationName,0,stationName);
}

void Radio::stop() {

}

void Radio::pause() {

}

void Radio::play() {
    saveCurrStation();
    emit setMessageText("Station saved",100);
    QEventLoop loop; QTimer::singleShot(400, &loop, SLOT(quit())); loop.exec();
    currFreq = returnFreq();
    emit isStoped();
}

void Radio::next() {
    currStation++;
    int maxNum = maindb->model->rowCount();
    if (currStation > maxNum) {
        currStation = maxNum + 1;
        emit setMessageText("--"+QString::number(currStation)+" -- empty station");
    } else {
        currFreq = returnFreq();
        setTunerFreq();        
    }
}

void Radio::prev() {
    currStation--;
    if (currStation <= 0) {
        currStation = 1;
    }
    currFreq = returnFreq();
    setTunerFreq();
}
