/*
	multimedia keyboard
	code based on examples from http://www.obdev.at/
	(V-USB is a software-only implementation of a low-speed USB device for Atmel’s AVR® microcontrollers)
	
	autor: Sergey Ivaykin (iwaikin@hotmail.com)
	license: GNU GPL
*/
#include <stdlib.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#define _NOP() do { __asm__ __volatile__ ("nop"); } while (0)

#include "usbdrv.h"
#include "oddebug.h"

#define UTIL_BIN4(x)        (uchar)((0##x & 01000)/64 + (0##x & 0100)/16 + (0##x & 010)/4 + (0##x & 1))
#define UTIL_BIN8(hi, lo)   (uchar)(UTIL_BIN4(hi) * 16 + UTIL_BIN4(lo))

#ifndef NULL
	#define NULL    ((void *)0)
#endif

// constants
#define adcPin		3	// select the input pin for the potentiometer
#define encPin0		4	// the number of the encoder pin
#define encPin1		1	// the number of the encoder pin

#define BUFFER_LEN	2

#define KEY_COUNT	4
#define KEY_NOKEY	0xff
#define DELTA		10

#define spin_undef	0
#define spin_left	1
#define spin_right	2

// variables for keys
#include "varForKeys.c"
int adcBuffer[BUFFER_LEN];
char keyPressed = 0;
static uchar    adcPending;

// variables for encoder
static uchar ls;
static uchar cs;
static char encCurrKey = 0;
static char encLastKey = 0;
static char rt[16] = {
  spin_undef, spin_left,  spin_right, spin_undef,
  spin_right, spin_undef, spin_undef, spin_left,
  spin_left,  spin_undef, spin_undef, spin_right,
  spin_undef, spin_right, spin_left,  spin_undef
};
static uchar msgEnc[] = { 0x00, 0xea, 0xe9 }; // no_key, vol--, vol++
// -----------------
static uchar    idleRate;           /* in 4 ms units */
static uchar    reportBuffer[3];    /* buffer for HID reports */
// variables for usb
PROGMEM const char usbHidReportDescriptor[USB_CFG_HID_REPORT_DESCRIPTOR_LENGTH] =
{
	0x05, 0x01,			//	USAGE_PAGE (Generic Desktop)
	0x09, 0x06,			//	USAGE (Keyboard)
	0xa1, 0x01,			//	COLLECTION (Application)
	0x85, 0x01,			//		REPORT_ID (1)
	0x05, 0x07,			//		USAGE_PAGE (Keyboard)
	0x19, 0xe0,			//		USAGE_MINIMUM (Keyboard LeftControl)
	0x29, 0xe7,			//		USAGE_MAXIMUM (Keyboard RightGUI)
	0x15, 0x00,			//		LOGICAL_MINIMUM (0)
	0x25, 0x01,			//		LOGICAL_MAXIMUM (1)
	0x75, 0x01,			//		REPORT_SIZE (1)
	0x95, 0x08,			//		REPORT_COUNT (8)
	0x81, 0x02,			//		INPUT (Data,Var,Abs)
	0x95, 0x01,			//		REPORT_COUNT (6) -- ??? 1
	0x75, 0x08,			//		REPORT_SIZE (8)
	0x25, 0x65,			//		LOGICAL_MAXIMUM (101)
	0x19, 0x00,			//		USAGE_MINIMUM (Reserved (no event indicated))
	0x29, 0x65,			//		USAGE_MAXIMUM (101)
	0x81, 0x00,			//		INPUT (Data,Ary,Abs)
	0xc0,				//	END_COLLECTION

	0x05, 0x0c,			//	USAGE_PAGE (Consumer Devices)
	0x09, 0x01, 		//	USAGE (Consumer Control)
	0xa1, 0x01,			//	COLLECTION (Application)
	0x85, 0x02,			//		REPORT_ID (2)
	0x19, 0x00,			//		USAGE_MINIMUM (Unassigned)
	0x2a, 0x3c, 0x02,	//		USAGE_MAXIMUM (AC Format)
	0x15, 0x00,			//		LOGICAL_MINIMUM (0)
	0x26, 0x3c, 0x02,	//		LOGICAL_MAXIMUM (572)
	0x95, 0x01,			//		REPORT_COUNT (1)
	0x75, 0x10,			//		REPORT_SIZE (16)
	0x81, 0x00,			//		INPUT (Data,Var,Abs)
	0xc0				//	END_COLLECTION
};

/* --- keyboard routines --- */
static void keyPushData(int value) {
  uchar i;
  for (i = BUFFER_LEN - 1; i; i--) adcBuffer[i] = adcBuffer[i - 1];
  adcBuffer[0] = value;
}

static char keyEvaluate(void) {
  uchar i;
  int val = 0;
  char retValue = KEY_NOKEY;
  
  for (i = 0; i < BUFFER_LEN; i++) val += adcBuffer[i];
  val /= BUFFER_LEN;
  
  for (i = 0; i < KEY_COUNT; i++) {
    if (abs(adcValues[i] - val) <= DELTA) {
      retValue = i;
      break;
    }
  }
  
  return retValue;
}

static char keyTest(void) {
  uchar i;
  char bufferFilled;
  int maxDifference = 0;

  for (i = 0; i < BUFFER_LEN - 1; i++) {
	maxDifference = (maxDifference > abs(adcBuffer[i] - adcBuffer[i + 1]) ? maxDifference : abs(adcBuffer[i] - adcBuffer[i + 1]));
  }
  if (maxDifference <= DELTA) bufferFilled = 1;
  else bufferFilled = 0;
  
  return bufferFilled;
}
/* ------------------------- */
/* --- encoder  routines --- */

static uchar encGetPins(void) {
	uchar tmp = PINB;
	tmp = ((tmp & 0b00010000) >> 4) | (tmp & 0b00000010);
	return tmp;
}

static int encRead(void) {
	uchar lscs;

	cs = encGetPins();

	if (cs != ls) {
		lscs = (ls << 2) | cs;
		ls = cs;
		return rt[lscs];
	}

	return spin_undef;
}
/* ------------------------- */

static void setup() {
	uchar i;

	/* Define directions for port pins */
	PORTB = 0b00010010; // (1<<PB4) | (1<<PB1);
	DDRB =  0;
	/* Insert nop for synchronization*/
	_NOP();

	// init kbd buffer
	for (i = 0; i < BUFFER_LEN; i++) adcBuffer[i] = 0;
	// init encoder buffer
	cs  = encGetPins();
	ls = cs;
	encCurrKey = cs;
}
/*---------------------------------------------------------------------------*/
static void timerInit(void)
{
    TCCR1 = 0x0b;           /* select clock: 16.5M/1k -> overflow rate = 16.5M/256k = 62.94 Hz */
}

static void timerPoll(void)
{
	static uchar timerCnt;

    if (TIFR & (1 << TOV1)) {
        TIFR = (1 << TOV1); /* clear overflow */
        if (++timerCnt >= 4) {       /* ~ 1/30 second interval */
            timerCnt = 0;
			adcPending = 1;
			ADCSRA |= (1 << ADSC);  /* start next conversion */
        }
    }
}

static void adcInit(void)
{
    ADMUX  = UTIL_BIN8(0000, 0011); /* Vref = Vcc, measure ADC3 */
    ADCSRA = UTIL_BIN8(1000, 0111); /* enable ADC, not free running, interrupt disable, rate = 1/128 */
}

static void adcPoll(void)
{
    if (adcPending && !(ADCSRA & (1 << ADSC))) {
        adcPending = 0;
		keyPushData(ADC);
    }
}
/* ------------------------------------------------------------------------- */
//usb
static void buildReport(uchar reportID)
{
    reportBuffer[0] = reportID;    /* report ID */
    reportBuffer[1] = 0;
    reportBuffer[2] = 0;
}
static void keyBuildReport(uchar key)
{
    reportBuffer[0] = 1;    // report ID
    reportBuffer[1] = 0;
    reportBuffer[2] = key;
}
static void encBuildReport(uchar key)
{
    reportBuffer[0] = 2;    /* report ID */
    reportBuffer[1] = key;
    reportBuffer[2] = 0;
//	if (key) LED_ON
}

/* ------------------------ interface to USB driver ------------------------ */
usbMsgLen_t usbFunctionSetup(uchar data[8])
{
	usbRequest_t    *rq = (void *)data;

    usbMsgPtr = reportBuffer;
    if ((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) {    /* class request type */
        if (rq->bRequest == USBRQ_HID_GET_REPORT) {  /* wValue: ReportType (highbyte), ReportID (lowbyte) */
            /* we only have one report type, so don't look at wValue */
            buildReport(rq->wValue.bytes[0]);
            return sizeof(reportBuffer);
        } else if (rq->bRequest == USBRQ_HID_GET_IDLE) {
            usbMsgPtr = &idleRate;
            return 1;
        } else if (rq->bRequest == USBRQ_HID_SET_IDLE) {
            idleRate = rq->wValue.bytes[1];
        }
    } else {
        /* no vendor specific requests implemented */
    }
	return 0;
}
/* ------------------------ Oscillator Calibration ------------------------- */
static void calibrateOscillator(void)
{
	uchar       step = 128;
	uchar       trialValue = 0, optimumValue;
	int         x, optimumDev, targetValue = (unsigned)(1499 * (double)F_CPU / 10.5e6 + 0.5);

    /* do a binary search: */
    do {
        OSCCAL = trialValue + step;
        x = usbMeasureFrameLength();    /* proportional to current real frequency */
        if (x < targetValue)             /* frequency still too low */
            trialValue += step;
        step >>= 1;
    } while (step > 0);
    /* We have a precision of +/- 1 for optimum OSCCAL here */
    /* now do a neighborhood search for optimum value */
    optimumValue = trialValue;
    optimumDev = x; /* this is certainly far away from optimum */
    for (OSCCAL = trialValue - 1; OSCCAL <= trialValue + 1; OSCCAL++) {
        x = usbMeasureFrameLength() - targetValue;
        if (x < 0)
            x = -x;
        if (x < optimumDev) {
            optimumDev = x;
            optimumValue = OSCCAL;
        }
    }
    OSCCAL = optimumValue;
}
void usbEventResetReady(void)
{
    /* Disable interrupts during oscillator calibration since
     * usbMeasureFrameLength() counts CPU cycles.
     */
    cli();
    calibrateOscillator();
    sei();
    eeprom_write_byte(0, OSCCAL);   /* store the calibrated value in EEPROM */
}
/* --------------------------------- main ---------------------------------- */
int main(void) {
	uchar   i;
	uchar	key;
	uchar   calibrationValue;

	setup();

    calibrationValue = eeprom_read_byte(0); /* calibration value from last time */
    if (calibrationValue != 0xff) {
        OSCCAL = calibrationValue;
    }
    odDebugInit();
    usbDeviceDisconnect();
    for (i = 0; i < 20; i++) {  /* 300 ms disconnect */
        _delay_ms(15);
    }
    usbDeviceConnect();

    wdt_enable(WDTO_1S);
    timerInit();
    usbInit();
    adcInit();
    sei();

    for (;;) {    /* main event loop */
		wdt_reset();
        usbPoll();
        if (usbInterruptIsReady()) { // we can send another key
			// encoder
			//key = encEvaluate();
			key = encCurrKey;
			if (encLastKey != key) {
				encLastKey = key;
				encBuildReport(msgEnc[key]);
				usbSetInterrupt(reportBuffer, sizeof(reportBuffer));
			}
        }
        timerPoll();

		wdt_reset();
        usbPoll();
        if (usbInterruptIsReady()) { // we can send another key
			// keyboard
			if (keyTest()) {
				key = keyEvaluate();
				if (key != KEY_NOKEY) {
					keyPressed = 1;
					keyBuildReport(keyMsg[key]);
					usbSetInterrupt(reportBuffer, sizeof(reportBuffer));
				} 
			} else {
				if (keyPressed) {
					keyPressed = 0;
					keyBuildReport(0);
					usbSetInterrupt(reportBuffer, sizeof(reportBuffer));
				}
			}
        }
        timerPoll();

        adcPoll();
		//encPushData(encRead());
		encCurrKey = encRead();
	}
    return 0;
}
