/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "tests/support.h"
#include "libsyncml/sml_error_internals.h"
#include "libsyncml/sml_notification_internals.h"
#include <libsyncml/data_sync_api/sml_data_sync_defines.h>

/* If the notification is a normal SyncML message then a manager is
 * required. The easiest way are two function which create and destroy
 * a default manager - only for testing.
 */

SmlTransport *_tsp = NULL;

SmlManager* _start_manager()
{
	unsigned int defaultMaxMsgSize = 10240;
	unsigned int defaultMaxObjSize = 1024000;

	GError *error = NULL;
	_tsp = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	sml_fail_unless(_tsp != NULL, "%s", GET_ERROR_MESSAGE(error));
	sml_fail_unless(error == NULL, "%s", GET_ERROR_MESSAGE(error));
	
	SmlManager *manager = smlManagerNew(_tsp, &error);
	sml_fail_unless(manager != NULL, "%s", GET_ERROR_MESSAGE(error));
	sml_fail_unless(error == NULL, "%s", GET_ERROR_MESSAGE(error));

	smlManagerSetLocalMaxMsgSize(manager, defaultMaxMsgSize);
	smlManagerSetLocalMaxObjSize(manager, defaultMaxObjSize);

	return manager;
}

void _stop_manager(SmlManager *manager)
{
	smlManagerFree(manager);
	smlTransportFree(_tsp);
}

START_TEST (san_new)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_USER, 0, "test", "/", SML_MIMETYPE_UNKNOWN, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_empty)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_UNSPECIFIED, SML_SAN_INITIATOR_USER, 0, "tttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	unsigned char target[] = {
		/* old implementation without digest support
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		*/
		0x95, 0x5d, 0x6f, 0x61, 0xa4, 0x20, 0xb1, 0xa7, 0x46, 0x00, 0xc1, 0x12, 0x81, 0x06, 0x5e, 0x64, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x04, 0x74, 0x74, 0x74, 0x74, //server identifier
		0x00}; // num syncs + future
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(size == sizeof(target), NULL);
	sml_fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_empty2)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	unsigned char target[] = {
		/* old implementation without digest support
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		*/
		0x9e, 0x0d, 0x43, 0x31, 0x4f, 0x18, 0xb6, 0x96, 0x6d, 0x88, 0xf1, 0xa2, 0xdb, 0x00, 0x06, 0x15, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x05, 0x74, 0x74, 0x74, 0x74, 0x74, //server identifier
		0x00}; // num syncs + future
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(size == sizeof(target), NULL);
	sml_fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_alert)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	unsigned char target[] = {
		/* old implementation without digest support
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		*/
		0x7e, 0x39, 0x00, 0x91, 0x5e, 0x39, 0x3a, 0x69, 0xd2, 0x60, 0x18, 0x1f, 0x65, 0x65, 0x49, 0xe5, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x05, 0x74, 0x74, 0x74, 0x74, 0x74, //server identifier
		0x10,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x04, 0x74, 0x74, 0x74, 0x74}; //Server URI;
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(size == sizeof(target), NULL);
	sml_fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_alert2)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_REFRESH_FROM_SERVER_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	unsigned char target[] = {
		/* old implementation without digest support
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		*/
		0x6d, 0x47, 0xd6, 0x78, 0xb2, 0xe0, 0xa3, 0x72, 0x75, 0x03, 0xcc, 0xae, 0xd0, 0x39, 0x3d, 0xea, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x05, 0x74, 0x74, 0x74, 0x74, 0x74, //server identifier
		0x20,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x04, 0x74, 0x74, 0x74, 0x74, //Server URI;
		0xA0, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x04, 0x74, 0x74, 0x74, 0x74}; //Server URI;
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(size == sizeof(target), NULL);
	sml_fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_alert_auth)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "OpenSync", "/", SML_MIMETYPE_UNKNOWN, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "contacts", &error), NULL);
	sml_fail_unless(error == NULL, NULL);

	SmlCred *cred = smlCredNewAuth(
				SML_AUTH_TYPE_MD5,
				"username",
				"password",
				&error);
	smlNotificationSetCred(san, cred);
	smlCredUnref(cred);
	
	unsigned char target[] = {
		0x61, 0x59, 0x71, 0x16, 0x2e, 0x7d, 0x3e, 0x83, 0xa8, 0x23, 0x48, 0xa8, 0x5b, 0x21, 0x79, 0x49, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x08, 0x4f, 0x70, 0x65, 0x6e, 0x53, 0x79, 0x6e, 0x63, //server identifier
		0x10,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x08, 0x63, 0x6f, 0x6e, 0x74, 0x61, 0x63, 0x74, 0x73}; //Server URI;
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(size == sizeof(target), NULL);
	sml_fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_11_assemble_parse)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_11, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_XML, &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlManager *manager = _start_manager();
	smlNotificationSetManager(san, manager);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	smlNotificationFree(san);

	_stop_manager(manager);
	smlSafeCFree(&buffer);

	/* SyncML 1.1 SAN parsing is not supported today.
	 * Such messages are assembled and parsed like
	 * normal SyncML messages with the default XML
	 * parser and assembler which is tested separately.
	 */
	//san = smlNotificationParse(buffer, size, &error);
	//sml_fail_unless(san != NULL, NULL);
	//sml_fail_unless(error == NULL, NULL);
	//g_free(buffer);
	
	//sml_fail_unless(smlNotificationGetVersion(san) == SML_SAN_VERSION_11, NULL);
	//sml_fail_unless(smlNotificationGetMode(san) == SML_SAN_UIMODE_USER, NULL);
	//sml_fail_unless(smlNotificationGetInitiator(san) == SML_SAN_INITIATOR_SERVER, NULL);
	//sml_fail_unless(smlNotificationGetSessionID(san) == 65535, NULL);
	//sml_fail_unless(!strcmp(smlNotificationGetIdentifier(san), "ttttt"), NULL);
	
	//sml_fail_unless(smlNotificationNumAlerts(san) == 1, NULL);
	//SmlSanAlert *alert = smlNotificationNthAlert(san, 0);
	//sml_fail_unless(alert != NULL, NULL);
	//sml_fail_unless(smlSanAlertGetType(alert) == SML_ALERT_TWO_WAY_BY_SERVER, NULL);
	//sml_fail_unless(!strcmp(smlSanAlertGetContentType(alert), SML_ELEMENT_TEXT_VCARD), NULL);
	//sml_fail_unless(!strcmp(smlSanAlertGetServerURI(alert), "tttt"), NULL);
	
	//smlNotificationFree(san);


}
END_TEST

START_TEST (san_parse)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x20,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00, //Server URI;
		0xA0, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00 }; //Server URI;
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationGetVersion(san) == SML_SAN_VERSION_12, NULL);
	sml_fail_unless(smlNotificationGetMode(san) == SML_SAN_UIMODE_USER, NULL);
	sml_fail_unless(smlNotificationGetInitiator(san) == SML_SAN_INITIATOR_SERVER, NULL);
	sml_fail_unless(smlNotificationGetSessionID(san) == 65535, NULL);
	sml_fail_unless(!strcmp(smlNotificationGetIdentifier(san), "ttttt"), NULL);
	
	sml_fail_unless(smlNotificationNumAlerts(san) == 2, NULL);
	SmlSanAlert *alert = smlNotificationNthAlert(san, 0);
	sml_fail_unless(alert != NULL, NULL);
	sml_fail_unless(smlSanAlertGetType(alert) == SML_ALERT_TWO_WAY_BY_SERVER, NULL);
	sml_fail_unless(!strcmp(smlSanAlertGetContentType(alert), SML_ELEMENT_TEXT_VCARD), NULL);
	sml_fail_unless(!strcmp(smlSanAlertGetServerURI(alert), "tttt"), NULL);
	
	alert = smlNotificationNthAlert(san, 1);
	sml_fail_unless(alert != NULL, NULL);
	sml_fail_unless(smlSanAlertGetType(alert) == SML_ALERT_REFRESH_FROM_SERVER_BY_SERVER, NULL);
	sml_fail_unless(!strcmp(smlSanAlertGetContentType(alert), SML_ELEMENT_TEXT_VCARD), NULL);
	sml_fail_unless(!strcmp(smlSanAlertGetServerURI(alert), "tttt"), NULL);
	
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_parse2)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x00, //server identifier
		0x00}; //num syncs
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationGetVersion(san) == SML_SAN_VERSION_12, NULL);
	sml_fail_unless(smlNotificationGetMode(san) == SML_SAN_UIMODE_UNSPECIFIED, NULL);
	sml_fail_unless(smlNotificationGetInitiator(san) == SML_SAN_INITIATOR_USER, NULL);
	sml_fail_unless(smlNotificationGetSessionID(san) == 0, NULL);
	sml_fail_unless(smlNotificationGetIdentifier(san) == NULL, NULL);
	
	sml_fail_unless(smlNotificationNumAlerts(san) == 0, NULL);
	
	smlNotificationFree(san);
}
END_TEST

START_TEST (san_parse_error)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x00}; //num syncs
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

START_TEST (san_parse_error2)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x01, //server identifier
		0x00}; //num syncs
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

START_TEST (san_parse_error3)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x01, 0x00,//server identifier
		0x10}; //num syncs
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

START_TEST (san_parse_error4)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x02, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x01, 0x00,//server identifier
		0x00}; //num syncs
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

START_TEST (san_parse_error5)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x50, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00}; //Server URI;
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

START_TEST (san_parse_error6)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x70, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x01}; //Server URI;
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

START_TEST (san_parse_error7)
{
	setup_testbed(NULL);
	
	unsigned char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x70, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x01, 0x00, 0x00}; //Server URI;
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationParse((char *) target, sizeof(target), &error);
	sml_fail_unless(san == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	SML_ERROR_FREE(error);
}
END_TEST

/*START_TEST (san_send)
{
	setup_testbed(NULL);
	
	GError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", &error);
	sml_fail_unless(san != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlNotificationNewAlert(san, SML_ALERT_REFRESH_FROM_SERVER_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "ttttaaaa", &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	char *buffer = NULL;
	gsize size = 0;
	sml_fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	sml_fail_unless(error == NULL, NULL);
	
	gboolean smlNotificationSend(SmlNotification *san, SmlTransport *tsp, SmlTransportSendCb callback, void *userdata, GError **error)
	
	
	sml_fail_unless(size == sizeof(target), NULL);
	sml_fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST*/

@SML_TESTCASE_CODE@

