/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "tests/support.h"

#include <libsyncml/dev_inf_api/sml_dev_inf_property.h>

#include <string.h>

START_TEST (dev_inf_property_create)
{
	setup_testbed(NULL);

	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	g_object_unref(prop);
}
END_TEST

START_TEST (dev_inf_property_empty_set)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);

	/* enforce parameter name */

	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, NULL, &error) == FALSE, "A property must have a name.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	/* this is allowed */
	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, "", &error) == TRUE, "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, NULL, &error) == FALSE, "A property must have a name.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	/* test the normal properties */

	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, NULL, &error), "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, NULL, &error), "%s", error?error->message:"No GError set.");

	/* ValEnums does not accept empty values */

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, NULL, &error) == FALSE, "ValEnums must be not empty.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "", &error) == FALSE, "ValEnums must be not empty.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, NULL, &error) == FALSE, "ValEnums must be not empty.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	/* parameter checks must be a little bit more careful */

	sml_fail_unless(sml_dev_inf_property_add_param(prop, NULL, &error) == FALSE, "Empty parameters must fail.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;
	SmlDevInfPropParam *propParam = sml_dev_inf_prop_param_new();
	sml_fail_unless(propParam != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, propParam, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(propParam);
	propParam = NULL;
	sml_fail_unless(sml_dev_inf_property_add_param(prop, NULL, &error) == FALSE, "Empty parameters must fail.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;

	g_object_unref(prop);
}
END_TEST

START_TEST (dev_inf_property_empty_get)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);

	/* enforce parameter name */

	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) == NULL, NULL);

	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, NULL, &error) == FALSE, "The parameter name must be set.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) == NULL, NULL);

	/* this is allowed */
	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, "", &error) == TRUE, "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) != NULL, NULL);
	sml_fail_unless(strlen(sml_dev_inf_property_get_prop_name(prop)) == 0, NULL);

	/* test the normal properties */

	sml_fail_unless(sml_dev_inf_property_get_data_type(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_data_type(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_data_type(prop) == NULL, NULL);

	sml_fail_unless(sml_dev_inf_property_get_display_name(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_display_name(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_display_name(prop) == NULL, NULL);

	/* ValEnums does not accept empty values */

	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, NULL, &error) == FALSE, "ValEnums must be not empty.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "", &error) == FALSE, "ValEnums must be not empty.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "value", &error) == TRUE, "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 1, NULL);

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, NULL, &error) == FALSE, "ValEnums must be not empty.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 1, NULL);

	/* parameter checks must be a little bit more careful */

	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 0, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, NULL, &error) == FALSE, "Empty parameters must fail.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 0, NULL);

	SmlDevInfPropParam *propParam = sml_dev_inf_prop_param_new();
	sml_fail_unless(propParam != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, propParam, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(propParam);
	propParam = NULL;
	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 1, NULL);

	sml_fail_unless(sml_dev_inf_property_add_param(prop, NULL, &error) == FALSE, "Empty parameters must fail.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 1, NULL);

	g_object_unref(prop);
}
END_TEST

START_TEST (dev_inf_property_size_integrity)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);

	sml_fail_unless(sml_dev_inf_property_set_size(prop, 10, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_size(prop, 0, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 10, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 0, &error) == TRUE, "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_property_set_size(prop, 10, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 10, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 20, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 0, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 5, &error) == FALSE, "The max size cannot be lower then the size.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_property_set_size(prop, 0, &error) == TRUE, "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 10, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_size(prop, 10, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_size(prop, 5, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_size(prop, 0, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_size(prop, 20, &error) == FALSE, "The size cannot be greater then the max size.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_property_set_max_size(prop, 0, &error) == TRUE, "%s", error?error->message:"No GError set.");

	g_object_unref(prop);
}
END_TEST

START_TEST (dev_inf_property_filled_set)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);

	/* tests of functions without error handling */

	sml_dev_inf_property_set_no_truncate(prop, TRUE);
	sml_dev_inf_property_set_no_truncate(prop, FALSE);
	sml_dev_inf_property_set_max_occur(prop, 10);
	sml_dev_inf_property_set_max_occur(prop, 0);

	/* tests of functions with error handling */

	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, "property name", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, "property data type", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, "property display name", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "property first value of enumeration", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "property second value of enumeration", &error), "%s", error?error->message:"No GError set.");

	/* parameter related tests */

	SmlDevInfPropParam *propParam = sml_dev_inf_prop_param_new();
	sml_fail_unless(propParam != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, propParam, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(propParam);
	propParam = sml_dev_inf_prop_param_new();
	sml_fail_unless(propParam != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, propParam, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(propParam);

	g_object_unref(prop);
}
END_TEST

START_TEST (dev_inf_property_filled_get)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);

	/* tests of functions without error handling */

	sml_fail_unless(sml_dev_inf_property_get_no_truncate(prop) == FALSE, "The default value of NoTruncate is wrong.", NULL);
	sml_dev_inf_property_set_no_truncate(prop, TRUE);
	sml_fail_unless(sml_dev_inf_property_get_no_truncate(prop) == TRUE, NULL);
	sml_dev_inf_property_set_no_truncate(prop, FALSE);
	sml_fail_unless(sml_dev_inf_property_get_no_truncate(prop) == FALSE, NULL);

	sml_fail_unless(sml_dev_inf_property_get_max_occur(prop) == 0, "The default value of MaxOccur is wrong.", NULL);
	sml_dev_inf_property_set_max_occur(prop, 10);
	sml_fail_unless(sml_dev_inf_property_get_max_occur(prop) == 10, NULL);
	sml_dev_inf_property_set_max_occur(prop, 0);
	sml_fail_unless(sml_dev_inf_property_get_max_occur(prop) == 0, NULL);

	/* tests of functions with error handling */

	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, "test_1", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) != NULL, NULL);
	sml_fail_unless(strcmp(sml_dev_inf_property_get_prop_name(prop), "test_1") == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_get_data_type(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_data_type(prop, "test_2", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_data_type(prop) != NULL, NULL);
	sml_fail_unless(strcmp(sml_dev_inf_property_get_data_type(prop), "test_2") == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_get_display_name(prop) == NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_set_display_name(prop, "test_3", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_get_display_name(prop) != NULL, NULL);
	sml_fail_unless(strcmp(sml_dev_inf_property_get_display_name(prop), "test_3") == 0, NULL);

	/* tests for enumerated values */

	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "test_4", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 1, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_val_enum(prop, 0) != NULL, NULL);
	sml_fail_unless(strcmp(sml_dev_inf_property_get_nth_val_enum(prop, 0), "test_4") == 0, NULL);

	sml_fail_unless(sml_dev_inf_property_add_val_enum(prop, "test_5", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_num_val_enums(prop) == 2, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_val_enum(prop, 0) != NULL, NULL);
	sml_fail_unless(strcmp(sml_dev_inf_property_get_nth_val_enum(prop, 0), "test_4") == 0, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_val_enum(prop, 1) != NULL, NULL);
	sml_fail_unless(strcmp(sml_dev_inf_property_get_nth_val_enum(prop, 1), "test_5") == 0, NULL);

	/* parameter related tests */

	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 0, NULL);

	SmlDevInfPropParam *propParam = sml_dev_inf_prop_param_new();
	sml_fail_unless(propParam != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, propParam, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 1, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_param(prop, 0) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_param(prop, 0) == propParam, NULL);
	g_object_unref(propParam);

	sml_fail_unless(sml_dev_inf_property_add_param(prop, NULL, &error) == FALSE, "Empty parameters must fail.", NULL);
	sml_fail_unless(error != NULL, "No GError set.", NULL);
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 1, NULL);

	propParam = sml_dev_inf_prop_param_new();
	sml_fail_unless(propParam != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_add_param(prop, propParam, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_property_num_params(prop) == 2, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_param(prop, 0) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_param(prop, 0) != propParam, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_param(prop, 1) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_property_get_nth_param(prop, 1) == propParam, NULL);
	g_object_unref(propParam);

	g_object_unref(prop);
}
END_TEST

START_TEST (dev_inf_property_references)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, "Cannot create SmlDevInfProperty.");
	sml_fail_unless(sml_dev_inf_property_set_prop_name(prop, "test name", &error), "%s", error?error->message:"No GError set.");

	g_object_ref(prop);

	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) != NULL, "The property name was not set.");

	g_object_unref(prop);

	sml_fail_unless(sml_dev_inf_property_get_prop_name(prop) != NULL, "The property name is already cleaned up.");

	g_object_unref(prop);
}
END_TEST

@SML_TESTCASE_CODE@

