/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "tests/support.h"

#include <libsyncml/dev_inf_api/sml_dev_inf_ctcap.h>

START_TEST (dev_inf_ctcap_create)
{
	setup_testbed(NULL);

	GError *error = NULL;

	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(NULL, &error);
	sml_fail_unless(ctcap == NULL, NULL);
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;

	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");

	ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	g_object_unref(ct);
	g_object_unref(ctcap);
}
END_TEST

START_TEST (dev_inf_ctcap_empty_set)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	g_object_unref(ct);
	ct = NULL;

	/* test enforcement of content type */

	sml_fail_unless(sml_dev_inf_ctcap_set_content_type(ctcap, NULL, &error) == FALSE, NULL);
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;

	/* property checks must be a little bit more careful */

	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, NULL, &error) == FALSE, "Empty properties must fail.");
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;

	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, prop, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(prop);
	prop = NULL;

	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, NULL, &error) == FALSE, "Empty properties must fail.");
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;

	g_object_unref(ctcap);
}
END_TEST

START_TEST (dev_inf_ctcap_empty_get)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");

	/* test enforcement of content type */

	sml_fail_unless(sml_dev_inf_ctcap_set_content_type(ctcap, NULL, &error) == FALSE, NULL);
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_ctcap_get_content_type(ctcap) == ct, "The original content type was replaced.");
	g_object_unref(ct);
	ct = NULL;

	/* property checks must be a little bit more careful */

	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, NULL, &error) == FALSE, "Empty properties must fail.");
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 0, NULL);

	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, prop, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(prop);
	prop = NULL;
	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 1, NULL);

	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, NULL, &error) == FALSE, "Empty properties must fail.");
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 1, NULL);

	g_object_unref(ctcap);
}
END_TEST

START_TEST (dev_inf_ctcap_filled_set)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	g_object_unref(ct);
	ct = NULL;

	/* content type related tests */

	ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");

	sml_fail_unless(sml_dev_inf_ctcap_set_content_type(ctcap, ct, &error), "%s", error?error->message:"No GError set.");

	g_object_unref(ct);
	ct = NULL;

	/* property related tests */

	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, prop, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(prop);
	prop = NULL;

	prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, prop, &error) == TRUE, "%s", error?error->message:"No GError set.");
	g_object_unref(prop);
	prop = NULL;

	g_object_unref(ctcap);
}
END_TEST

START_TEST (dev_inf_ctcap_filled_get)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");

	/* content type related tests */

	sml_fail_unless(sml_dev_inf_ctcap_get_content_type(ctcap) == ct, "The original content type is missing.");
	g_object_unref(ct);
	ct = NULL;

	ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_ctcap_set_content_type(ctcap, ct, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_ctcap_get_content_type(ctcap) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_content_type(ctcap) == ct, NULL);
	g_object_unref(ct);
	ct = NULL;

	/* property related tests */

	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 0, NULL);

	SmlDevInfProperty *prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, prop, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 1, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_nth_property(ctcap, 0) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_nth_property(ctcap, 0) == prop, NULL);
	g_object_unref(prop);
	prop = NULL;

	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, NULL, &error) == FALSE, "Empty properties must fail.");
	sml_fail_unless(error != NULL, "No GError set.");
	g_error_free(error);
	error = NULL;
	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 1, NULL);

	prop = sml_dev_inf_property_new();
	sml_fail_unless(prop != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_add_property(ctcap, prop, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_ctcap_num_properties(ctcap) == 2, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_nth_property(ctcap, 0) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_nth_property(ctcap, 0) != prop, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_nth_property(ctcap, 1) != NULL, NULL);
	sml_fail_unless(sml_dev_inf_ctcap_get_nth_property(ctcap, 1) == prop, NULL);
	g_object_unref(prop);
	prop = NULL;

	g_object_unref(ctcap);
}
END_TEST

START_TEST (dev_inf_ctcap_references)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	g_object_unref(ct);
	ct = NULL;

	g_object_ref(ctcap);

	sml_fail_unless(sml_dev_inf_ctcap_get_content_type(ctcap) != NULL, "The content type was not set.");

	g_object_unref(ctcap);

	sml_fail_unless(sml_dev_inf_ctcap_get_content_type(ctcap) != NULL, "The content type is already cleaned up.");

	g_object_unref(ctcap);
}
END_TEST

@SML_TESTCASE_CODE@

