/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "tests/support.h"

#include <libsyncml/dev_inf_api/sml_dev_inf.h>

START_TEST (dev_inf_create)
{
	setup_testbed(NULL);

	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);
	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_references)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error), "%s", error?error->message:"No GError set.");

	g_object_ref(devinf);

	sml_fail_unless(sml_dev_inf_get_dev_id(devinf) != NULL, "The device ID was not set.");

	g_object_unref(devinf);

	sml_fail_unless(sml_dev_inf_get_dev_id(devinf) != NULL, "The device ID is already cleaned up.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_enforce_ver_dtd)
{
	setup_testbed(NULL);

	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_ver_dtd(devinf) == SML_DEVINF_VERSION_UNKNOWN, "The default VerDTD must be SML_DEVINF_VERSION_UNKNOWN.");

	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_UNKNOWN);
	sml_fail_unless(sml_dev_inf_get_ver_dtd(devinf) == SML_DEVINF_VERSION_UNKNOWN, "The default VerDTD must be still set.");

	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_get_ver_dtd(devinf) == SML_DEVINF_VERSION_10, "The VerDTD was set to SyncML 1.0 DevInf.");

	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_get_ver_dtd(devinf) == SML_DEVINF_VERSION_11, "The VerDTD was set to SyncML 1.1 DevInf.");

	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_get_ver_dtd(devinf) == SML_DEVINF_VERSION_12, "The VerDTD was set to SyncML 1.2 DevInf.");

	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_UNKNOWN);
	sml_fail_unless(sml_dev_inf_get_ver_dtd(devinf) == SML_DEVINF_VERSION_12, "The VerDTD must still be set to SyncML 1.2 DevInf.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_enforce_dev_id)
{
	setup_testbed(NULL);

	const gchar* devid = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_dev_id(devinf) == NULL, "The default DevID must be NULL.");

	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, NULL, &error) == FALSE, "The DevID cannot be NULL.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_get_dev_id(devinf) == NULL, "The default DevID must be still set.");

	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "", &error) == FALSE, "The DevID cannot be the empty string.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_get_dev_id(devinf) == NULL, "The default DevID must be still set.");

	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");

	devid = sml_dev_inf_get_dev_id(devinf);
	sml_fail_unless(devid != NULL, "The device ID must be set now.");
	sml_fail_unless(strcmp(devid, "my device") == 0, "The device ID mus be 'my device'.");

	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, NULL, &error) == FALSE, "The DevID cannot be NULL.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	devid = sml_dev_inf_get_dev_id(devinf);
	sml_fail_unless(devid != NULL, "The device ID must be still set.");
	sml_fail_unless(strcmp(devid, "my device") == 0, "The device ID mus be still 'my device'.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_enforce_dev_typ)
{
	setup_testbed(NULL);

	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_dev_typ(devinf) == SML_DEVINF_DEVTYP_UNKNOWN, "The default DevTyp must be SML_DEVINF_DEVTYP_UNKNOWN.");

	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_UNKNOWN);

	sml_fail_unless(sml_dev_inf_get_dev_typ(devinf) == SML_DEVINF_DEVTYP_UNKNOWN, "The default DevTyp must be still set.");

	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);

	sml_fail_unless(sml_dev_inf_get_dev_typ(devinf) == SML_DEVINF_DEVTYP_SERVER, "The DevTyp was already set.");

	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_UNKNOWN);

	sml_fail_unless(sml_dev_inf_get_dev_typ(devinf) == SML_DEVINF_DEVTYP_SERVER, "The DevTyp must be still set.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_man)
{
	setup_testbed(NULL);

	const gchar* man = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_man(devinf) == NULL, "The default manufacturer must be NULL.");

	sml_fail_unless(sml_dev_inf_set_man(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_man(devinf) == NULL, "The default manufacturer must be still set.");

	sml_fail_unless(sml_dev_inf_set_man(devinf, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_man(devinf) == NULL, "The default manufacturer must be still set.");

	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error), "%s", error?error->message:"No GError set.");
	man = sml_dev_inf_get_man(devinf);
	sml_fail_unless(man != NULL, "The manufacturer must be set now.");
	sml_fail_unless(strcmp(man, "OpenSync") == 0, "The manufacturer must be 'OpenSync'.");

	sml_fail_unless(sml_dev_inf_set_man(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_man(devinf) == NULL, "The manufacturer must be NULL again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_mod)
{
	setup_testbed(NULL);

	const gchar* mod = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_mod(devinf) == NULL, "The default model must be NULL.");

	sml_fail_unless(sml_dev_inf_set_mod(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_mod(devinf) == NULL, "The default model must be still set.");

	sml_fail_unless(sml_dev_inf_set_mod(devinf, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_mod(devinf) == NULL, "The default model must be still set.");

	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error), "%s", error?error->message:"No GError set.");
	mod = sml_dev_inf_get_mod(devinf);
	sml_fail_unless(mod != NULL, "The model must be set now.");
	sml_fail_unless(strcmp(mod, "libsyncml") == 0, "The model must be 'libsyncml'.");

	sml_fail_unless(sml_dev_inf_set_mod(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_mod(devinf) == NULL, "The model must be NULL again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_oem)
{
	setup_testbed(NULL);

	const gchar* oem = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_oem(devinf) == NULL, "The default OEM must be NULL.");

	sml_fail_unless(sml_dev_inf_set_oem(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_oem(devinf) == NULL, "The default OEM must be still set.");

	sml_fail_unless(sml_dev_inf_set_oem(devinf, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_oem(devinf) == NULL, "The default OEM must be still set.");

	sml_fail_unless(sml_dev_inf_set_oem(devinf, "uname -a", &error), "%s", error?error->message:"No GError set.");
	oem = sml_dev_inf_get_oem(devinf);
	sml_fail_unless(oem != NULL, "The OEM must be set now.");
	sml_fail_unless(strcmp(oem, "uname -a") == 0, "The OEM must be 'uname -a'.");

	sml_fail_unless(sml_dev_inf_set_oem(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_oem(devinf) == NULL, "The OEM must be NULL again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_fwv)
{
	setup_testbed(NULL);

	const gchar* fwv = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_fwv(devinf) == NULL, "The default firmware version must be NULL.");

	sml_fail_unless(sml_dev_inf_set_fwv(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_fwv(devinf) == NULL, "The default firmware version must be still set.");

	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_fwv(devinf) == NULL, "The default firmware version must be still set.");

	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error), "%s", error?error->message:"No GError set.");
	fwv = sml_dev_inf_get_fwv(devinf);
	sml_fail_unless(fwv != NULL, "The firmware version must be set now.");
	sml_fail_unless(strcmp(fwv, "2.6.29") == 0, "The firmware version must be '2.6.29'.");

	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "V 1.23\nmore noise", &error), "%s", error?error->message:"No GError set.");
	fwv = sml_dev_inf_get_fwv(devinf);
	sml_fail_unless(fwv != NULL, "The firmware version must be set now.");
	sml_fail_unless(strcmp(fwv, "V 1.23\nmore noise") == 0, "The firmware version must be 'V 1.23\\nmore noise'.");

	sml_fail_unless(sml_dev_inf_set_fwv(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_fwv(devinf) == NULL, "The firmware version must be NULL again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_swv)
{
	setup_testbed(NULL);

	const gchar* swv = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_swv(devinf) == NULL, "The default software version must be NULL.");

	sml_fail_unless(sml_dev_inf_set_swv(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_swv(devinf) == NULL, "The default software version must be still set.");

	sml_fail_unless(sml_dev_inf_set_swv(devinf, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_swv(devinf) == NULL, "The default software version must be still set.");

	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error), "%s", error?error->message:"No GError set.");
	swv = sml_dev_inf_get_swv(devinf);
	sml_fail_unless(swv != NULL, "The software version must be set now.");
	sml_fail_unless(strcmp(swv, "0.6.x") == 0, "The software version must be '0.6.x'.");

	sml_fail_unless(sml_dev_inf_set_swv(devinf, "V 1.23\nmore noise", &error), "%s", error?error->message:"No GError set.");
	swv = sml_dev_inf_get_swv(devinf);
	sml_fail_unless(swv != NULL, "The software version must be set now.");
	sml_fail_unless(strcmp(swv, "V 1.23\nmore noise") == 0, "The software version must be 'V 1.23\\nmore noise'.");

	sml_fail_unless(sml_dev_inf_set_swv(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_swv(devinf) == NULL, "The software version must be NULL again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_hwv)
{
	setup_testbed(NULL);

	const gchar* hwv = NULL;
	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_hwv(devinf) == NULL, "The default hardware version must be NULL.");

	sml_fail_unless(sml_dev_inf_set_hwv(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_hwv(devinf) == NULL, "The default hardware version must be still set.");

	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "", &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_hwv(devinf) == NULL, "The default hardware version must be still set.");

	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "x86 IA64", &error), "%s", error?error->message:"No GError set.");
	hwv = sml_dev_inf_get_hwv(devinf);
	sml_fail_unless(hwv != NULL, "The hardware version must be set now.");
	sml_fail_unless(strcmp(hwv, "x86 IA64") == 0, "The hardware version must be 'x86 IA64'.");

	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "V 1.23\nmore noise", &error), "%s", error?error->message:"No GError set.");
	hwv = sml_dev_inf_get_hwv(devinf);
	sml_fail_unless(hwv != NULL, "The hardware version must be set now.");
	sml_fail_unless(strcmp(hwv, "V 1.23\nmore noise") == 0, "The hardware version must be 'V 1.23\\nmore noise'.");

	sml_fail_unless(sml_dev_inf_set_hwv(devinf, NULL, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_get_hwv(devinf) == NULL, "The hardware version must be NULL again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_support_utc)
{
	setup_testbed(NULL);

	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_support_utc(devinf) == FALSE, "The UTC support must be FALSE by default.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_fail_unless(sml_dev_inf_get_support_utc(devinf) == TRUE, "The UTC support must be set to TRUE.");
	sml_dev_inf_set_support_utc(devinf, FALSE);
	sml_fail_unless(sml_dev_inf_get_support_utc(devinf) == FALSE, "The UTC support must be FALSE again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_support_large_objs)
{
	setup_testbed(NULL);

	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_support_large_objs(devinf) == FALSE, "The large objects support must be FALSE by default.");
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_fail_unless(sml_dev_inf_get_support_large_objs(devinf) == TRUE, "The large objects support must be set to TRUE.");
	sml_dev_inf_set_support_large_objs(devinf, FALSE);
	sml_fail_unless(sml_dev_inf_get_support_large_objs(devinf) == FALSE, "The large objects support must be FALSE again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_option_support_number_of_changes)
{
	setup_testbed(NULL);

	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	sml_fail_unless(sml_dev_inf_get_support_number_of_changes(devinf) == FALSE, "The number of changes support must be FALSE by default.");
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	sml_fail_unless(sml_dev_inf_get_support_number_of_changes(devinf) == TRUE, "The number of changes support must be set to TRUE.");
	sml_dev_inf_set_support_number_of_changes(devinf, FALSE);
	sml_fail_unless(sml_dev_inf_get_support_number_of_changes(devinf) == FALSE, "The number of changes support must be FALSE again.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_enforce_data_stores)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	sml_fail_unless(sml_dev_inf_num_data_stores(devinf) == 0, "There are no default data stores.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) == NULL, "There is no default data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 5) == NULL, "There is no default data store.");

	sml_fail_unless(sml_dev_inf_add_data_store(devinf, NULL, &error) == FALSE, "Only real data stores can be added.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_num_data_stores(devinf) == 0, "There are no default data stores.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) == NULL, "There is no default data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 5) == NULL, "There is no default data store.");

	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	
	sml_fail_unless(sml_dev_inf_num_data_stores(devinf) == 1, "There must be one data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) != NULL, "There is a first data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) == datastore, "The first data store is a wrong one.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 1) == NULL, "There is only one data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 5) == NULL, "There is only one data store.");

	sml_fail_unless(sml_dev_inf_add_data_store(devinf, NULL, &error) == FALSE, "Only real data stores can be added.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_num_data_stores(devinf) == 1, "There must be one data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) != NULL, "There is a first data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) == datastore, "The first data store is a wrong one.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 1) == NULL, "There is only one data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 5) == NULL, "There is only one data store.");

	g_object_unref(datastore);
	datastore = NULL;

	sml_fail_unless(sml_dev_inf_num_data_stores(devinf) == 1, "There must be one data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 0) != NULL, "There is a first data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 1) == NULL, "There is only one data store.");
	sml_fail_unless(sml_dev_inf_get_nth_data_store(devinf, 5) == NULL, "There is only one data store.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_enforce_ctcaps)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	SmlDevInfContentType *ct = sml_dev_inf_content_type_new(NULL, NULL, &error);
	sml_fail_unless(ct != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	SmlDevInfCTCap *ctcap = sml_dev_inf_ctcap_new(ct, &error);
	sml_fail_unless(ctcap != NULL, "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, "%s", error?error->message:"No GError set.");
	g_object_unref(ct);
	ct = NULL;

	sml_fail_unless(sml_dev_inf_num_ctcaps(devinf) == 0, "There are no default CTCaps.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) == NULL, "There is no default CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 5) == NULL, "There is no default CTCap.");

	sml_fail_unless(sml_dev_inf_add_ctcap(devinf, NULL, &error) == FALSE, "Only real CTCaps can be added.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_num_ctcaps(devinf) == 0, "There are no default CTCaps.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) == NULL, "There is no default CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 5) == NULL, "There is no default CTCap.");

	sml_fail_unless(sml_dev_inf_add_ctcap(devinf, ctcap, &error) == TRUE, "%s", error?error->message:"No GError set.");
	
	sml_fail_unless(sml_dev_inf_num_ctcaps(devinf) == 1, "There must be one CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) != NULL, "There is a first CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) == ctcap, "The first CTCap is a wrong one.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 1) == NULL, "There is only one CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 5) == NULL, "There is only one CTCap.");

	sml_fail_unless(sml_dev_inf_add_ctcap(devinf, NULL, &error) == FALSE, "Only real CTCaps can be added.");
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	sml_fail_unless(sml_dev_inf_num_ctcaps(devinf) == 1, "There must be one CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) != NULL, "There is a first CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) == ctcap, "The first CTCap is a wrong one.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 1) == NULL, "There is only one CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 5) == NULL, "There is only one CTCap.");

	g_object_unref(ctcap);
	ctcap = NULL;

	sml_fail_unless(sml_dev_inf_num_ctcaps(devinf) == 1, "There must be one CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 0) != NULL, "There is a first CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 1) == NULL, "There is only one CTCap.");
	sml_fail_unless(sml_dev_inf_get_nth_ctcap(devinf, 5) == NULL, "There is only one CTCap.");

	g_object_unref(devinf);
}
END_TEST

START_TEST (dev_inf_compliance_error)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = sml_dev_inf_new();
	sml_fail_unless(devinf != NULL, NULL);

	/* The actual object is not compliant and there is no error pointer
	 * but the function should return NULL and should not throw an assertion.
	 */
	sml_fail_unless(sml_dev_inf_is_compliant(devinf, NULL) == FALSE, NULL);

	/* This is the usual case.
	 * The object is not compliant and the error pointer is available.
	 */
	sml_fail_unless(sml_dev_inf_is_compliant(devinf, &error) == FALSE, NULL);
	sml_fail_unless(error != NULL, "The error must be set if an error appears.");
	g_error_free(error);
	error = NULL;

	g_object_unref(devinf);
}
END_TEST

#define INIT_DEV_INF_OBJECT \
	devinf = sml_dev_inf_new(); \
	sml_fail_unless(devinf != NULL, NULL);

#define CHECK_COMPLIANCE_ERROR \
	sml_fail_unless(sml_dev_inf_is_compliant(devinf, NULL) == FALSE, "missing expected error (&error == NULL)"); \
	sml_fail_unless(sml_dev_inf_is_compliant(devinf, &error) == FALSE, "missing expected error (&error != NULL)"); \
	sml_fail_unless(error != NULL, "The error must be set if an error appears."); \
	g_error_free(error); \
	error = NULL; \
	g_object_unref(devinf); \
	devinf = NULL;

#define CHECK_COMPLIANCE_SUCCESS \
	sml_fail_unless(sml_dev_inf_is_compliant(devinf, &error) == TRUE, "%s", error?error->message:"No GError set."); \
	sml_fail_unless(sml_dev_inf_is_compliant(devinf, NULL) == TRUE, NULL); \
	g_object_unref(devinf); \
	devinf = NULL;

START_TEST (dev_inf_compliance_1_0)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* missing VerDTD, DevID, DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	CHECK_COMPLIANCE_ERROR

	/* missing DevID, DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	CHECK_COMPLIANCE_ERROR

	/* missing VerDTD, DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing VerDTD, DevID and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	CHECK_COMPLIANCE_ERROR

	/* missing VerDTD, data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing DevID and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	CHECK_COMPLIANCE_ERROR

	/* missing DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing data store */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	CHECK_COMPLIANCE_ERROR

	/* compliant */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_compliance_1_1)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* missing VerDTD, DevID, DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	CHECK_COMPLIANCE_ERROR

	/* missing DevID, DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	CHECK_COMPLIANCE_ERROR

	/* missing VerDTD, DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing VerDTD, DevID and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	CHECK_COMPLIANCE_ERROR

	/* missing VerDTD, data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing DevID and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	CHECK_COMPLIANCE_ERROR

	/* missing DevTyp and data store*/
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing data store */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	CHECK_COMPLIANCE_ERROR

	/* compliant */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_compliance_1_2)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* missing VerDTD */
	INIT_DEV_INF_OBJECT
	CHECK_COMPLIANCE_ERROR

	/* missing Man */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	CHECK_COMPLIANCE_ERROR

	/* missing Mod */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing FwV */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing SwV */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing HwV */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing DevID */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing DevTyp */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing DataStore */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	CHECK_COMPLIANCE_ERROR

	/* compliant */
	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_PHONE);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_overdefined_1_0)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* compliant */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_SUCCESS

	/* set elements which are unknown to SyncML 1.0 */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_SUCCESS
	
	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_filtered_1_0)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* set elements which are unknown to SyncML 1.0 */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);

	/* check the filtering mechanism for unknown elements */

	sml_fail_unless(sml_dev_inf_get_support_utc(devinf) == FALSE, "The UTC support must be filtered.");
	sml_fail_unless(sml_dev_inf_get_support_large_objs(devinf) == FALSE, "The large objects support must be filtered.");
	sml_fail_unless(sml_dev_inf_get_support_number_of_changes(devinf) == FALSE, "The number of changes support must be filtered.");

	/* compliant */

	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_server_1_0)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* compliant - without UTC, SupportLargeObjs and SupportNumberOfChanges */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_10);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_server_1_1)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* missing UTC, SupportLargeObjs and SupportNumberOfChanges */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing UTC */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_ERROR

	/* missing SupportLargeObjs */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_ERROR

	/* missing SupportNumberOfChanges */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	CHECK_COMPLIANCE_ERROR

	/* compliant */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_11);
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

START_TEST (dev_inf_server_1_2)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlDevInf *devinf = NULL;

	SmlDevInfDataStore *datastore = sml_dev_inf_data_store_new("Contacts", &error);
	sml_fail_unless(datastore != NULL, NULL);

	/* missing UTC, SupportLargeObjs and SupportNumberOfChanges */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	CHECK_COMPLIANCE_ERROR

	/* missing UTC */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_ERROR

	/* missing SupportLargeObjs */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_ERROR

	/* missing SupportNumberOfChanges */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	CHECK_COMPLIANCE_ERROR

	/* compliant */

	INIT_DEV_INF_OBJECT
	sml_dev_inf_set_ver_dtd(devinf, SML_DEVINF_VERSION_12);
	sml_fail_unless(sml_dev_inf_set_man(devinf, "OpenSync", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_mod(devinf, "libsyncml", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_fwv(devinf, "2.6.29", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_swv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_hwv(devinf, "0.6.x", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_fail_unless(sml_dev_inf_set_dev_id(devinf, "my device", &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_dev_typ(devinf, SML_DEVINF_DEVTYP_SERVER);
	sml_fail_unless(sml_dev_inf_add_data_store(devinf, datastore, &error) == TRUE, "%s", error?error->message:"No GError set.");
	sml_dev_inf_set_support_utc(devinf, TRUE);
	sml_dev_inf_set_support_large_objs(devinf, TRUE);
	sml_dev_inf_set_support_number_of_changes(devinf, TRUE);
	CHECK_COMPLIANCE_SUCCESS

	g_object_unref(datastore);
}
END_TEST

@SML_TESTCASE_CODE@

