/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "tests/support.h"

#include <libsyncml/data_sync_api/sml_map_item.h>

START_TEST (map_item_new)
{
	setup_testbed(NULL);
	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_local)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_local(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_local_not_null)
{
	setup_testbed(NULL);

	GError *error = NULL;

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_local(item, NULL, &error), "The local URI must be enforced.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_local_no_name)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");
	sml_location_set_name(location, "my 1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_local(item, location, &error), "The location must not have a name.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_local_no_parent)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");
	sml_location_set_parent_uri(location, "56");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_local(item, location, &error), "The location must not have a parent URI.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_local_missing_uri)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_local(item, location, &error), "The location must have an URI.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_get_local)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_get_local_uri(item) == NULL, "The local URI is not set until now.");

	sml_fail_unless(sml_map_item_set_local(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(sml_map_item_get_local_uri(item) != NULL, "The local URI must be set now.");
	sml_fail_unless(strcmp(sml_map_item_get_local_uri(item), "1234") == 0, "The local URI must be 1234.");

	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_remote)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_remote(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_remote_not_null)
{
	setup_testbed(NULL);

	GError *error = NULL;

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_remote(item, NULL, &error), "The remote URI must be enforced.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_remote_no_name)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");
	sml_location_set_name(location, "my 1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_remote(item, location, &error), "The location must not have a name.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_remote_no_parent)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");
	sml_location_set_parent_uri(location, "56");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_remote(item, location, &error), "The location must not have a parent URI.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_set_remote_missing_uri)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(!sml_map_item_set_remote(item, location, &error), "The location must have an URI.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_get_remote)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_get_remote_uri(item) == NULL, "The remote URI is not set until now.");

	sml_fail_unless(sml_map_item_set_remote(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(sml_map_item_get_remote_uri(item) != NULL, "The remote URI must be set now.");
	sml_fail_unless(strcmp(sml_map_item_get_remote_uri(item), "1234") == 0, "The remote URI must be 1234.");

	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_compliance)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_remote(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);
	sml_fail_unless(sml_map_item_set_local(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(sml_map_item_is_compliant(item, NULL), "The item is compliant.");
	sml_fail_unless(sml_map_item_is_compliant(item, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_compliance_missing_local)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_remote(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(!sml_map_item_is_compliant(item, NULL), "The item is not compliant.");
	sml_fail_unless(!sml_map_item_is_compliant(item, &error), "The item is not compliant.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_compliance_missing_remote)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_local(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(!sml_map_item_is_compliant(item, NULL), "The item is not compliant.");
	sml_fail_unless(!sml_map_item_is_compliant(item, &error), "The item is not compliant.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_compliance_wrong_local)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *local = sml_location_new();
	sml_fail_unless(local != NULL, NULL);
	sml_location_set_uri(local, "1234");
	SmlLocation *remote = sml_location_new();
	sml_fail_unless(remote != NULL, NULL);
	sml_location_set_uri(remote, "5678");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_remote(item, remote, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);
	sml_fail_unless(sml_map_item_set_local(item, local, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(sml_map_item_is_compliant(item, NULL), "The item is compliant.");
	sml_fail_unless(sml_map_item_is_compliant(item, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	/* add name */
	sml_location_set_name(local, "my 1234");

	sml_fail_unless(!sml_map_item_is_compliant(item, NULL), "The item is not compliant.");
	sml_fail_unless(!sml_map_item_is_compliant(item, &error), "The item is not compliant.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(local);
	g_object_unref(remote);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_compliance_wrong_remote)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *local = sml_location_new();
	sml_fail_unless(local != NULL, NULL);
	sml_location_set_uri(local, "1234");
	SmlLocation *remote = sml_location_new();
	sml_fail_unless(remote != NULL, NULL);
	sml_location_set_uri(remote, "5678");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_remote(item, remote, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);
	sml_fail_unless(sml_map_item_set_local(item, local, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(sml_map_item_is_compliant(item, NULL), "The item is compliant.");
	sml_fail_unless(sml_map_item_is_compliant(item, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	/* add parent */
	sml_location_set_parent_uri(remote, "56");

	sml_fail_unless(!sml_map_item_is_compliant(item, NULL), "The item is not compliant.");
	sml_fail_unless(!sml_map_item_is_compliant(item, &error), "The item is not compliant.");
	sml_fail_unless(error != NULL, NULL);

	g_error_free(error);
	g_object_unref(local);
	g_object_unref(remote);
	g_object_unref(item);
}
END_TEST

START_TEST (map_item_references)
{
	setup_testbed(NULL);

	GError *error = NULL;
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	sml_location_set_uri(location, "1234");

	SmlMapItem *item = sml_map_item_new();
	sml_fail_unless(item != NULL, NULL);

	sml_fail_unless(sml_map_item_set_local(item, location, &error), "%s", error?error->message:"No GError set.");
	sml_fail_unless(error == NULL, NULL);

	g_object_ref(item);

	sml_fail_unless(sml_map_item_get_local_uri(item) != NULL, "The local URI was not set.");

	g_object_unref(item);

	sml_fail_unless(sml_map_item_get_local_uri(item) != NULL, "The local URI is already cleaned up.");

	g_object_unref(location);
	g_object_unref(item);
}
END_TEST

@SML_TESTCASE_CODE@

