/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "tests/support.h"

#include <libsyncml/data_sync_api/sml_location.h>
#include <string.h>

START_TEST (location_new)
{
	setup_testbed(NULL);
	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);
	g_object_unref(location);
}
END_TEST

START_TEST (location_uri)
{
	setup_testbed(NULL);

	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);

	sml_fail_unless(sml_location_get_uri(location) == NULL, "The default URI must be NULL.");

	sml_location_set_uri(location, NULL);
	sml_fail_unless(sml_location_get_uri(location) == NULL, "The URI must still be NULL.");
	
	sml_location_set_uri(location, "http://localhost");
	const gchar *uri = sml_location_get_uri(location);
	sml_fail_unless(uri != NULL, "The URI is now set.");
	sml_fail_unless(strcmp(uri, "http://localhost") == 0, "The URI must be set to http://localhost.");

	sml_location_set_uri(location, NULL);
	sml_fail_unless(sml_location_get_uri(location) == NULL, "The URI must be NULL again.");

	g_object_unref(location);
}
END_TEST

START_TEST (location_name)
{
	setup_testbed(NULL);

	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);

	sml_fail_unless(sml_location_get_name(location) == NULL, "The default name must be NULL.");

	sml_location_set_name(location, NULL);
	sml_fail_unless(sml_location_get_name(location) == NULL, "The name must still be NULL.");
	
	sml_location_set_name(location, "John Doe");
	const gchar *name = sml_location_get_name(location);
	sml_fail_unless(name != NULL, "The name is now set.");
	sml_fail_unless(strcmp(name, "John Doe") == 0, "The name must be set to 'John Doe'.");

	sml_location_set_name(location, NULL);
	sml_fail_unless(sml_location_get_name(location) == NULL, "The name must be NULL again.");

	g_object_unref(location);
}
END_TEST

START_TEST (location_parent_uri)
{
	setup_testbed(NULL);

	SmlLocation *location = sml_location_new();
	sml_fail_unless(location != NULL, NULL);

	sml_fail_unless(sml_location_get_parent_uri(location) == NULL, "The default parent URI must be NULL.");

	sml_location_set_parent_uri(location, NULL);
	sml_fail_unless(sml_location_get_parent_uri(location) == NULL, "The parent URI must still be NULL.");
	
	sml_location_set_parent_uri(location, "http://localhost");
	const gchar *parent_uri = sml_location_get_parent_uri(location);
	sml_fail_unless(parent_uri != NULL, "The parent URI is now set.");
	sml_fail_unless(strcmp(parent_uri, "http://localhost") == 0, "The parent URI must be set to http://localhost.");

	sml_location_set_parent_uri(location, NULL);
	sml_fail_unless(sml_location_get_parent_uri(location) == NULL, "The parent URI must be NULL again.");

	g_object_unref(location);
}
END_TEST

START_TEST (location_compare_equal_relative_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_uri(objloc, "./test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_uri(objloc2, "./test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_different_relative_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_uri(objloc, "./test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_uri(objloc2, "./test2");

	sml_fail_unless(!sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_absolute_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_uri(objloc, "/test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_uri(objloc2, "/test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_different_absolute_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_uri(objloc, "/test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_uri(objloc2, "/test2");

	sml_fail_unless(!sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_relative_parent_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "./test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "./test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_different_relative_parent_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "./test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "./test2");

	sml_fail_unless(!sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_absolute_parent_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_different_absolute_parent_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test2");

	sml_fail_unless(!sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_full_uri_relative)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test");
	sml_location_set_uri(objloc, "./test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test");
	sml_location_set_uri(objloc2, "./test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_full_uri_normal)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test");
	sml_location_set_uri(objloc, "test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test");
	sml_location_set_uri(objloc2, "./test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_full_uri_complex)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test/asd");
	sml_location_set_uri(objloc, "test/");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test/.//asd//");
	sml_location_set_uri(objloc2, "././test");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_different_full_uri_wrong_parent)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test/asd2");
	sml_location_set_uri(objloc, "test");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test/.//asd//");
	sml_location_set_uri(objloc2, "././test");

	sml_fail_unless(!sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_different_full_uri_wrong_uri)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/test/asd");
	sml_location_set_uri(objloc, "tes3t");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, "/test/.//asd//");
	sml_location_set_uri(objloc2, "././test");

	sml_fail_unless(!sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_compare_equal_full_uri_root)
{
	setup_testbed(NULL);
	
	SmlLocation *objloc = sml_location_new();
	sml_fail_unless(objloc != NULL, NULL);
	sml_location_set_parent_uri(objloc, "/");
	sml_location_set_uri(objloc, "");

	SmlLocation *objloc2 = sml_location_new();
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_location_set_parent_uri(objloc2, ".//");
	sml_location_set_uri(objloc2, "///");

	sml_fail_unless(sml_location_is_equal(objloc, objloc2), NULL);

	g_object_unref(objloc);
	g_object_unref(objloc2);
}
END_TEST

START_TEST (location_references)
{
	setup_testbed(NULL);

	SmlLocation *loc = sml_location_new();
	sml_fail_unless(loc != NULL, NULL);

	sml_location_set_uri(loc, "test");

	g_object_ref(loc);

	sml_fail_unless(sml_location_get_uri(loc) != NULL, "The location URI was not set.");

	g_object_unref(loc);

	sml_fail_unless(sml_location_get_uri(loc) != NULL, "The location URI is already cleaned up.");

	g_object_unref(loc);
}
END_TEST

@SML_TESTCASE_CODE@

