/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef _SML_TRANSPORT_INTERNALS_H_
#define _SML_TRANSPORT_INTERNALS_H_

#include "sml_transport.h"
#include "sml_queue_internals.h"
#include "sml_thread.h"

typedef gboolean (* SmlTransportSetConfigOptionFn)   (SmlTransport *tsp, const gchar *name, const gchar *value, GError **error);
typedef gboolean (* SmlTransportSetConnectionTypeFn) (SmlTransport *tsp, SmlTransportConnectionType type, GError **error);
typedef gboolean (* SmlTransportInitializeFn)        (SmlTransport *tsp, GError **error);

/* This function is used for transport clients
 * to support redirects from the server.
 * Example: Funambol works with such redirects
 *          because of Tomcat's session management.
 */
typedef gboolean (* SmlTransportSetResponseUriFn) (SmlTransport *tsp, const gchar *uri, GError **error);

/* This function is used to add some kind of session id
 * to the URL of the server. This is necessary to
 * avoid session hijacking.
 */
typedef gchar * (* SmlTransportGetResponseUriFn) (SmlLink *link, SmlSession *session, GError **error);

typedef gboolean (* SmlTransportFinalizeFn) (void *data, GError **error);

typedef void (* SmlTransportConnectFn)    (void *data);
typedef void (* SmlTransportDisconnectFn) (void *data, void *link_data);
typedef void (* SmlTransportSendFn)       (void *userdata, void *link_data, SmlTransportData *data, GError *error);

typedef struct SmlTransportFunctions {
	SmlTransportSetConfigOptionFn set_config_option;
	SmlTransportSetConnectionTypeFn set_connection_type;
	SmlTransportInitializeFn initialize;
	SmlTransportSetResponseUriFn set_response_uri;
	SmlTransportGetResponseUriFn get_response_uri;
	SmlTransportFinalizeFn finalize;
	SmlTransportConnectFn connect;
	SmlTransportDisconnectFn disconnect;
	SmlTransportSendFn send;
} SmlTransportFunctions;

typedef enum SmlTransportState {
	SML_TRANSPORT_UNINITIALIZED = 0,
	SML_TRANSPORT_INITIALIZED   = 1,
	SML_TRANSPORT_CONNECTED     = 2,
	SML_TRANSPORT_DATA_ERROR    = 3,
	SML_TRANSPORT_DISCONNECTED  = 4,
	SML_TRANSPORT_ERROR         = 5,
	SML_TRANSPORT_FINALIZED     = 6
} SmlTransportState;

struct SmlTransport {
	GMainContext *context;
	SmlThread *thread;
	
	SmlTransportState state;
	
	SmlTransportType type;
	SmlTransportFunctions functions;
	void *transport_data;
	
	SmlQueue *command_queue;
	SmlTransportEventCb event_callback;
	void *event_callback_userdata;
	gint event_callback_ref_count;
	GError *cached_error;
	
	gboolean connected;
	GHashTable *links;
	GMutex *links_mutex;
	gsize connections;
	GMutex *connections_mutex;

	gboolean async;
};

struct SmlLink {
	SmlTransport *tsp;
	void *link_data;
	gint32 refCount;
};

typedef enum SmlTransportCommandType {
	SML_TRANSPORT_CMD_UNKNOWN    = 0,
	SML_TRANSPORT_CMD_SEND       = 1,
	SML_TRANSPORT_CMD_CONNECT    = 2,
	SML_TRANSPORT_CMD_DISCONNECT = 3
} SmlTransportCommandType;

typedef struct SmlTransportCommand {
	SmlTransportCommandType type;
	SmlTransportData *data;
	const void *config;
	SmlLink *link;
	GError *error;
} SmlTransportCommand;

struct SmlTransportData {
	gchar *data;
	gsize size;
	SmlMimeType type;
	gboolean ownsData;
	gint32 refCount;
	/** This field specifies if the data transported
	 * here needs an reply from the other side. If it does not
	 * need one, the transports might take special precautions */
	gboolean needsAnswer;
	/** This is only needed to switch the mime type when using the OBEX Transporting
	 * and sending a SyncML 1.2 SAN package. The transport send() function for OBEX transport
	 * make use of a PUT/GET command sequence. To avoid that GET command request with mimetype
	 * SAN, we have to store the regular mimetype to continue the sync in this struct.
	 * So the SyncML 1.2 notification looks like this:
	 * PUT SML_MIMETYPE_SAN
	 * -- wait for response --
	 * GET SML_MIMETYPE_XML / SML_MIMETYPE_WBXML (not SML_MIMETYPE_SAN!)
	 * -- wait for resposne --
	 */
	SmlMimeType type_get;
};

void     smlTransportWorkerHandler (void *message, void *userdata);
gboolean smlTransportReceiveEvent  (SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, const GError *error);

#endif //_SML_TRANSPORT_INTERNALS_H_
