/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef _SML_TRANSPORT_H_
#define _SML_TRANSPORT_H_

#include <glib.h>

#include "data_sync_api/sml_data_sync.h"
#include "sml_defines.h"

typedef struct SmlTransport     SmlTransport;
typedef struct SmlTransportData SmlTransportData;
typedef struct SmlLink          SmlLink;

#include "sml_session.h"

typedef enum SmlTransportEventType {
	SML_TRANSPORT_EVENT_UNKNOWN         = 0,
	SML_TRANSPORT_EVENT_CONNECT_DONE    = 1,
	SML_TRANSPORT_EVENT_DISCONNECT_DONE = 2,
	SML_TRANSPORT_EVENT_ERROR           = 3,
	SML_TRANSPORT_EVENT_DATA            = 4
} SmlTransportEventType;

typedef enum {
	SML_TRANSPORT_CONNECTION_TYPE_UNKNOWN   = 0,
	SML_TRANSPORT_CONNECTION_TYPE_SERIAL    = 1,
	SML_TRANSPORT_CONNECTION_TYPE_BLUETOOTH = 2,
	SML_TRANSPORT_CONNECTION_TYPE_IRDA      = 3,
	SML_TRANSPORT_CONNECTION_TYPE_NET       = 4,
	SML_TRANSPORT_CONNECTION_TYPE_USB       = 5
} SmlTransportConnectionType;

typedef gboolean (* SmlTransportEventCb) (SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, const GError *error, void *userdata);

SmlTransport* smlTransportNew  (SmlTransportType type, GError **error);
void          smlTransportFree (SmlTransport *tsp);

gboolean smlTransportSetConfigOption   (SmlTransport *tsp, const gchar *name, const gchar *value, GError **error);
gboolean smlTransportSetConnectionType (SmlTransport *tsp, SmlTransportConnectionType type, GError **error);
gboolean smlTransportInitialize        (SmlTransport *tsp, GError **error);

void              smlTransportSetEventCallback (SmlTransport *tsp, SmlTransportEventCb callback, void *userdata);
gboolean          smlTransportFinalize         (SmlTransport *tsp, GError **error);
gboolean          smlTransportConnect          (SmlTransport *tsp, GError **error);
gboolean          smlTransportDisconnect       (SmlTransport *tsp, SmlLink *link, GError **error);
gboolean          smlTransportIsConnected      (SmlTransport *tsp);
gboolean          smlTransportSend             (SmlTransport *tsp, SmlLink *link, SmlTransportData *data, GError **error);
void              smlTransportSendError        (SmlTransport *tsp, SmlLink *link, const GError *error);
SmlTransportType  smlTransportGetType          (SmlTransport *tsp);
gboolean          smlTransportSetResponseURI   (SmlTransport *tsp, const gchar *uri, GError **error);


SmlTransportData* smlTransportDataNew              (gchar *data, gsize size, SmlMimeType mimetype, gboolean ownsData, GError **error);
SmlTransportData* smlTransportDataRef              (SmlTransportData *data);
void              smlTransportDataDeref            (SmlTransportData *data);
const gchar*      smlTransportDataGetData          (SmlTransportData *data);
gsize             smlTransportDataGetSize          (SmlTransportData *data);
SmlMimeType       smlTransportDataGetType          (SmlTransportData *data);
void              smlTransportDataSetRequestedType (SmlTransportData *data, SmlMimeType mimetype);
void              smlTransportDataSetNeedsAnswer   (SmlTransportData *data, gboolean needsAnswer);

SmlLink*          smlLinkNew                   (SmlTransport *tsp, void *link_data, GError **error);
SmlLink*          smlLinkRef                   (SmlLink *link);
void              smlLinkDeref                 (SmlLink *link);
gchar*            smlLinkGetResponseURI        (SmlLink *link, SmlSession *session, GError **error);
gboolean          smlLinkIsConnected           (SmlLink *link);

#endif //_SML_TRANSPORT_H_
