/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef _SML_PARSE_H_
#define _SML_PARSE_H_

typedef struct SmlParser    SmlParser;
typedef struct SmlAssembler SmlAssembler;

#include "sml_manager.h"

typedef enum {
	SML_PARSER_RESULT_ERROR = 0,
	SML_PARSER_RESULT_OPEN = 1,
	SML_PARSER_RESULT_CLOSE = 2,
	SML_PARSER_RESULT_NORMAL = 3,
	SML_PARSER_RESULT_STATUS = 4,
	SML_PARSER_RESULT_OTHER = 5
} SmlParserResult;

typedef void            (* SmlParserFreeFunction)    (void *userdata);
typedef gboolean        (* SmlParserStartFunction)   (void *userdata, const gchar *data, gsize size, GError **error);
typedef gboolean        (* SmlParserEndFunction)     (void *userdata, gboolean *final, gboolean *end, GError **error);
typedef gboolean        (* SmlParserHeaderFunction)  (void *userdata, SmlHeader **header, SmlCred **cred, GError **error);
typedef gboolean        (* SmlParserStatusFunction)  (void *userdata, SmlStatus **status, GError **error);
typedef SmlParserResult (* SmlParserCommandFunction) (void *userdata, SmlCommand **cmd, GError **error);

typedef struct SmlParserFunctions {
	SmlParserFreeFunction free;
	SmlParserStartFunction start;
	SmlParserEndFunction end;
	SmlParserHeaderFunction get_header;
	SmlParserStatusFunction get_status;
	SmlParserCommandFunction get_cmd;
} SmlParserFunctions;

typedef void     (* SmlAssemblerFreeFunction)            (void *userdata);
typedef gboolean (* SmlAssemblerStartFunction)           (void *userdata, SmlSession *session, GError **error);
typedef gboolean (* SmlAssemblerEndFunction)             (void *userdata, GError **error);
typedef gboolean (* SmlAssemblerRunFunction)             (void *userdata, gchar **data, gsize *size, gboolean *end, gboolean final, gsize maxsize, GError **error);
typedef gsize    (* SmlAssemblerFlushFunction)           (void *userdata);
typedef gboolean (* SmlAssemblerStatusFunction)          (void *userdata, SmlStatus *status, GError **error);
typedef gboolean (* SmlAssemblerRemStatusFunction)       (void *userdata, GError **error);
typedef gboolean (* SmlAssemblerReserveStatusFunction)   (void *userdata, guint cmdRef, guint msgRef, guint cmdID, GError **error);
typedef gboolean (* SmlAssemblerStatusMissingFunction)   (void *userdata);
typedef gboolean (* SmlAssemblerStartCommandFunction)    (void *userdata, guint parentID, SmlCommand *cmd, GError **error);
typedef gboolean (* SmlAssemblerEndCommandFunction)      (void *userdata, guint parentID, GError **error);
typedef gboolean (* SmlAssemblerRemCommandFunction)      (void *userdata, guint parentID, GError **error);
typedef gboolean (* SmlAssemblerHeaderFunction)          (void *userdata, SmlSession *session, GError **error);
typedef gsize    (* SmlAssemblerCheckFunction)           (void *userdata, gboolean headeronly, GError **error);
typedef gboolean (* SmlAssemblerNextCmdRefFunction)      (void *userdata, gsize *cmdRef, gsize *msgRef);
typedef void     (* SmlAssemblerRestoreCommandsFunction) (void *userdata);

typedef struct SmlAssemblerFunctions {
	SmlAssemblerFreeFunction free;
	SmlAssemblerStartFunction start;
	SmlAssemblerEndFunction end;
	SmlAssemblerRunFunction run;
	SmlAssemblerFlushFunction flush;
	SmlAssemblerStatusFunction add_status;
	SmlAssemblerRemStatusFunction rem_status;
	SmlAssemblerReserveStatusFunction reserve_status;
	SmlAssemblerStatusMissingFunction missing_status;
	SmlAssemblerStartCommandFunction start_cmd;
	SmlAssemblerEndCommandFunction end_cmd;
	SmlAssemblerRemCommandFunction rem_cmd;
	SmlAssemblerHeaderFunction add_header;
	SmlAssemblerCheckFunction check_size;
	SmlAssemblerNextCmdRefFunction next_cmdref;
	SmlAssemblerRestoreCommandsFunction restore_cmds;
} SmlAssemblerFunctions;

typedef enum {
	SML_ASSEMBLER_RESULT_ERROR = 0,
	SML_ASSEMBLER_RESULT_OK = 1,
	SML_ASSEMBLER_RESULT_MISMATCH =2
} SmlAssemblerResult;

SmlParser*      smlParserNew        (SmlMimeType type, gsize limit, GError **error);
void            smlParserFree       (SmlParser *parser);
gboolean        smlParserStart      (SmlParser *parser, const gchar *data, gsize size, GError **error);
gboolean        smlParserGetHeader  (SmlParser *parser, SmlHeader **header, SmlCred **cred, GError **error);
SmlParserResult smlParserGetCommand (SmlParser *parser, SmlCommand **cmd, GError **error);
gboolean        smlParserGetStatus  (SmlParser *parser, SmlStatus **status, GError **error);
gboolean        smlParserEnd        (SmlParser *parser, gboolean *final, gboolean *end, GError **error);

SmlAssembler*      smlAssemblerNew             (SmlMimeType type, gsize limit, GError **error);
void               smlAssemblerFree            (SmlAssembler *assm);
gboolean           smlAssemblerStart           (SmlAssembler *assm, SmlSession *session, GError **error);
gsize              smlAssemblerFlush           (SmlAssembler *assm);
void               smlAssemblerRestoreCommands (SmlAssembler *assm);
gboolean           smlAssemblerRun             (SmlAssembler *assm, gchar **data, gsize *size, gboolean *end, gboolean final, GError **error);
SmlAssemblerResult smlAssemblerAddStatus       (SmlAssembler *assm, SmlStatus *status, GError **error);
SmlAssemblerResult smlAssemblerAddStatusFull   (SmlAssembler *assm, SmlStatus *status, gboolean force, GError **error);
SmlAssemblerResult smlAssemblerReserveStatus   (SmlAssembler *assm, guint cmdRef, guint msgRef, guint cmdID, GError **error);
gboolean           smlAssemblerAddHeader       (SmlAssembler *assm, SmlSession *session, GError **error);
SmlAssemblerResult smlAssemblerStartCommand    (SmlAssembler *assm, SmlCommand *parent, SmlCommand *cmd, GError **error);
gboolean           smlAssemblerEndCommand      (SmlAssembler *assm, SmlCommand *parent, GError **error);
gsize              smlAssemblerCheckSize       (SmlAssembler *assm, gboolean headeronly, GError **error);
void               smlAssemblerSetOption       (SmlAssembler *assm, const gchar *optionname, const gchar *value);
const gchar*       smlAssemblerGetOption       (SmlAssembler *assm, const gchar *optionname);

gsize    smlAssemblerSetRemoteMaxMsgSize (SmlAssembler *assm, gsize limit);
gsize    smlAssemblerGetRemoteMaxMsgSize (SmlAssembler *assm);
gboolean smlAssemblerGetSpace            (SmlAssembler *assm, gssize *space, SmlCommand *parent, SmlCommand *cmd, GError **error);

gsize smlAssemblerGetRemoteMaxObjSize (SmlAssembler *assm);
gsize smlAssemblerSetRemoteMaxObjSize (SmlAssembler *assm, gsize limit);

gboolean smlAssemblerIsEmpty         (SmlAssembler *assm);
gboolean smlAssemblerIsStatusMissing (SmlAssembler *assm);
gboolean smlAssemblerGetNextCmdRef   (SmlAssembler *assm, gsize *cmdRef, gsize *msgRef);

/* If you want to parse OMA DS 1.2 SANs then the manager is required. */
void smlParserSetManager (SmlParser *parser, SmlManager *manager);

#endif //_SML_PARSE_H_
