/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

/**
 * @defgroup SmlManager SyncML Manager
 * @ingroup PublicLowLevelAPI
 * @brief The public part
 * 
 */
/*@{*/

#ifndef _SML_MANAGER_H
#define _SML_MANAGER_H

typedef struct SmlManager SmlManager;
typedef struct SmlObject  SmlObject;

typedef enum {
	SML_MANAGER_CONNECT_DONE,
	SML_MANAGER_DISCONNECT_DONE,
	SML_MANAGER_TRANSPORT_ERROR,
	SML_MANAGER_SESSION_NEW,
	SML_MANAGER_SESSION_FINAL,
	SML_MANAGER_SESSION_END,
	SML_MANAGER_SESSION_FLUSH,
	SML_MANAGER_SESSION_WARNING,
	SML_MANAGER_SESSION_ERROR,
	SML_MANAGER_SESSION_ESTABLISHED
} SmlManagerEventType;

/* Always include header files after typedefs. */

#include "sml_session.h"

/* Always include callbacks after the inclusion of the header files. */

typedef void (* SmlManagerEventCb) (SmlManager *manager, SmlManagerEventType type, SmlSession *session, const GError *error, void *userdata);
typedef void (* SmlCommandCb)      (SmlSession *session, SmlCommand *cmd, void *userdata);
typedef void (* SmlHeaderCb)       (SmlSession *session, SmlHeader *header, SmlCred *cred, void *userdata);

SmlManager* smlManagerNew     (SmlTransport *tsp, GError **error);
void        smlManagerFree    (SmlManager *manager);
gboolean    smlManagerReceive (SmlManager *manager, GError **error);

void        smlManagerSetEventCallback (SmlManager *manager, SmlManagerEventCb callback, void *userdata);
SmlSession* smlManagerSessionFind      (SmlManager *manager, gsize sessionID);
gboolean    smlManagerSessionAdd       (SmlManager *manager, SmlSession *session, SmlLink *link, GError **error);
void        smlManagerSessionRemove    (SmlManager *manager, SmlSession *session);

gboolean smlManagerObjectRegister    (SmlManager *manager, SmlCommandType type, SmlSession *session, SmlLocation *location, SmlLocation *source, const gchar *contentType, SmlCommandCb callback, SmlCommandCb childCallback, void *userdata, GError **error);
void smlManagerRegisterHeaderHandler (SmlManager *manager, SmlHeaderCb callback, SmlStatusReplyCb statuscb, void *userdata);

void       smlManagerObjectFree           (SmlObject *object);
SmlObject* smlManagerObjectFind           (SmlManager *manager, SmlSession *session, SmlCommand *cmd);
gboolean   smlManagerDispatchChildCommand (SmlManager *manager, SmlSession *session, SmlCommand *parent, SmlCommand *cmd, GError **error);
gboolean   smlManagerDispatchHeader       (SmlManager *manager, SmlSession *session, SmlHeader *header, SmlCred *cred, GError **error);
gboolean   smlManagerDispatchCommand      (SmlManager *manager, SmlSession *session, SmlCommand *cmd, GError **error);

gboolean smlManagerStart (SmlManager *manager, GError **error);
void     smlManagerStop  (SmlManager *manager);
void     smlManagerRun   (SmlManager *manager);
void     smlManagerQuit  (SmlManager *manager);

void     smlManagerDispatch (SmlManager *manager);
gboolean smlManagerCheck    (SmlManager *manager);

void smlManagerSetLocalMaxMsgSize    (SmlManager *manager, gsize size);
void smlManagerSetLocalMaxObjSize    (SmlManager *manager, gsize size);
void smlManagerSetLocalMaxMsgChanges (SmlManager *manager, gsize size);

gsize smlManagerGetNewSessionID (SmlManager *manager);

SmlTransport*     smlManagerGetTransport       (SmlManager *manager);

SmlLink* smlManagerSessionGetLink (SmlManager *manager, SmlSession *session, GError **error);

void smlManagerSessionFinalLockRef   (SmlManager *manager, SmlSession *session);
void smlManagerSessionFinalLockUnref (SmlManager *manager, SmlSession *session);

#endif //_SML_MANAGER_H

/*@}*/
