/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

/**
 * @defgroup SmlElements SyncML Elements
 * @ingroup PublicLowLevelAPI
 * @brief The public part
 * 
 */
/*@{*/

#ifndef _SML_ELEMENTS_H_
#define _SML_ELEMENTS_H_

#include <glib.h>
#include "sml_defines.h"
#include <libsyncml/data_sync_api/sml_location.h>

typedef struct SmlCred    SmlCred;
typedef struct SmlAnchor  SmlAnchor;
typedef struct SmlItem    SmlItem;
typedef struct SmlHeader  SmlHeader;
typedef struct SmlChal    SmlChal;

SmlCred*     smlCredNewFromString (const gchar *type, const gchar *format, const gchar *data, GError **error);
SmlCred*     smlCredNewAuth       (SmlAuthType type, const gchar *username, const gchar *password, GError **error);
SmlCred*     smlCredNew           (SmlAuthType type, SmlFormatType format, const gchar *data, const gchar *username, GError **error);
void         smlCredRef           (SmlCred *cred);
void         smlCredUnref         (SmlCred *cred);
const gchar* smlCredGetUsername   (SmlCred *cred);
const gchar* smlCredGetPassword   (SmlCred *cred);
const gchar* smlCredGetData       (SmlCred *cred);
void         smlCredSetData       (SmlCred *cred, gchar *data);
SmlAuthType  smlCredGetType       (SmlCred *cred);

SmlAnchor*  smlAnchorNew     (const gchar *last, const gchar *next, GError **error);
const char* smlAnchorGetLast (SmlAnchor *anchor);
const char* smlAnchorGetNext (SmlAnchor *anchor);
void        smlAnchorFree    (SmlAnchor *anchor);

SmlItem*    smlItemNew        (gsize size, GError **error);
SmlItem*    smlItemNewForData (const gchar *data, gsize size, GError **error);
SmlItem*    smlItemRef        (SmlItem *item);
void        smlItemUnref      (SmlItem *item);
gboolean    smlItemAddData    (SmlItem *item, const gchar *data, gsize size, GError **error);
gboolean    smlItemCheck      (SmlItem *item);
gboolean    smlItemHasData    (SmlItem *item);
gboolean    smlItemGetData    (SmlItem *item, gchar **data, gsize *size, GError **error);
gsize       smlItemGetSize    (SmlItem *item);
const char* smlItemGetContent (SmlItem *item);
gboolean    smlItemStealData  (SmlItem *item, gchar **data, gsize *size, GError **error);

void         smlItemSetMoreData    (SmlItem *item, gboolean enable);
gboolean     smlItemGetMoreData    (SmlItem *item);
gboolean     smlItemSetContentType (SmlItem *item, const gchar *ct, GError **error);
const gchar* smlItemGetContentType (SmlItem *item);

void         smlItemSetSource (SmlItem *item, SmlLocation *source);
SmlLocation* smlItemGetSource (SmlItem *item);
void         smlItemSetTarget (SmlItem *item, SmlLocation *target);
SmlLocation* smlItemGetTarget (SmlItem *item);
void         smlItemSetRaw    (SmlItem *item, gboolean raw);

SmlItem *    smlItemGetFragment (SmlItem *orig_item, gsize start, gsize space, GError **error);

gsize              smlHeaderGetSessionID       (SmlHeader *header);
gsize              smlHeaderGetMessageID       (SmlHeader *header);
SmlProtocolVersion smlHeaderGetProtocolVersion (SmlHeader *header);
SmlProtocolType    smlHeaderGetProtocolType    (SmlHeader *header);
SmlLocation*       smlHeaderGetSource          (SmlHeader *header);
SmlLocation*       smlHeaderGetTarget          (SmlHeader *header);
gsize              smlHeaderGetMaxMsgSize      (SmlHeader *header);
const gchar*       smlHeaderGetResponseURI     (SmlHeader *header);
void               smlHeaderFree               (SmlHeader *header);

SmlChal*     smlChalNew           (SmlAuthType type, GError **error);
SmlChal*     smlChalNewFromBinary (SmlAuthType type, const gchar *nonce, gsize length, GError **error);
SmlChal*     smlChalNewFromBase64 (SmlAuthType type, const gchar *nonce, GError **error);
void         smlChalRef           (SmlChal *chal);
void         smlChalUnref         (SmlChal *chal);
const gchar* smlChalGetNonce      (SmlChal *chal); /* Base 64 only */
SmlAuthType  smlChalGetType       (SmlChal *chal);

#endif //_SML_ELEMENTS_H_

/*@}*/
