/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
/**
 * @defgroup SmlCommand SyncML Command
 * @ingroup PublicLowLevelAPI
 * @brief The public part
 * 
 */
/*@{*/

#ifndef _SML_COMMAND_H_
#define _SML_COMMAND_H_

#include <libsyncml/sml_error.h>
#include <libsyncml/dev_inf_api/sml_dev_inf.h>
#include <libsyncml/data_sync_api/sml_location.h>
#include <libsyncml/data_sync_api/sml_data_sync_change_item.h>
#include <libsyncml/data_sync_api/sml_data_sync_data_store_session.h>
#include "sml_elements.h"

typedef struct SmlCommand SmlCommand;
typedef struct SmlStatus  SmlStatus;

SmlStatus*     smlStatusNew           (SmlErrorType data, gsize cmdref, gsize msgref, SmlLocation *sourceref, SmlLocation *targeref, SmlCommandType type, GError **error);
SmlStatus*     smlStatusRef           (SmlStatus *status);
void           smlStatusUnref         (SmlStatus *status);
SmlErrorType   smlStatusGetCode       (SmlStatus *status);
SmlErrorClass  smlStatusGetClass      (SmlStatus *status);
SmlCommand*    smlStatusGetResult     (SmlStatus *status);
gboolean       smlStatusIsResult      (SmlStatus *status);
SmlCommandType smlStatusGetType       (SmlStatus *status);
SmlAnchor*     smlStatusGetAnchor     (SmlStatus *status);
SmlChal*       smlStatusGetChal       (SmlStatus *status);
void           smlStatusSetChal       (SmlStatus *status, SmlChal *chal);
gsize          smlStatusGetCommandRef (SmlStatus *status);
gsize          smlStatusGetMessageRef (SmlStatus *status);
const gchar*   smlStatusGetData       (SmlStatus *status);
SmlLocation*   smlStatusGetSourceRef  (SmlStatus *status);
SmlLocation*   smlStatusGetTargetRef  (SmlStatus *status);

SmlCommand* smlCommandNew      (SmlCommandType type, GError **error);
SmlStatus*  smlCommandNewReply (const SmlCommand *cmd, SmlErrorType code, GError **error);
SmlCommand* smlCommandRef      (SmlCommand *cmd);
void        smlCommandUnref    (SmlCommand *cmd);

SmlCommand* smlCommandNewAlert         (SmlAlertType type, SmlLocation *target, SmlLocation *source, const gchar *next, const gchar *last, const gchar *contenttype, GError **error);
SmlCommand* smlCommandNewSync          (SmlLocation *target, SmlLocation *source, gsize num_changes, GError **error);
SmlCommand* smlCommandNewChange        (SmlDataSyncChangeItem *item, GError **error);
SmlCommand* smlCommandNewResult        (SmlCommand *cmd, SmlLocation *source, gchar *data, gsize size, const gchar *contenttype, GError **error);
SmlCommand* smlCommandNewPut           (SmlLocation *target, SmlLocation *source, const gchar *data, gsize size, const gchar *contenttype, GError **error);
SmlCommand* smlCommandNewGet           (SmlLocation *target, const gchar *contenttype, GError **error);

SmlCommand* smlCommandNewDevInfResult (SmlCommand *cmd, SmlDevInf *devinf, SmlDevInfVersion version, GError **error);
SmlCommand* smlCommandNewDevInfPut    (SmlDevInf *devinf, SmlDevInfVersion version, GError **error);
SmlCommand* smlCommandNewDevInfGet    (SmlDevInfVersion version, GError **error);

SmlCommand* smlCommandNewMap     (SmlLocation *target, SmlLocation *source, GError **error);
gboolean    smlCommandAddMapItem (SmlCommand *map, SmlMapItem *item, GError **error);

void         smlCommandSetID         (SmlCommand *cmd, gsize id);
gsize        smlCommandGetID         (SmlCommand *cmd);
void         smlCommandSetMessageID  (SmlCommand *cmd, gsize id);
gsize        smlCommandGetMessageID  (SmlCommand *cmd);
gsize        smlCommandGetMaxObjSize (SmlCommand *cmd);
gsize        smlCommandGetSize       (SmlCommand *cmd);
void         smlCommandSetSize       (SmlCommand *cmd, gsize size);
void         smlCommandSetNoResp     (SmlCommand *cmd, gboolean noResp);
gboolean     smlCommandGetPushedBack (SmlCommand *cmd);
void         smlCommandSetPushedBack (SmlCommand *cmd, gboolean pushedBack);
SmlItem*     smlCommandGetItem(SmlCommand *cmd);

SmlCommandType smlCommandGetType        (SmlCommand *cmd);
SmlAlertType   smlCommandGetAlertType   (SmlCommand *cmd);
SmlChangeType  smlCommandGetChangeType  (SmlCommand *cmd);
SmlLocation*   smlCommandGetSource      (SmlCommand *cmd);
SmlLocation*   smlCommandGetTarget      (SmlCommand *cmd);
const gchar*   smlCommandGetContentType (SmlCommand *cmd);
SmlAnchor*     smlCommandGetAnchor      (SmlCommand *cmd);

G_CONST_RETURN gchar *smlCommandTypeToString(SmlCommandType type, GError **error);

gsize          smlCommandGetNumberOfChanges   (SmlCommand *cmd); /* NumberOfChanges in sync command */
gsize          smlCommandGetNumChanges        (SmlCommand *cmd); /* number of items in change command */
SmlDataSyncChangeItem* smlCommandGetNthChange (SmlCommand *cmd, gsize n);
gsize          smlCommandGetNumMappings       (SmlCommand *cmd); /* number of items in change command */
SmlMapItem*    smlCommandGetNthMapping        (SmlCommand *cmd, gsize n);
SmlCommand*    smlCommandCloneWithNumMappings (SmlCommand *cmd, gsize n, GError **error);
gboolean       smlCommandRemoveNumMappings    (SmlCommand *cmd, gsize n, GError **error);

SmlStatus* smlCommandResultsGetStatus (SmlCommand *cmd);
void       smlCommandTransferItems    (SmlCommand *source, SmlCommand *target, gsize start);
void       smlCommandFreeFirstChange  (SmlCommand *cmd);

gsize       smlCommandGetNthItemSize (SmlCommand *orig_cmd, gsize n, GError **error);
SmlCommand* smlCommandGetFragment    (SmlCommand *orig_cmd, gsize start, gsize space, GError **error);

#endif //_SML_COMMAND_H_

/*@}*/

