/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

/**
 * @defgroup SmlObjectDatastoreServer SyncML Datastore Server Object
 * @ingroup PublicObjectAPI
 * @brief The public part
 * 
 */
/*@{*/

#ifndef _SML_DS_SERVER_H_
#define _SML_DS_SERVER_H_

#include "../sml_notification.h"
#include <libsyncml/data_sync_api/sml_map_item.h>

typedef struct SmlDsServer SmlDsServer;
typedef struct SmlDsSession SmlDsSession;

typedef void     (* SmlDsSessionConnectCb)      (SmlDsSession *dsession, void *userdata);
typedef gboolean (* SmlDsSessionAlertCb)        (SmlDsSession *dsession, SmlAlertType type, const gchar *last, const gchar *next, void *userdata);
typedef void     (* SmlDsSessionSyncCb)         (SmlDsSession *dsession, gsize numchanges, void *userdata);
typedef gboolean (* SmlDsSessionChangeCb)       (SmlDsSession *dsession, SmlDataSyncChangeItem *item, void *userdata, GError **error);
typedef void     (* SmlDsSessionChangeStatusCb) (SmlDsSession *dsession, SmlStatus *status, void *userdata);
typedef void     (* SmlDsSessionMapCb)          (SmlDsSession *dsession, SmlMapItem *item, void *userdata);

SmlDsServer*    smlDsServerNew                (const gchar *type, SmlLocation *location, GError **error);
SmlDsServer*    smlDsClientNew                (const gchar *type, SmlLocation *location, SmlLocation *target, GError **error);
SmlDsServerType smlDsServerGetServerType      (SmlDsServer *server);
void            smlDsServerFree               (SmlDsServer *server);
gboolean        smlDsServerRegister           (SmlDsServer *server, SmlManager *manager, GError **error);
gboolean        smlDsServerAddSan             (SmlDsServer *server, SmlNotification *san, GError **error);
void            smlDsServerReset              (SmlDsServer *server);
void            smlDsServerSetConnectCallback (SmlDsServer *server, SmlDsSessionConnectCb callback, void *userdata);
const gchar*    smlDsServerGetLocation        (SmlDsServer *server);
const gchar*    smlDsServerGetContentType     (SmlDsServer *server);
SmlDsSession*   smlDsServerSendAlert          (SmlDsServer *server, SmlSession *session, SmlAlertType type, const gchar *last, const gchar *next, SmlStatusReplyCb callback, void *userdata, GError **error);

SmlDsSession* smlDsSessionNew            (SmlDsServer *server, SmlSession *session, GError **error);
SmlDsSession* smlDsSessionRef            (SmlDsSession *dsession);
void          smlDsSessionUnref          (SmlDsSession *dsession);
void          smlDsSessionDispatch       (SmlDsSession *dsession);
gboolean      smlDsSessionCheck          (SmlDsSession *dsession);
void          smlDsSessionGetAlert       (SmlDsSession *dsession, SmlDsSessionAlertCb callback, void *userdata);
gboolean      smlDsSessionSendAlert      (SmlDsSession *dsession, SmlAlertType type, const gchar *last, const gchar *next, SmlStatusReplyCb callback, void *userdata, GError **error);
void          smlDsSessionGetChanges     (SmlDsSession *dsession, SmlDsSessionChangeCb chgCallback, void *userdata);
void          smlDsSessionGetSync        (SmlDsSession *dsession, SmlDsSessionSyncCb chgCallback, void *userdata);
gboolean      smlDsSessionSendSync       (SmlDsSession *dsession, gsize num_changes, SmlStatusReplyCb callback, void *userdata, GError **error);
gboolean      smlDsSessionQueueChange    (SmlDsSession *dsession, SmlDataSyncChangeItem *item, SmlDsSessionChangeStatusCb callback, void *userdata, GError **error);
gboolean      smlDsSessionCloseSync      (SmlDsSession *dsession, GError **error);
void          smlDsSessionGetMapping     (SmlDsSession *dsession, SmlDsSessionMapCb mapCallback, void *userdata);
gboolean      smlDsSessionQueueMap       (SmlDsSession *dsession, SmlMapItem *item, GError **error);
gboolean      smlDsSessionCloseMap       (SmlDsSession *dsession, SmlStatusReplyCb callback, void *userdata, GError **error);
const gchar*  smlDsSessionGetLocation    (SmlDsSession *dsession);
const gchar*  smlDsSessionGetContentType (SmlDsSession *dsession);
SmlDsServer*  smlDsSessionGetServer      (SmlDsSession *dsession);
SmlLocation*  smlDsSessionGetTarget      (SmlDsSession *dsession);
SmlSession*   smlDsSessionGetSession     (SmlDsSession *dsession);

/* multi session safe SAN handling */

typedef SmlErrorType (* SmlDsServerSanSessionCb) (SmlDsServer *dsserver, SmlSession *session, SmlAlertType type, void *userdata);
void smlDsServerSetSanSessionCallback (SmlDsServer *server, SmlDsServerSanSessionCb callback, void *userdata);

#endif //_SML_DS_SERVER_H_
/*@}*/
