/* sml_dev_inf_ctcap.c
 *
 * Copyright (C) 2009 Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

#include "sml_dev_inf_ctcap.h"
#include <libsyncml/syncml.h>
#include "../sml_error_internals.h"

G_DEFINE_TYPE (SmlDevInfCTCap, sml_dev_inf_ctcap, G_TYPE_OBJECT)

enum
{
	PROP_0,
	PROP_CONTENT_TYPE,
	PROP_PROPERTIES
};

struct _SmlDevInfCTCapPrivate
{
	SmlDevInfContentType* content_type;
	GList*                properties;
};

static void
sml_dev_inf_ctcap_get_property (GObject    *object,
                                guint       property_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
	switch (property_id) {
	case PROP_CONTENT_TYPE:
		g_value_set_object (value, SML_DEV_INF_CTCAP (object)->priv->content_type);
		break;
	case PROP_PROPERTIES:
		g_value_set_pointer (value, SML_DEV_INF_CTCAP (object)->priv->properties);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
sml_dev_inf_ctcap_set_property (GObject      *object,
                                guint         property_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
	switch (property_id) {
	case PROP_CONTENT_TYPE:
        	if (SML_DEV_INF_CTCAP (object)->priv->content_type)
        		g_object_unref (SML_DEV_INF_CTCAP (object)->priv->content_type);
		g_object_ref(SML_DEV_INF_CONTENT_TYPE (value));
		SML_DEV_INF_CTCAP (object)->priv->content_type = SML_DEV_INF_CONTENT_TYPE (value);
		break;
	case PROP_PROPERTIES:
		SML_DEV_INF_CTCAP (object)->priv->properties = (GList *) g_value_get_pointer(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
sml_dev_inf_ctcap_finalize (GObject *object)
{
	SmlDevInfCTCap *self = SML_DEV_INF_CTCAP (object);
	g_object_unref(self->priv->content_type);
	while(self->priv->properties) {
		SmlDevInfProperty *prop = self->priv->properties->data;
		self->priv->properties = g_list_remove(self->priv->properties, prop);
		g_object_unref(prop);
	}
	/* all pointers must be NULL */
	self->priv->content_type = NULL;
	G_OBJECT_CLASS (sml_dev_inf_ctcap_parent_class)->finalize (object);
}

static void
sml_dev_inf_ctcap_class_init (SmlDevInfCTCapClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (SmlDevInfCTCapPrivate));

	object_class->get_property = sml_dev_inf_ctcap_get_property;
	object_class->set_property = sml_dev_inf_ctcap_set_property;
	object_class->finalize     = sml_dev_inf_ctcap_finalize;

	/**
	 * SmlDevInfCTCap:ContentType:
	 *
	 * The The versioned content type of the content type capabilities property.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_CONTENT_TYPE,
	                                 g_param_spec_object ("ContentType",
	                                                      "",
	                                                      "The versioned content type of the content type capabilities",
	                                                      G_TYPE_OBJECT,
	                                                      G_PARAM_READWRITE));
	/**
	 * SmlDevInfCTCap:Properties:
	 *
	 * The A list of SmlDevInfProperty objects property.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_PROPERTIES,
	                                 g_param_spec_pointer ("Properties",
	                                                      "Properties",
	                                                      "A list of SmlDevInfProperty objects",
	                                                      G_PARAM_PRIVATE));

}

static void
sml_dev_inf_ctcap_init (SmlDevInfCTCap *self)
{
	self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
	                                          SML_TYPE_DEV_INF_CTCAP,
	                                          SmlDevInfCTCapPrivate);
}

/**
 * sml_dev_inf_ctcap_new:
 *
 * Creates a new instance of #SmlDevInfCTCap.
 *
 * Return value: the newly created #SmlDevInfCTCap instance
 */
SmlDevInfCTCap*
sml_dev_inf_ctcap_new (SmlDevInfContentType *ct, GError **error)
{
	CHECK_ERROR_REF

	if (ct == NULL) {
		*error = g_error_new(SML_ERROR, SML_ERROR_GENERIC, "Every CTCap must have a content type.");
		return NULL;
	}
	SmlDevInfCTCap *self = g_object_new (SML_TYPE_DEV_INF_CTCAP, NULL);
	g_object_ref(ct);
	self->priv->content_type = ct;
	return self;
}

/**
 * sml_dev_inf_ctcap_get_content_type:
 * @self: A #SmlDevInfCTCap
 *
 * Gets the  property.
 *
 * Return value: 
 */
SmlDevInfContentType*
sml_dev_inf_ctcap_get_content_type (SmlDevInfCTCap *self)
{
	g_return_val_if_fail (SML_IS_DEV_INF_CTCAP (self), NULL);
	return self->priv->content_type;
}

/**
 * sml_dev_inf_ctcap_set_content_type:
 * @self: A #SmlDevInfCTCap
 * @contenttype:
 *
 * Sets the  property.
 */
gboolean
sml_dev_inf_ctcap_set_content_type (SmlDevInfCTCap *self,
                                    SmlDevInfContentType *content_type,
                                    GError **error)
{
	CHECK_ERROR_REF
	sml_return_val_error_if_fail (SML_IS_DEV_INF_CTCAP (self), FALSE, error, SML_ERROR_GENERIC, "There is no SmlDevInfCTCap object.");
	sml_return_val_error_if_fail (content_type, FALSE, error, SML_ERROR_GENERIC, "The content type cannot be deleted only replace is allowed.");
	sml_return_val_error_if_fail (SML_IS_DEV_INF_CONTENT_TYPE (content_type), FALSE, error, SML_ERROR_GENERIC, "The content_type must be of type SmlDevInfContentType.");

	if (self->priv->content_type)
		g_object_unref (self->priv->content_type);

	self->priv->content_type = g_object_ref (content_type);
	sml_return_val_error_if_fail (self->priv->content_type, FALSE, error, SML_ERROR_GENERIC, "Cannot copy content type - out of memory.");

	return TRUE;
}

/**
 * sml_dev_inf_ctcap_num_properties:
 * @self: A #SmlDevInfCTCap
 *
 * 
 */
gsize
sml_dev_inf_ctcap_num_properties (SmlDevInfCTCap *self)
{
	g_return_val_if_fail (SML_IS_DEV_INF_CTCAP (self), 0);
	return g_list_length(self->priv->properties);
}

/**
 * sml_dev_inf_ctcap_get_nth_property:
 * @self: A #SmlDevInfCTCap
 *
 * 
 */
SmlDevInfProperty*
sml_dev_inf_ctcap_get_nth_property (SmlDevInfCTCap *self,
                                    gsize n)
{
	g_return_val_if_fail (SML_IS_DEV_INF_CTCAP (self), NULL);
	return (SmlDevInfProperty *) g_list_nth_data(self->priv->properties, n);
}

/**
 * sml_dev_inf_ctcap_add_property:
 * @self: A #SmlDevInfCTCap
 *
 * 
 */
gboolean
sml_dev_inf_ctcap_add_property (SmlDevInfCTCap *self,
                                SmlDevInfProperty *property,
                                GError **error)
{
	CHECK_ERROR_REF
	sml_return_val_error_if_fail (SML_IS_DEV_INF_CTCAP (self), FALSE, error, SML_ERROR_GENERIC, "There is no SmlDevInfCTCap object.");
	sml_return_val_error_if_fail (property, FALSE, error, SML_ERROR_GENERIC, "The property object is missing.");
	sml_return_val_error_if_fail (SML_IS_DEV_INF_PROPERTY (property), FALSE, error, SML_ERROR_GENERIC, "The property must be of type SmlDevInfProperty.");

	g_object_ref(property);
	self->priv->properties = g_list_append(self->priv->properties, property);
	sml_return_val_error_if_fail (self->priv->properties, FALSE, error, SML_ERROR_GENERIC, "Cannot add Property to Property list of CTCap.");

	return TRUE;
}
