/* sml_dev_inf_content_type.c
 *
 * Copyright (C) 2009 Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

#include "sml_dev_inf_content_type.h"
#include <libsyncml/syncml.h>
#include "../sml_error_internals.h"

G_DEFINE_TYPE (SmlDevInfContentType, sml_dev_inf_content_type, G_TYPE_OBJECT)

enum
{
	PROP_0,
	PROP_CTTYPE,
	PROP_VERCT
};

struct _SmlDevInfContentTypePrivate
{
        gchar*  cttype;
        gchar*  verct;
};

static void
sml_dev_inf_content_type_get_property (GObject    *object,
                                       guint       property_id,
                                       GValue     *value,
                                       GParamSpec *pspec)
{
	switch (property_id) {
	case PROP_CTTYPE:
		g_value_set_string (value, SML_DEV_INF_CONTENT_TYPE (object)->priv->cttype);
		break;
	case PROP_VERCT:
		g_value_set_string (value, SML_DEV_INF_CONTENT_TYPE (object)->priv->verct);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
sml_dev_inf_content_type_set_property (GObject      *object,
                                       guint         property_id,
                                       const GValue *value,
                                       GParamSpec   *pspec)
{
	switch (property_id) {
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
sml_dev_inf_content_type_finalize (GObject *object)
{
	g_return_if_fail (SML_IS_DEV_INF_CONTENT_TYPE (object));
	SmlDevInfContentType *self = (SmlDevInfContentType*) object;
	if (self->priv->cttype != NULL) {
		g_free(self->priv->cttype);
		self->priv->cttype = NULL;
	}
	if (self->priv->verct != NULL) {
		g_free(self->priv->verct);
		self->priv->verct = NULL;
	}
	G_OBJECT_CLASS (sml_dev_inf_content_type_parent_class)->finalize (object);
}

static void
sml_dev_inf_content_type_class_init (SmlDevInfContentTypeClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (SmlDevInfContentTypePrivate));

	object_class->get_property = sml_dev_inf_content_type_get_property;
	object_class->set_property = sml_dev_inf_content_type_set_property;
	object_class->finalize     = sml_dev_inf_content_type_finalize;

	/**
	 * SmlDevInfContentType:CTType:
	 *
	 * The  property.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_CTTYPE,
	                                 g_param_spec_string ("CTType",
	                                                      "",
	                                                      "",
	                                                      "text/plain",
	                                                      G_PARAM_READABLE));
	/**
	 * SmlDevInfContentType:VerCT:
	 *
	 * The  property.
	 */
	g_object_class_install_property (object_class,
	                                 PROP_VERCT,
	                                 g_param_spec_string ("VerCT",
	                                                      "",
	                                                      "",
	                                                      "1.0",
	                                                      G_PARAM_READABLE));

}

static void
sml_dev_inf_content_type_init (SmlDevInfContentType *self)
{
	self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
	                                          SML_TYPE_DEV_INF_CONTENT_TYPE,
	                                          SmlDevInfContentTypePrivate);
}

/**
 * sml_dev_inf_content_type_new:
 *
 * Creates a new instance of #SmlDevInfContentType.
 *
 * Return value: the newly created #SmlDevInfContentType instance
 */
SmlDevInfContentType*
sml_dev_inf_content_type_new (const gchar* cttype,
                              const gchar* verct,
                              GError **error)
{
	CHECK_ERROR_REF

	SmlDevInfContentType* self = g_object_new (SML_TYPE_DEV_INF_CONTENT_TYPE, NULL);
	if (cttype != NULL) {
		self->priv->cttype = g_strdup(cttype);
	}
	if (verct != NULL) {
		self->priv->verct = g_strdup(verct);
	}
	return self;
}

/**
 * sml_dev_inf_content_type_get_cttype:
 * @self: A #SmlDevInfContentType
 *
 * Gets the  property.
 *
 * Return value: 
 */
G_CONST_RETURN gchar*
sml_dev_inf_content_type_get_cttype (SmlDevInfContentType *self)
{
	g_return_val_if_fail (SML_IS_DEV_INF_CONTENT_TYPE (self), NULL);
	return self->priv->cttype;
}

/**
 * sml_dev_inf_content_type_get_verct:
 * @self: A #SmlDevInfContentType
 *
 * Gets the  property.
 *
 * Return value: 
 */
G_CONST_RETURN gchar*
sml_dev_inf_content_type_get_verct (SmlDevInfContentType *self)
{
	g_return_val_if_fail (SML_IS_DEV_INF_CONTENT_TYPE (self), NULL);
	return self->priv->verct;
}

/**
 * sml_dev_inf_content_type_set_cttype:
 * @self: A #SmlDevInfContentType
 * @cttype: The content type string
 * @error: A potential failure reason
 *
 * Sets the  property.
 *
 * Return value: 
 */
gboolean
sml_dev_inf_content_type_set_cttype (SmlDevInfContentType *self, const gchar* cttype, GError **error)
{
	CHECK_ERROR_REF
	sml_return_val_error_if_fail (SML_IS_DEV_INF_CONTENT_TYPE (self), FALSE, error, SML_ERROR_GENERIC, "There is no SmlDevInfContentType object.");
	sml_return_val_error_if_fail (cttype != NULL, FALSE, error, SML_ERROR_GENERIC, "The CTType must always be set for a SmlDevInfContentType object.");

	if (self->priv->cttype) {
		g_free(self->priv->cttype);
		self->priv->cttype = NULL;
	}

	self->priv->cttype = g_strdup(cttype);
	sml_return_val_error_if_fail (self->priv->cttype, FALSE, error, SML_ERROR_GENERIC, "Cannot copy cttype - out of memory.");

	return TRUE;
}

/**
 * sml_dev_inf_content_type_get_verct:
 * @self: A #SmlDevInfContentType
 * @verct: The version of the content type
 * @error: A potential failure reason
 *
 * Gets the  property.
 *
 * Return value: 
 */
gboolean
sml_dev_inf_content_type_set_verct (SmlDevInfContentType *self, const gchar* verct, GError **error)
{
	CHECK_ERROR_REF
	sml_return_val_error_if_fail (SML_IS_DEV_INF_CONTENT_TYPE (self), FALSE, error, SML_ERROR_GENERIC, "There is no SmlDevInfContentType object.");
	sml_return_val_error_if_fail (verct != NULL, FALSE, error, SML_ERROR_GENERIC, "The VerCT must always be set for a SmlDevInfContentType object.");

	if (self->priv->verct) {
		g_free(self->priv->verct);
		self->priv->verct = NULL;
	}

	self->priv->verct = g_strdup(verct);
	sml_return_val_error_if_fail (self->priv->verct, FALSE, error, SML_ERROR_GENERIC, "Cannot copy verct - out of memory.");

	return TRUE;
}

