/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "transport.h"
#include "libsyncml/sml_error_internals.h"
#include "libsyncml/sml_support.h"
#include <string.h>

#include "sml_data_sync_defines.h"
#include "sml_data_sync_private.h"
#include "sml_data_sync_data_store_private.h"

gboolean
smlDataSyncTransportObexClientInit (SmlDataSync *self,
                                    GError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, self, error);
	CHECK_ERROR_REF

	if (self->priv->url &&
	    !smlTransportSetConfigOption(
			self->priv->tsp,
			SML_TRANSPORT_CONFIG_URL, self->priv->url,
			error))
		goto error;

	smlTrace(TRACE_EXIT, "%s - TRUE", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, (*error)->message);
	return FALSE;
}

gboolean
smlDataSyncTransportObexClientConnect (SmlDataSync *self,
                                       GError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, self, error);
	CHECK_ERROR_REF

	/* Connect transport layer */

	if (!smlTransportConnect(self->priv->tsp, error))
		goto error;

	/* Create a SAN */

	SmlNotificationVersion sanVersion = SML_SAN_VERSION_UNKNOWN;
	switch(self->priv->version)
	{
		case SML_VERSION_10:
			sanVersion = SML_SAN_VERSION_10;
			break;
		case SML_VERSION_11:
			sanVersion = SML_SAN_VERSION_11;
			break;
		case SML_VERSION_12:
			sanVersion = SML_SAN_VERSION_12;
			break;
		case SML_VERSION_UNKNOWN:
			g_set_error(error, SML_ERROR, SML_ERROR_GENERIC, "Unknown SyncML SAN Version.");
			goto error;
			break;
	}

	SmlNotification *san = smlNotificationNew(
				sanVersion,
				SML_SAN_UIMODE_UNSPECIFIED,
				SML_SAN_INITIATOR_USER, 1,
				sml_location_get_uri(self->priv->local), "/",
				self->priv->use_wbxml ? SML_MIMETYPE_WBXML : SML_MIMETYPE_XML,
				error);
	if (!san)
		goto error;

	smlNotificationSetManager(san, self->priv->manager);

	/* init digest - SAN uses always MD5 */
	if (self->priv->username != NULL && strlen(self->priv->username))
	{
		SmlCred *cred = smlCredNewAuth(
					SML_AUTH_TYPE_MD5,
					self->priv->username,
					self->priv->password,
					error);
		if (!cred)
			goto error;
		smlNotificationSetCred(san, cred);
		smlCredUnref(cred);
		cred = NULL;
	}

	GList *o = self->priv->data_stores;
	for (; o; o = o->next) { 
		SmlDataSyncDataStore *datastore = o->data;
		if (!smlDsServerAddSan(datastore->priv->data_store_server, san, error))
			goto error;
	}

	if (!smlNotificationSend(san, self->priv->tsp, error))
		goto error;

	smlNotificationFree(san);
	san = NULL;

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, (*error)->message);
	return FALSE;
}

