/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "transport.h"
#include "libsyncml/sml_error_internals.h"
#include "libsyncml/sml_support.h"

#include "sml_data_sync_defines.h"
#include "sml_data_sync_private.h"
#include "sml_data_sync_session_private.h"
#include "data_sync_client.h"
#include "data_sync_devinf.h"


gboolean
smlDataSyncTransportHttpClientInit (SmlDataSync *self,
                                    GError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, self, error);
	CHECK_ERROR_REF

	if (self->priv->url != NULL &&
	    !smlTransportSetConfigOption(
			self->priv->tsp,
			SML_TRANSPORT_CONFIG_URL, self->priv->url,
			error))
		goto error;
	if (self->priv->username != NULL &&
	    !smlTransportSetConfigOption(
			self->priv->tsp,
			SML_TRANSPORT_CONFIG_USERNAME, self->priv->username,
			error))
		goto error;
	if (self->priv->password != NULL &&
	    !smlTransportSetConfigOption(
			self->priv->tsp,
			SML_TRANSPORT_CONFIG_PASSWORD, self->priv->password,
			error))
		goto error;

	smlTrace(TRACE_EXIT, "%s - TRUE", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, (*error)->message);
	return FALSE;
}

static SmlDataSyncSession*
smlDataSyncTransportHttpClientInitNewSession (SmlDataSync *self,
                                              GError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, self, error);
	smlAssert(self->priv->url);
	smlAssert(self->priv->tsp);
	smlAssert(self->priv->manager);
	smlAssert(self->priv->agent);

	gsize sessionString = smlManagerGetNewSessionID(self->priv->manager);
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_CLIENT,
				self->priv->use_wbxml?SML_MIMETYPE_WBXML:SML_MIMETYPE_XML,
				self->priv->version,
				SML_PROTOCOL_SYNCML,
				self->priv->remote, self->priv->local,
				sessionString, 0, error);
	if (!session)
		goto error;

	SmlDataSyncSession *dss = sml_data_sync_get_session(self, session, error);
	if (!dss)
		goto error;

	/* register all the add-ons */
	if (!smlManagerSessionAdd(self->priv->manager, session, NULL, error))
		goto error;
	if (!smlDevInfAgentRegisterSession(self->priv->agent, self->priv->manager, session, error))
        	goto error;

	smlTrace(TRACE_EXIT, "%s", __func__);
	return dss;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, (*error)->message);
	return NULL;
}

gboolean
smlDataSyncTransportHttpClientConnect (SmlDataSync *self,
                                       GError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, self, error);
	CHECK_ERROR_REF

	/* init new session */
	SmlDataSyncSession *dss = smlDataSyncTransportHttpClientInitNewSession(self, error);
	if (!dss)
		goto error;

	/* send the device information */
	if (!sml_data_sync_dev_inf_manage_session(dss, TRUE, error))
		goto error;
	smlTrace(TRACE_INTERNAL, "%s: sent devinf", __func__);

	/* prepare correct alert type */
	GList *o = self->priv->data_stores;
	for (; o; o = o->next) { 
		SmlDataSyncDataStore *datastore = o->data;

		if (!sml_data_sync_session_client_send_alert(dss, datastore, SML_ALERT_UNKNOWN, error))
			goto error;
	}
	smlTrace(TRACE_INTERNAL, "%s: all datastores created their alerts", __func__);

	/* If no SAN and no server alerted sync is used
	 * then we must set the actual package to 1.
	 * Otherwise the state management thinks the first
	 * received final is part of a SAN package.
	 */
	dss->priv->actual_package = SML_PACKAGE_1;

	/* flush package 1 */
	if (!smlSessionFlush(dss->priv->session, TRUE, error))
		goto error;

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, (*error)->message);
	return FALSE;
}

