﻿/* Copyright (c) Tamir Khason (http://khason.net/). All rights reserved.
 * 
 * Released under MS-PL on CodePlex (http://www.codeplex.com/FM/)
 */

#region using
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
#endregion

namespace USBFM {
   /// <summary>General interfaces for Radio Device</summary>
   /// <example>
   /// <code>
   /// using (var device = USBRadioDevice.FindDevice(0x0000, 0x1111)) {
   ///  if(!device.IsInvalid) Console.WriteLine("FM USB Device {0} was found!",device);
   ///  else Console.WriteLine("Device was not found");
   /// }
   /// </code>
   /// </example>
   public class USBRadioDevice : USBDevice, INotifyPropertyChanged {
      internal USBRadioDevice(uint pid, uint vid) : base(pid, vid) { }
      /// <summary>Radio data capabilities (<see cref="RadioData"/>) of <see cref="USBRadioDevice"/></summary>
      public RadioData Capabilities { get; internal set; }
      /// <summary>Radio station frequencies presets on <see cref="USBRadioDevice"/></summary>
      public double[] Presets { get; internal set; }
      private double _CurrentStation;
      /// <summary>Current station <see cref="USBRadioDevice"/> tuned to</summary>
      public double CurrentStation { get { return _CurrentStation; } internal set { if (_CurrentStation != value) { _CurrentStation = value; OnPropertyChanged("CurrentStation");  } } }
      /// <summary>True if current station of <see cref="USBRadioDevice"/> transmittes in stereo mode</summary>
      public bool IsStereo { get; internal set; }
      private double _CurrentFrequency;
      /// <summary>Current frequency <see cref="USBRadioDevice"/> tuned to</summary>
      public double CurrentFrequency { get { return _CurrentFrequency; } internal set { if (_CurrentFrequency != value) { _CurrentFrequency = value; OnPropertyChanged("CurrentFrequency"); } } }
      /// <summary>True if <see cref="USBRadioDevice"/> currently in tuning mode and cannot be used</summary>
      public bool IsTuning { get; internal set; }
      /// <summary>True if all sounds of <see cref="USBRadioDevice"/> are currently muted</summary>
      public bool IsMuted { get; internal set; }
      private RDSData _rds;
      /// <summary>Current RDS information (<see cref="RDSData"/>) recieved by <see cref="USBRadioDevice"/></summary>
      public RDSData RDS { get { return _rds; } internal set {  _rds = value; OnPropertyChanged("RDS");  } }

      /// <summary>Method to seek and find <see cref="USBRadioDevice"/> in the operation system</summary>
      /// <param name="pid">Product ID of <see cref="USBRadioDevice"/> to find</param>
      /// <param name="vid">Vendor ID of <see cref="USBRadioDevice"/> to find</param>
      /// <returns>Fully initialized and ready to work instance of <see cref="USBRadioDevice"/></returns>      
      public static new USBRadioDevice FindDevice(uint pid, uint vid) {
         var device = new USBRadioDevice(pid, vid);
         var fillRegisters = device.InitRegisters(); 
         var fmData = device.ReadFMInfo();
         var initData = device.InitFMInfo();
         if (!device.IsInvalid && fillRegisters && fmData && initData) return device;
         else throw new ApplicationException(string.Format("Cannot initialize radio device {0}.", pid));
      }

      #region INotifyPropertyChanged Members

      public event PropertyChangedEventHandler PropertyChanged;

      internal void OnPropertyChanged(string name) { if (PropertyChanged != null)PropertyChanged(this, new PropertyChangedEventArgs(name)); }

      #endregion
   }

   /// <summary>Radio data for <see cref="USBRadioDevice"/></summary>
   public class RadioData {
      /// <summary>Software version of <see cref="USBRadioDevice"/></summary>
      public ushort SoftwareVersion { get; internal set; }
      /// <summary>Hardware version of <see cref="USBRadioDevice"/></summary>
      public ushort HardwareVersion { get; internal set; }
      /// <summary>Part number of <see cref="USBRadioDevice"/></summary>
      public ushort PartNumber { get; internal set; }
      /// <summary>Manufacturer ID of <see cref="USBRadioDevice"/></summary>
      public ushort ManufacturerID { get; internal set; }
      /// <summary>FM chip version version of <see cref="USBRadioDevice"/></summary>
      public ushort ChipVersion { get; internal set; }
      /// <summary>Device version of <see cref="USBRadioDevice"/></summary>
      public ushort DeviceVersion { get; internal set; }
      /// <summary>Firmware version of <see cref="USBRadioDevice"/></summary>
      public ushort FirmwareVersion { get; internal set; }
      /// <summary>Station seek treshold of <see cref="USBRadioDevice"/></summary>
      public ushort SeekThreshold { get; internal set; }
      internal ushort Band { get; set; }
      /// <summary>Minimal frequency of <see cref="USBRadioDevice"/></summary>
      public decimal MinFreq { get { return Band == USBFM.USBRDSDeviceMethods.Native.DATA_BAND_76_90MHZ ? 76 : 87.5m; } }
      /// <summary>Maximum frequency of <see cref="USBRadioDevice"/></summary>
      public decimal MaxFreq { get { return Band == USBFM.USBRDSDeviceMethods.Native.DATA_BAND_76_90MHZ ? 90 : 108; } }
      /// <summary>Tune treshold spacing of <see cref="USBRadioDevice"/></summary>
      public ushort Spacing { get; internal set; }
      /// <summary>Deemphasis value of <see cref="USBRadioDevice"/></summary>
      public ushort Deemphasis { get; internal set; }
      /// <summary>Special capabilities and extension flags for <see cref="USBRadioDevice"/></summary>
      public int ExFlags { get; internal set; }
      internal ushort[] PI { get; set; }
      internal ushort EonPI { get; set; }
      internal double[] PI_Freq { get; set; }
   }

   /// <summary>Radio data system information for <see cref="USBRadioDevice"/></summary>
   public class RDSData : INotifyPropertyChanged {
      #region properties
      private double _CurrentStation;
      /// <summary>Station <see cref="USBRadioDevice"/> tuned to</summary>
      public double CurrentStation { get { return _CurrentStation; } internal set { if (value != _CurrentStation) { _CurrentStation = value; OnPropertyChanged("CurrentStation"); } } }
      private byte _SignalStrength;
      /// <summary>Signal strength of <see cref="USBRadioDevice"/></summary>
      public byte SignalStrength { get { return _SignalStrength; } internal set { if (value != _SignalStrength) { _SignalStrength = value; OnPropertyChanged("SignalStrength"); } } }
      private bool _IsStereo;
      /// <summary>True if <see cref="USBRadioDevice"/> recieves stereo sound</summary>
      public bool IsStereo { get { return _IsStereo; } internal set { if (value != _IsStereo) { _IsStereo = value; OnPropertyChanged("IsStereo"); } } }
      public string RDSText { get; internal set; }
      private string _PS;
      /// <summary>Program service info for <see cref="USBRadioDevice"/></summary>
      public string PS { get { return _PS; } internal set { if (value != _PS) { _PS = value; OnPropertyChanged("PS"); } } }
      /// <summary>Program information info for <see cref="USBRadioDevice"/></summary>
      public ushort PI { get; internal set; }
      /// <summary>Program type info for <see cref="USBRadioDevice"/></summary>
      public byte PTY { get; internal set; }
      private bool _HasTA;
      /// <summary>True if current radio data contains traffic anouncement information for <see cref="USBRadioDevice"/></summary>
      public bool HasTA { get { return _HasTA; } internal set { if (value != _HasTA) { _HasTA = value; OnPropertyChanged("HasTA"); } } }
      private bool _HasTP;
      /// <summary>True if current radio data contains traffic program information for <see cref="USBRadioDevice"/></summary>
      public bool HasTP { get { return _HasTP; } internal set { if (value != _HasTP) { _HasTP = value; OnPropertyChanged("HasTP"); } } }
      private bool _HasMS;
      /// <summary>True if current radio data contains music speech information for <see cref="USBRadioDevice"/></summary>
      public bool HasMS { get { return _HasMS; } internal set { if (value != _HasMS) { _HasMS = value; OnPropertyChanged("HasMS"); } } }
      private string _PTYString;
      /// <summary>Program type string for <see cref="USBRadioDevice"/></summary>
      public string PTYString { get { return _PTYString; } internal set { if (value != _PTYString) { _PTYString = value; OnPropertyChanged("PTYString"); } } }
      private List<double> _AF;
      /// <summary>List of alternative frequencines for <see cref="USBRadioDevice"/></summary>
      public List<double> AF { get { return _AF; } internal set { if (value != _AF) { _AF = value; OnPropertyChanged("AF"); } } }
      private ushort _TrafficPI;
      /// <summary>Traffic program info code for <see cref="USBRadioDevice"/></summary>
      public ushort TrafficPI { get { return _TrafficPI; } internal set { if (value != _TrafficPI) { _TrafficPI = value; OnPropertyChanged("TrafficPI "); } } }
      private string _PICountry;
      /// <summary>Country program info string for <see cref="USBRadioDevice"/></summary>
      public string PICountry { get { return _PICountry; } internal set { if (value != _PICountry) { _PICountry = value; OnPropertyChanged("PICountry"); } } }
      private string _PIRegion;
      /// <summary>Region program info string for <see cref="USBRadioDevice"/></summary>
      public string PIRegion { get { return _PIRegion; } internal set { if (value != _PIRegion) { _PIRegion = value; OnPropertyChanged("PIRegion"); } } }
      private string _TMC;
      /// <summary>Traffic message channel string for <see cref="USBRadioDevice"/></summary>
      public string TMC { get { return _TMC; } internal set { if (value != _TMC) { _TMC = value; OnPropertyChanged("TMC"); } } }
      #endregion

      #region private
      internal ushort rdsBlocksValid, validationLimit, internalPI;
      internal RDSGroupType groupType;
      internal byte ecc, c_ecc;
      internal byte piValidateCount, ptyValidateCount;
      internal ushort piNonValidated;
      internal byte ptyNonValidated;
      internal byte TrafficStart, TrafficEnd;

      private Triple<byte, byte, byte>[] psTmp = new Triple<byte, byte, byte>[8];
      private Triple<ushort, ushort, byte>[] rtTmp = new Triple<ushort, ushort, byte>[64];

      internal void updatePI(ushort data) {
         if (piNonValidated != data) {
            piNonValidated = data;
            piValidateCount = 1;
         } else piValidateCount++;

         if (piValidateCount > validationLimit) {
            internalPI = piNonValidated;
            ushort _prn = (ushort)(internalPI & 0xff);
            byte _area = (byte)((internalPI >> 8) & 0xf);
            byte _country = (byte)((internalPI >> 12) & 0xf);
            PIRegion = PIAreas.DecodeName(_area);
            PICountry = PICountries.DecodeName(_country, ecc);
         }
      }

      internal void updatePTY(byte data) {
         if (ptyNonValidated != data) {
            ptyNonValidated = data;
            ptyValidateCount = 1;
         } else ptyValidateCount++;
         if (ptyValidateCount > validationLimit) {
            PTYString = PTYTexts.DecodeText(ptyNonValidated);
         }
      }

      internal void updatePS(byte addr, byte data) {
         //init all
         if (psTmp[addr].IsInDefaultValue()) {
            for (int i = 0; i < psTmp.Length; i++) psTmp[i] = new Triple<byte, byte, byte>();
         }
         if (psTmp[addr].Value1 == data) {
            if (psTmp[addr].Value3 < PS_VALIDATE_LIMIT) psTmp[addr].Value3++;
            else {
               psTmp[addr].Value3 = PS_VALIDATE_LIMIT;
               psTmp[addr].Value2 = data;
            }
         } else if (psTmp[addr].Value2 == data) {
            if (psTmp[addr].Value3 >= PS_VALIDATE_LIMIT) {
               psTmp.Where(t => t.Value3 > 1).Apply(t => t.Value3--);
            }
            psTmp[addr].Value3 = PS_VALIDATE_LIMIT + 1;
            psTmp[addr].Value2 = psTmp[addr].Value1;
            psTmp[addr].Value1 = data;
         } else if (psTmp[addr].Value3 != 0) {
            psTmp[addr].Value1 = data;
            psTmp[addr].Value3 = 1;
         } else psTmp[addr].Value2 = data;

         if (!psTmp.Any(t => t.Value3 < PS_VALIDATE_LIMIT)) {
            PS = Encoding.Default.GetString(psTmp.Select(t => t.Value1).ToArray());
         }
      }

      internal void updateRT(bool abflag, int addr, ushort[] registers, byte errorFlags) {
         //init all
         if (rtTmp[addr].IsInDefaultValue()) {
            for (int i = 0; i < rtTmp.Length; i++) rtTmp[i] = new Triple<ushort, ushort, byte>();
         }
         for (int i = 0; i < registers.Length; i++) {
            if (rtTmp[addr + i].Value1 == registers[i]) {
               if (rtTmp[addr + i].Value3 < RT_VALIDATE_LIMIT) rtTmp[addr + i].Value3++;
               else {
                  rtTmp[addr + i].Value3 = RT_VALIDATE_LIMIT;
                  rtTmp[addr + i].Value2 = registers[i];
               }
            } else if (rtTmp[addr + i].Value2 == registers[i]) {
               if (rtTmp[addr + i].Value3 >= RT_VALIDATE_LIMIT) {
                  rtTmp.Where(t => t.Value3 > 1).Apply(t => t.Value3--);
                  if (!rtTmp.Any(t => t.Value3 < RT_VALIDATE_LIMIT) | rtTmp.Any(t => t.Value1 == 0x0d)) {
                     Console.WriteLine();
                  }

               }
               rtTmp[addr + i].Value3 = RT_VALIDATE_LIMIT + 1;
               rtTmp[addr + i].Value2 = rtTmp[addr + i].Value1;
               rtTmp[addr + i].Value1 = registers[i];
            } else rtTmp[addr+i].Value2 = registers[i];
         }
      }

      private const byte RT_VALIDATE_LIMIT = 2;
      private const byte PS_VALIDATE_LIMIT = 2;

      #region INotifyPropertyChanged Members

      public event PropertyChangedEventHandler PropertyChanged;

      internal void OnPropertyChanged(string name) {
         if (PropertyChanged != null) PropertyChanged(this, new PropertyChangedEventArgs(name));
      }

      #endregion
      #endregion

      /// <summary>Converts all significant properties of <see cref="USBRadioDevice"/> to human readble string</summary>
      /// <returns>Human readble string</returns>
      public override string ToString() {
         return string.Format("RDS Info: [\n{0}\n{1}\n{2}\nMS:{3}, TA:{4}, TP:{5}\nAF:{6}\nPS:{7}\n]",
                     PIRegion,
                     PICountry,
                     PTYString,
                     HasMS,
                     HasTA,
                     HasTP,
                     AF != null ? string.Join(",", AF.ConvertAll(f => f.ToString()).ToArray()) : "N/A",
                     PS); 
      }
   }

   /// <summary>RDS group types for <see cref="RDSData"/></summary>
   public enum RDSGroupType : byte {
      // 0A, Basic tuning and switching info
      TYPE_0A = (0 * 2 + 0),
      // 0B, Basic tuning and switching info
      TYPE_0B = (0 * 2 + 1),
      // 1A, Program Item Number and Slow Labelling Codes
      TYPE_1A = (1 * 2 + 0),
      // 1B, Program Item Number and Slow Labelling Codes
      TYPE_1B = (1 * 2 + 1),
      // 2A, Radio Text
      TYPE_2A = (2 * 2 + 0),
      // 2B, Radio Text
      TYPE_2B = (2 * 2 + 1),
      TYPE_3A = (3 * 2 + 0),
      TYPE_3B = (3 * 2 + 1),
      TYPE_4A = (4 * 2 + 0),
      TYPE_4B = (4 * 2 + 1),
      TYPE_5A = (5 * 2 + 0),
      TYPE_5B = (5 * 2 + 1),
      TYPE_6A = (6 * 2 + 0),
      TYPE_6B = (6 * 2 + 1),
      TYPE_7A = (7 * 2 + 0),
      TYPE_7B = (8 * 2 + 0),
      // 8A, TMC
      TYPE_8A = (8 * 2 + 0),
      TYPE_8B = (8 * 2 + 1),
      TYPE_9A = (9 * 2 + 0),
      TYPE_9B = (9 * 2 + 1),
      // 10A, Programme Type Name
      TYPE_10A = (10 * 2 + 0),
      TYPE_10B = (10 * 2 + 1),
      TYPE_11A = (11 * 2 + 0),
      TYPE_11B = (11 * 2 + 1),
      TYPE_12A = (12 * 2 + 0),
      TYPE_12B = (12 * 2 + 1),
      TYPE_13A = (13 * 2 + 0),
      TYPE_13B = (13 * 2 + 1),
      // 14A, Enhanced Other Networks
      TYPE_14A = (14 * 2 + 0),
      // 14B, EON
      TYPE_14B = (14 * 2 + 1),
      TYPE_15A = (15 * 2 + 0),
      // 15B, Fast Basic Tuning
      TYPE_15B = (15 * 2 + 1),
   }

   /// <summary>Program information areas lookup table for <see cref="RDSData"/></summary>
   public class PIAreas : List<PIArea> {
      private PIAreas():base() {
         base.Add(new PIArea { Code = 0x0, Name = "Local" });
         base.Add(new PIArea { Code = 0x1, Name = "International"});
         base.Add(new PIArea { Code = 0x2, Name = "National"});
         base.Add(new PIArea { Code = 0x3, Name = "Supra-Regional"});
         base.Add(new PIArea { Code = 0x4, Name = "Region 1"});
         base.Add(new PIArea { Code = 0x5, Name = "Region 2"});
         base.Add(new PIArea { Code = 0x6, Name = "Region 3"});
         base.Add(new PIArea { Code = 0x7, Name = "Region 4"});
         base.Add(new PIArea { Code = 0x8, Name = "Region 5"});
         base.Add(new PIArea { Code = 0x9, Name = "Region 6"});
         base.Add(new PIArea { Code = 0xa, Name = "Region 7"});
         base.Add(new PIArea { Code = 0xb, Name = "Region 8"});
         base.Add(new PIArea { Code = 0xc, Name = "Region 9"});
         base.Add(new PIArea { Code = 0xd, Name = "Region 10"});
         base.Add(new PIArea { Code = 0xe, Name = "Region 11"});
         base.Add(new PIArea { Code = 0xf, Name = "Region 12"});
         base.Add(PIArea.Unknown);
      }
      /// <summary>Get all predefined <see cref="PIAreas"/>s</summary>
      public static PIAreas Areas = new PIAreas();

      /// <summary>Decodes byte country code and extended country code in <see cref="PIAreas"/> value</summary>
      /// <param name="code">Byte country code</param>
      /// <returns>String value of <see cref="PIAreas"/></returns>
      public static string DecodeName(byte code) {
         var area = Areas.FirstOrDefault(a => a.Code == code);
         if (area == default(PIArea)) return PIArea.Unknown.Name;
         else return area.Name;
      }
   }

   /// <summary>Program information area lookup item for <see cref="RDSData"/></summary>
   public class PIArea {
      /// <summary>PI Area Code</summary>
      public byte Code { get; internal set; }
      /// <summary>PI Area String Value</summary>
      public string Name { get; internal set; }
      /// <summary>Undefined PI area code value</summary>
      public static PIArea Unknown = new PIArea { Name = "Unknown", Code = byte.MinValue };
   }

   /// <summary>Program information countries lookup table for <see cref="RDSData"/></summary>
   public class PICountries : List<PICountry> {
      private PICountries() : base() {
         base.Add(new PICountry { CountryCode = 0x1, ExtendedCountryCode = 0xe0, Name = "Germany" });
         base.Add(new PICountry { CountryCode = 0x1, ExtendedCountryCode = 0xe1, Name = "Greece" });
         base.Add(new PICountry { CountryCode = 0x1, ExtendedCountryCode = 0xe2, Name = "Morocco" });
         base.Add(new PICountry { CountryCode = 0x1, ExtendedCountryCode = 0xe4, Name = "Moldova" });

         base.Add(new PICountry { CountryCode = 0x2, ExtendedCountryCode = 0xe0, Name = "Algeria" });
         base.Add(new PICountry { CountryCode = 0x2, ExtendedCountryCode = 0xe1, Name = "Cyprus" });
         base.Add(new PICountry { CountryCode = 0x2, ExtendedCountryCode = 0xe2, Name = "Czech Republic" });
         base.Add(new PICountry { CountryCode = 0x2, ExtendedCountryCode = 0xe3, Name = "Ireland" });
         base.Add(new PICountry { CountryCode = 0x2, ExtendedCountryCode = 0xe4, Name = "Estonia" });
         
         base.Add(new PICountry { CountryCode = 0x3, ExtendedCountryCode = 0xe0, Name = "Andorra" });
         base.Add(new PICountry { CountryCode = 0x3, ExtendedCountryCode = 0xe1, Name = "San Marino" });
         base.Add(new PICountry { CountryCode = 0x3, ExtendedCountryCode = 0xe2, Name = "Poland" });
         base.Add(new PICountry { CountryCode = 0x3, ExtendedCountryCode = 0xe3, Name = "Turkey" });
         
         base.Add(new PICountry { CountryCode = 0x4, ExtendedCountryCode = 0xe0, Name = "Israel" });
         base.Add(new PICountry { CountryCode = 0x4, ExtendedCountryCode = 0xe1, Name = "Switzerland" });
         base.Add(new PICountry { CountryCode = 0x4, ExtendedCountryCode = 0xe2, Name = "Vatican City" });
         base.Add(new PICountry { CountryCode = 0x4, ExtendedCountryCode = 0xe3, Name = "Macedonia" });

         base.Add(new PICountry { CountryCode = 0x5, ExtendedCountryCode = 0xe0, Name = "Italy" });
         base.Add(new PICountry { CountryCode = 0x5, ExtendedCountryCode = 0xe1, Name = "Jordan" });
         base.Add(new PICountry { CountryCode = 0x5, ExtendedCountryCode = 0xe2, Name = "Slovakia" });

         base.Add(new PICountry { CountryCode = 0x6, ExtendedCountryCode = 0xe0, Name = "Belgium" });
         base.Add(new PICountry { CountryCode = 0x6, ExtendedCountryCode = 0xe1, Name = "Finland" });
         base.Add(new PICountry { CountryCode = 0x6, ExtendedCountryCode = 0xe2, Name = "Syria" });
         base.Add(new PICountry { CountryCode = 0x6, ExtendedCountryCode = 0xe4, Name = "Ukraine" });

         base.Add(new PICountry { CountryCode = 0x7, ExtendedCountryCode = 0xe0, Name = "Russia" });
         base.Add(new PICountry { CountryCode = 0x7, ExtendedCountryCode = 0xe1, Name = "Luxembourg" });
         base.Add(new PICountry { CountryCode = 0x7, ExtendedCountryCode = 0xe2, Name = "Tunisia" });

         base.Add(new PICountry { CountryCode = 0x8, ExtendedCountryCode = 0xe0, Name = "Palestine" });
         base.Add(new PICountry { CountryCode = 0x8, ExtendedCountryCode = 0xe1, Name = "Bulgaria" });
         base.Add(new PICountry { CountryCode = 0x8, ExtendedCountryCode = 0xe2, Name = "Netherlands" });
         base.Add(new PICountry { CountryCode = 0x8, ExtendedCountryCode = 0xe3, Name = "Portugal" });

         base.Add(new PICountry { CountryCode = 0x9, ExtendedCountryCode = 0xe0, Name = "Albania" });
         base.Add(new PICountry { CountryCode = 0x9, ExtendedCountryCode = 0xe1, Name = "Denmark" });
         base.Add(new PICountry { CountryCode = 0x9, ExtendedCountryCode = 0xe2, Name = "Litchenstein" });
         base.Add(new PICountry { CountryCode = 0x9, ExtendedCountryCode = 0xe3, Name = "Latvia" });
         base.Add(new PICountry { CountryCode = 0x9, ExtendedCountryCode = 0xe4, Name = "Portugal" });

         base.Add(new PICountry { CountryCode = 0xa, ExtendedCountryCode = 0xe0, Name = "Austria" });
         base.Add(new PICountry { CountryCode = 0xa, ExtendedCountryCode = 0xe1, Name = "Gibralter" });
         base.Add(new PICountry { CountryCode = 0xa, ExtendedCountryCode = 0xe2, Name = "Iceland" });
         base.Add(new PICountry { CountryCode = 0xa, ExtendedCountryCode = 0xe3, Name = "Lebanon" });

         base.Add(new PICountry { CountryCode = 0xb, ExtendedCountryCode = 0xe0, Name = "Hungary" });
         base.Add(new PICountry { CountryCode = 0xb, ExtendedCountryCode = 0xe1, Name = "Iraq" });
         base.Add(new PICountry { CountryCode = 0xb, ExtendedCountryCode = 0xe2, Name = "Monaco" });

         base.Add(new PICountry { CountryCode = 0xc, ExtendedCountryCode = 0xe0, Name = "Malta" });
         base.Add(new PICountry { CountryCode = 0xc, ExtendedCountryCode = 0xe1, Name = "United Kingdom" });
         base.Add(new PICountry { CountryCode = 0xc, ExtendedCountryCode = 0xe2, Name = "Lithuania" });
         base.Add(new PICountry { CountryCode = 0xc, ExtendedCountryCode = 0xe3, Name = "Croatia" });

         base.Add(new PICountry { CountryCode = 0xd, ExtendedCountryCode = 0xe0, Name = "Germany" });
         base.Add(new PICountry { CountryCode = 0xd, ExtendedCountryCode = 0xe1, Name = "Libya" });
         base.Add(new PICountry { CountryCode = 0xd, ExtendedCountryCode = 0xe2, Name = "Yugoslavia" });

         base.Add(new PICountry { CountryCode = 0xe, ExtendedCountryCode = 0xe0, Name = "Romania" });
         base.Add(new PICountry { CountryCode = 0xe, ExtendedCountryCode = 0xe1, Name = "Spain" });
         base.Add(new PICountry { CountryCode = 0xe, ExtendedCountryCode = 0xe2, Name = "Sweden" });

         base.Add(new PICountry { CountryCode = 0xf, ExtendedCountryCode = 0xe0, Name = "Egypt" });
         base.Add(new PICountry { CountryCode = 0xf, ExtendedCountryCode = 0xe1, Name = "France" });
         base.Add(new PICountry { CountryCode = 0xf, ExtendedCountryCode = 0xe2, Name = "Norway" });
         base.Add(new PICountry { CountryCode = 0xf, ExtendedCountryCode = 0xe3, Name = "Belarus" });
         base.Add(new PICountry { CountryCode = 0xf, ExtendedCountryCode = 0xe4, Name = "Bosnia Herzegovina" });
      }

      /// <summary>Get all predefined <see cref="PICountry"/>s</summary>
      public static PICountries Countries = new PICountries();

      /// <summary>Decodes byte country code and extended country code in <see cref="PICountry"/> value</summary>
      /// <param name="countryCode">Byte country code</param>
      /// <param name="extendedCountryCode">Byte extended country code -1 if unused</param>
      /// <returns>String value of <see cref="PICountry"/></returns>
      public static string DecodeName(byte countryCode, byte extendedCountryCode) {
         var countries = Countries.Where(c => c.CountryCode == countryCode);
         if (countries.Count() <= 0) return PICountry.Unknown.Name;
         var country = countries.FirstOrDefault(c => c.ExtendedCountryCode == extendedCountryCode);
         if (country == default(PICountry)) return string.Join("/", countries.Select(c=>c.Name).ToArray());
         else return country.Name;
      }
   }

   /// <summary>Program information country lookup item for <see cref="RDSData"/></summary>
   public class PICountry {
      /// <summary>PI Country Code</summary>
      public byte CountryCode { get; internal set; }
      /// <summary>PI Country Extended Code</summary>
      public byte ExtendedCountryCode { get; internal set; }
      /// <summary>PI Country string value</summary>
      public string Name { get; internal set; }
      /// <summary>Undefined PI country code</summary>
      public static PICountry Unknown = new PICountry { Name = "Unknown", CountryCode = byte.MinValue, ExtendedCountryCode = byte.MinValue };
   }

   /// <summary>Program type lookup table for <see cref="RDSData"/></summary>
   public class PTYTexts : List<PTYText> {
      private PTYTexts()
         : base() {
         base.Add(new PTYText { Code = 1, Text = "News" });
         base.Add(new PTYText { Code = 2, Text = "Current Affairs" });
         base.Add(new PTYText { Code = 3, Text = "Information" });
         base.Add(new PTYText { Code = 4, Text = "Sport" });
         base.Add(new PTYText { Code = 5, Text = "Education" });
         base.Add(new PTYText { Code = 6, Text = "Drama" });
         base.Add(new PTYText { Code = 7, Text = "Culture" });
         base.Add(new PTYText { Code = 8, Text = "Science" });
         base.Add(new PTYText { Code = 9, Text = "Varied" });
         base.Add(new PTYText { Code = 10, Text = "Pop Music" });
         base.Add(new PTYText { Code = 11, Text = "Rock Music" });
         base.Add(new PTYText { Code = 12, Text = "Easy Listening" });
         base.Add(new PTYText { Code = 13, Text = "Light Classic Music" });
         base.Add(new PTYText { Code = 14, Text = "Serious Classic Music" });
         base.Add(new PTYText { Code = 15, Text = "Other Music" });
         base.Add(new PTYText { Code = 16, Text = "Weather & Metr" });
         base.Add(new PTYText { Code = 17, Text = "Finance" });
         base.Add(new PTYText { Code = 18, Text = "Children's Progs" });
         base.Add(new PTYText { Code = 19, Text = "Social Affairs" });
         base.Add(new PTYText { Code = 20, Text = "Religion" });
         base.Add(new PTYText { Code = 21, Text = "Phone In" });
         base.Add(new PTYText { Code = 22, Text = "Travel & Touring" });
         base.Add(new PTYText { Code = 23, Text = "Leisure & Hobby" });
         base.Add(new PTYText { Code = 24, Text = "Jazz Music" });
         base.Add(new PTYText { Code = 25, Text = "Country Music" });
         base.Add(new PTYText { Code = 26, Text = "National Music" });
         base.Add(new PTYText { Code = 27, Text = "Oldies Music" });
         base.Add(new PTYText { Code = 28, Text = "Folk Music" });
         base.Add(new PTYText { Code = 29, Text = "Documentary" });
         base.Add(new PTYText { Code = 30, Text = "Alarm Test" });
         base.Add(new PTYText { Code = 31, Text = "Alarm - Alarm !" });
      }

      public static PTYTexts Texts = new PTYTexts();
      public static string DecodeText(byte code) {
         var text = Texts.FirstOrDefault(t => t.Code == code);
         if (text != default(PTYText)) return text.Text;
         else return string.Empty;
      }
   }

   /// <summary>Program type lookup item for <see cref="RDSData"/></summary>
   public class PTYText {
      /// <summary>PTY Code</summary>
      public byte Code { get; internal set; }
      /// <summary>PTY value string</summary>
      public string Text { get; internal set; }
   }

}
