﻿/* Copyright (c) Tamir Khason (http://khason.net/). All rights reserved.
 * 
 * Released under MS-PL on CodePlex (http://www.codeplex.com/FM/)
 */

#region using
using System;
using System.Runtime.InteropServices;
using System.Security.Permissions;
#endregion

namespace USBFM {
   /// <summary>Extension methods to work with <see cref="USBDevice"/> and <see cref="USBRadioDevice"/> devices</summary>
   public static class USBRDSDeviceMethods {
      /// <summary>Tunes <see cref="USBRadioDevice"/> to <see cref="double"/>frequency</summary>
      /// <param name="device"><see cref="USBRadioDevice"/> to tune</param>
      /// <param name="frequency">Frequency to tune to</param>
      /// <returns>True if operation was performed</returns>
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      public static bool Tune(this USBRadioDevice device, double frequency) {
         var result = false;
         if (((device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_BAND) != 0 && (frequency >= 76.0) && (frequency <= 90.0)) ||
      ((device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_BAND) == 0 && (frequency >= 87.5) && (frequency <= 108.0))) {
            var channel = frequency.FromFrequency(device);
            device.Registers[Native.CHANNEL] = (ushort)(device.Registers[Native.CHANNEL] & ~Native.CHANNEL_CHAN);
            device.Registers[Native.CHANNEL] |= (ushort)(channel | Native.CHANNEL_TUNE);
            device.IsTuning = true;
            if (device.WriteRegisters(ReportType.Channel, device.Registers[Native.CHANNEL])) {
               var dt = DateTime.Now;
               bool status = false, error = false;
               while (!status && !error) {
                  var registers = device.ReadRegisters(ReportType.StatusRssi);
                  error = registers == default(ushort[]);
                  if(!error) status = ((registers[0] & Native.STATUSRSSI_STC) != 0);
                  error = DateTime.Now.Subtract(dt).TotalSeconds > Native.POLL_TIMEOUT_SECONDS;
               }

               if (status && !error) {
                  device.Registers[Native.CHANNEL] = (ushort)(device.Registers[Native.CHANNEL] & ~Native.CHANNEL_TUNE);

                  result = device.WriteRegisters(ReportType.Channel, device.Registers[Native.CHANNEL]);

                  var registers = device.ReadRegisters(ReportType.ReadChan);
                  if(registers != default(ushort[])) device.Registers[Native.READCHAN] = registers[0];
               }
            } else device.Registers[Native.CHANNEL] = (ushort)(device.Registers[Native.CHANNEL] & ~Native.CHANNEL_TUNE);
            device.IsTuning = false;
            device.CurrentFrequency = ((ushort)(device.Registers[Native.READCHAN] & Native.READCHAN_READCHAN)).ToFrequency(device);
            if (result) {
               result = device.WriteFMInfo();
               //device.ReadFMInfo();
               //device.CurrentFrequency;
            }
         }
         return result;
      }

      /// <summary>Tunes <see cref="USBRadioDevice"/> up or down</summary>
      /// <param name="device"><see cref="USBRadioDevice"/> to tune</param>
      /// <param name="tuneUp">True if tune up, else down</param>
      /// <returns>True if operation was performed</returns>
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      public static bool Tune(this USBRadioDevice device, bool tuneUp) {
         var channel = (ushort)(device.Registers[Native.READCHAN] & Native.READCHAN_READCHAN);
         if (tuneUp) {
            channel++;
            if (channel == Native.CHANNEL_CHAN + 1)
               channel = 0x0000;
         } else {
            channel--;
            if (channel == 0xFFFF)
               channel = Native.CHANNEL_CHAN;
         }
         while (channel.ToFrequency(device) > ((device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_BAND) !=0 ? 90.0 : 108.0)) {
            if (tuneUp) channel = 0x0000;
            else channel--;
         }
         return device.Tune(channel.ToFrequency(device));
      }

      /// <summary>Seeks <see cref="USBRadioDevice"/> for good frequency</summary>
      /// <param name="device"><see cref="USBRadioDevice"/> to seek</param>
      /// <param name="seekUp">True if seek up, else down</param>
      /// <returns>True if operation was performed</returns>
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      public static bool Seek(this USBRadioDevice device, bool seekUp) {
         var result = false;
         if (seekUp) device.Registers[Native.POWERCFG] |= Native.POWERCFG_SEEKUP;
         else device.Registers[Native.POWERCFG] = (ushort)(device.Registers[Native.POWERCFG] & ~Native.POWERCFG_SEEKUP);

         device.Registers[Native.POWERCFG] |= Native.POWERCFG_SEEK;
         device.StopAudio();
         device.IsTuning = true;

         if (device.WriteRegisters(ReportType.PowerCfg, device.Registers[Native.POWERCFG])) {
            var dt = DateTime.Now;
            bool status = false, error = false;
            while (!status && !error) {
               var registers = device.ReadRegisters(ReportType.StatusRssi);
               error = registers == default(ushort[]);
               if (!error) status = ((registers[0] & Native.STATUSRSSI_STC) != 0);
               error = DateTime.Now.Subtract(dt).TotalSeconds > Native.POLL_TIMEOUT_SECONDS;
            }
            if (status && !error) {
               device.Registers[Native.POWERCFG] = (ushort)(device.Registers[Native.POWERCFG] & ~Native.POWERCFG_SEEK);

               result = device.WriteRegisters(ReportType.PowerCfg, device.Registers[Native.POWERCFG]);

               var registers = device.ReadRegisters(ReportType.ReadChan);
               if (registers != default(ushort[])) device.Registers[Native.READCHAN] = registers[0];
            }
         } else device.Registers[Native.POWERCFG] = (ushort)(device.Registers[Native.POWERCFG] & ~Native.POWERCFG_SEEK);

         device.IsTuning = false;
         device.CurrentFrequency = ((ushort)(device.Registers[Native.READCHAN] & Native.READCHAN_READCHAN)).ToFrequency(device);
         device.InitAudio();
         return result;
      }

      /// <summary>Initializes Direct Sound echo thread on given <see cref="USBRadioDevice"/></summary>
      /// <param name="device"><see cref="USBRadioDevice"/> to init sound</param>
      public static void InitAudio(this USBRadioDevice device) { DirectSoundMethods.InitEchoService("radio"); }

      /// <summary>Stops Direct Sound echo thread on given <see cref="USBRadioDevice"/></summary>
      /// <param name="device"><see cref="USBRadioDevice"/> to stop sound</param>
      public static void StopAudio(this USBRadioDevice device) { DirectSoundMethods.StopEchoService(); }

      /// <summary>Mutes and unmutes sound on given <see cref="USBRadioDevice"/></summary>
      /// <param name="device"><see cref="USBRadioDevice"/> to mute</param>
      /// <param name="mute">True to mute, False to unmue</param>
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      public static void Mute(this USBRadioDevice device, bool mute) {
         if (!mute) device.Registers[Native.POWERCFG] |= Native.POWERCFG_DMUTE;
         else device.Registers[Native.POWERCFG] = (ushort)(device.Registers[Native.POWERCFG] & ~Native.POWERCFG_DMUTE);
         device.IsMuted = mute & device.WriteRegisters(ReportType.PowerCfg);
      }

      #region private
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal static bool InitRegisters(this USBDevice device) {
         var result = false;
         var registers = device.ReadRegisters(ReportType.Entire);
         result = registers != default(ushort[]);
         if (result) device.Registers = registers;
         return result;
      }

      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal static bool ReadFMInfo(this USBRadioDevice device) {
         var result = false;
         var scratch = device.ReadRegisters(ReportType.Scratch);
         result = scratch != default(ushort[]);
         if (result) {
            device.Capabilities = new RadioData {
               SoftwareVersion = scratch[1],
               HardwareVersion = scratch[2],
               PartNumber = (ushort)((device.Registers[Native.DEVICEID] & Native.DEVICEID_PN) >> 12),
               ManufacturerID = (ushort)(device.Registers[Native.DEVICEID] & Native.DEVICEID_MFGID),
               ChipVersion = (ushort)((device.Registers[Native.CHIPID] & Native.CHIPID_REV) >> 10),
               DeviceVersion = (ushort)((device.Registers[Native.CHIPID] & Native.CHIPID_DEV) >> 9),
               FirmwareVersion = (ushort)(device.Registers[Native.CHIPID] & Native.CHIPID_FIRMWARE),
               ExFlags = Native.FLAG_DEDUP,
               PI = new ushort[Native.PI_NUM],
               PI_Freq = new double[Native.PI_NUM],
               //EonPI = 49666 //BBC R2
            };
            if ((scratch[0] == 0x00)) {
               device.Presets = new double[] { 88, 89.1, 89.7, 90, 91, 92, 95, 97.2, 99, 100, 103, 104 };
               device.CurrentStation = 88;
               device.Capabilities.SeekThreshold = Native.PREFERRED_SEEK_THRESHOLD;
               device.Capabilities.Band = Native.DATA_BAND_875_108MHZ;
               device.Capabilities.Spacing = Native.DATA_SPACING_200KHZ;
               device.Capabilities.Deemphasis = Native.DATA_DEEMPHASIS_75;
               device.IsStereo = Native.DATA_MONOSTEREO_STEREO == 1;
               device.Capabilities.EonPI = 49666; //BBC R2

            } else {
               device.Presets = new double[] {
                  ((ushort)(((scratch[3] & 0xFF) << 2) | ((scratch[4] & 0xC0) >> 6))).ToFrequency(device),
                  ((ushort)(((scratch[4] & 0x3F) << 4) | ((scratch[5] & 0xF0) >> 4))).ToFrequency(device),
                  ((ushort)(((scratch[5] & 0x0F) << 6) | ((scratch[6] & 0xFC) >> 2))).ToFrequency(device),
                  ((ushort)(((scratch[6] & 0x03) << 8) | (scratch[7] & 0xFF))).ToFrequency(device),
                  ((ushort)(((scratch[8] & 0xFF) << 2) | ((scratch[9] & 0xC0) >> 6))).ToFrequency(device),
                  ((ushort)(((scratch[9] & 0x3F) << 4) | ((scratch[10] & 0xF0) >> 4))).ToFrequency(device),
                  ((ushort)(((scratch[10] & 0x0F) << 6) | ((scratch[11] & 0xFC) >> 2))).ToFrequency(device),
                  ((ushort)(((scratch[11] & 0x03) << 8) | (scratch[12] & 0xFF))).ToFrequency(device),
                  ((ushort)(((scratch[13] & 0xFF) << 2) | ((scratch[14] & 0xC0) >> 6))).ToFrequency(device),
                  ((ushort)(((scratch[14] & 0x3F) << 4) | ((scratch[15] & 0xF0) >> 4))).ToFrequency(device),
                  ((ushort)(((scratch[15] & 0x0F) << 6) | ((scratch[16] & 0xFC) >> 2))).ToFrequency(device),
                  ((ushort)(((scratch[16] & 0x03) << 8) | (scratch[17] & 0xFF))).ToFrequency(device)
            };
               device.CurrentStation = ((ushort)(((scratch[18] & 0xFF) << 2) | ((scratch[19] & 0xC0) >> 6))).ToFrequency(device);
               device.Capabilities.Band = (ushort)(scratch[19] & Native.DATA_BAND);
               device.Capabilities.Spacing = (ushort)(scratch[19] & Native.DATA_SPACING);
               if ((device.Capabilities.ExFlags & Native.FLAG_100Khz) != 0) device.Capabilities.Spacing = Native.DATA_SPACING_100KHZ;
               if ((device.Capabilities.ExFlags & Native.FLAG_200Khz) != 0) device.Capabilities.Spacing = Native.DATA_SPACING_200KHZ;
               device.Capabilities.Deemphasis = (ushort)(scratch[19] & Native.DATA_DEEMPHASIS);
               device.IsStereo = (scratch[19] & Native.DATA_MONOSTEREO) != 0;
               device.Capabilities.SeekThreshold = scratch[20] == 0 ? Native.PREFERRED_SEEK_THRESHOLD : scratch[20];

               //store PI
               var data = 22;
               var topBit = (ushort)(((scratch[(Native.FLASH_PI_NUM * 3) + 22] & 0xff) << 8) + ((scratch[(Native.FLASH_PI_NUM * 3) + 23] & 0xff) << 8));
               ushort PI, Freq;
               device.Capabilities.PI = new ushort[Native.FLASH_PI_NUM];
               device.Capabilities.PI_Freq = new double[Native.FLASH_PI_NUM];
               for (byte i = 0; i < Native.FLASH_PI_NUM; i++) {
                  PI = (ushort)(((scratch[data++]) & 0xff) << 8);
                  PI += (ushort)(scratch[data++] & 0xff);
                  Freq = (ushort)(((topBit >> i) << 8) + scratch[data++]);
                  if ((Freq != 0) && (Freq != 0x1ff) && (PI != 0) && (PI != 0xffff)) {
                     device.Capabilities.PI[i] = PI;
                     device.Capabilities.PI_Freq[i] = Freq.ToFrequency(device);
                  }
               }
               device.Capabilities.EonPI = (ushort)((scratch[60] << 8) & 0xff00);
               device.Capabilities.EonPI += (ushort)(scratch[61] & 0xff);
            }
         }
         return result;
      }

      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal static bool WriteFMInfo(this USBRadioDevice device) {
         var result = false;
         var scratch = new ushort[Native.SCRATCH_PAGE_SIZE];
         if (device.Capabilities.Band == Native.DATA_BAND_875_108MHZ) {
            if ((device.CurrentFrequency > 108.0) || (device.CurrentFrequency < 87.5))
               device.CurrentFrequency = 100.1;
         } else {
            if ((device.CurrentFrequency > 90.0) || (device.CurrentFrequency < 76.0))
               device.CurrentFrequency = 82.0;
         }
         scratch[0] |= 1;
         scratch[1] |= device.Capabilities.SoftwareVersion;
         scratch[2] |= device.Capabilities.HardwareVersion;
         scratch[3] |= (ushort)((device.Presets[0].FromFrequency(device) >> 2) & 0xFF);
         scratch[4] |= (ushort)((device.Presets[0].FromFrequency(device) << 6) & 0xC0);
         scratch[4] |= (ushort)((device.Presets[1].FromFrequency(device) >> 4) & 0x3F);
         scratch[5] |= (ushort)((device.Presets[1].FromFrequency(device) << 4) & 0xF0);
         scratch[5] |= (ushort)((device.Presets[2].FromFrequency(device) >> 6) & 0x0F);
         scratch[6] |= (ushort)((device.Presets[2].FromFrequency(device) << 2) & 0xFC);
         scratch[6] |= (ushort)((device.Presets[3].FromFrequency(device) >> 8) & 0x03);
         scratch[7] |= (ushort)(device.Presets[3].FromFrequency(device) & 0xFF);
         scratch[8] |= (ushort)((device.Presets[4].FromFrequency(device) >> 2) & 0xFF);
         scratch[9] |= (ushort)((device.Presets[4].FromFrequency(device) << 6) & 0xC0);
         scratch[9] |= (ushort)((device.Presets[5].FromFrequency(device) >> 4) & 0x3F);
         scratch[10] |= (ushort)((device.Presets[5].FromFrequency(device) << 4) & 0xF0);
         scratch[10] |= (ushort)((device.Presets[6].FromFrequency(device) >> 6) & 0x0F);
         scratch[11] |= (ushort)((device.Presets[6].FromFrequency(device) << 2) & 0xFC);
         scratch[11] |= (ushort)((device.Presets[7].FromFrequency(device) >> 8) & 0x03);
         scratch[12] |= (ushort)(device.Presets[7].FromFrequency(device) & 0xFF);
         scratch[13] |= (ushort)((device.Presets[8].FromFrequency(device) >> 2) & 0xFF);
         scratch[14] |= (ushort)((device.Presets[8].FromFrequency(device) << 6) & 0xC0);
         scratch[14] |= (ushort)((device.Presets[9].FromFrequency(device) >> 4) & 0x3F);
         scratch[15] |= (ushort)((device.Presets[9].FromFrequency(device) << 4) & 0xF0);
         scratch[15] |= (ushort)((device.Presets[10].FromFrequency(device) >> 6) & 0x0F);
         scratch[16] |= (ushort)((device.Presets[10].FromFrequency(device) << 2) & 0xFC);
         scratch[16] |= (ushort)((device.Presets[11].FromFrequency(device) >> 8) & 0x03);
         scratch[17] |= (ushort)(device.Presets[11].FromFrequency(device) & 0xFF);
         scratch[18] |= (ushort)((device.CurrentFrequency.FromFrequency(device) >> 2) & 0xFF);
         scratch[19] |= (ushort)(((device.CurrentFrequency.FromFrequency(device) << 6) & 0xC0) | device.Capabilities.Band | (device.IsStereo ? 1 : 0) | device.Capabilities.Spacing | device.Capabilities.Deemphasis);
         scratch[20] |= device.Capabilities.SeekThreshold;

         var k = 22;
         ushort topBit = 0;
         for (int i = 0; i < Native.FLASH_PI_NUM; i++) {
            scratch[k++] = (ushort)((device.Capabilities.PI[i] >> 8) & 0xff);
            scratch[k++] = (ushort)(device.Capabilities.PI[i] & 0xff);
            scratch[k++] = (ushort)(device.Capabilities.PI_Freq[i].FromFrequency(device) & 0xff);
            topBit |= (ushort)(((device.Capabilities.PI_Freq[i].FromFrequency(device) & 0x100) >> 8) << i);
         }
         scratch[(Native.FLASH_PI_NUM * 3) + 22] = (ushort)((topBit >> 8) & 0xff);
         scratch[(Native.FLASH_PI_NUM * 3) + 23] = (ushort)(topBit & 0xff);

         scratch[60] = (ushort)((device.Capabilities.EonPI >> 8) & 0xff);
         scratch[61] = (ushort)(device.Capabilities.EonPI & 0xff);

         //checksum
         ushort cSum = 0;
         for (int i = 0; i < (Native.SCRATCH_PAGE_SIZE - 1); i++) cSum += scratch[i];

         cSum = (ushort)(cSum & 0xff);
         cSum = (ushort)(~cSum + 1);
         cSum = (ushort)(cSum & 0xff);

         scratch[Native.SCRATCH_PAGE_SIZE - 1] = cSum;


         result = device.WriteRegisters(ReportType.Scratch, scratch);
         return result;
      }

      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal static bool InitFMInfo(this USBRadioDevice device) {
         var result = false;
         if (device.Capabilities != default(RadioData)) {

            device.Registers[Native.SYSCONFIG1] |= Native.SYSCONFIG1_RDS;
            switch (device.Capabilities.Band & Native.DATA_BAND) {
               case Native.DATA_BAND_875_108MHZ: device.Registers[Native.SYSCONFIG2] = (ushort)(device.Registers[Native.SYSCONFIG2] & ~Native.SYSCONFIG2_BAND); break;
               case Native.DATA_BAND_76_90MHZ: device.Registers[Native.SYSCONFIG2] |= Native.SYSCONFIG2_BAND; break;
               default: device.Registers[Native.SYSCONFIG2] = (ushort)(device.Registers[Native.SYSCONFIG2] & ~Native.SYSCONFIG2_BAND); break;
            }

            device.Registers[Native.SYSCONFIG2] = (ushort)(device.Registers[Native.SYSCONFIG2] & ~Native.SYSCONFIG2_SPACE);
            switch (device.Capabilities.Spacing & Native.DATA_SPACING) {
               case Native.DATA_SPACING_200KHZ: device.Registers[Native.SYSCONFIG2] |= Native.SYSCONFIG2_SPACE_200KHZ; break;
               case Native.DATA_SPACING_100KHZ: device.Registers[Native.SYSCONFIG2] |= Native.SYSCONFIG2_SPACE_100KHZ; break;
               case Native.DATA_SPACING_50KHZ: device.Registers[Native.SYSCONFIG2] |= Native.SYSCONFIG2_SPACE_50KHZ; break;
               default: device.Registers[Native.SYSCONFIG2] |= Native.SYSCONFIG2_SPACE_200KHZ; break;
            }

            switch (device.Capabilities.Deemphasis & Native.DATA_DEEMPHASIS) {
               case Native.DATA_DEEMPHASIS_75: device.Registers[Native.SYSCONFIG1] = (ushort)(device.Registers[Native.SYSCONFIG1] & ~Native.SYSCONFIG1_DE); break;
               case Native.DATA_DEEMPHASIS_50: device.Registers[Native.SYSCONFIG1] |= Native.SYSCONFIG1_DE; break;
               default: device.Registers[Native.SYSCONFIG1] = (ushort)(device.Registers[Native.SYSCONFIG1] & ~Native.SYSCONFIG1_DE); break;
            }

            switch (device.IsStereo ? 1 : 0 & Native.DATA_MONOSTEREO) {
               case Native.DATA_MONOSTEREO_STEREO: device.Registers[Native.POWERCFG] &= (ushort)(device.Registers[Native.POWERCFG] & ~Native.POWERCFG_MONO); break;
               case Native.DATA_MONOSTEREO_MONO: device.Registers[Native.POWERCFG] |= Native.POWERCFG_MONO; break;
               default: device.Registers[Native.POWERCFG] = (ushort)(device.Registers[Native.POWERCFG] & ~Native.POWERCFG_MONO); break;
            }

            device.Registers[Native.SYSCONFIG2] = (ushort)(device.Registers[Native.SYSCONFIG2] & ~Native.SYSCONFIG2_SEEKTH);
            device.Registers[Native.SYSCONFIG2] |= (ushort)(device.Capabilities.SeekThreshold << 8);

            result = device.WriteRegisters(ReportType.SysConfig1, device.Registers[Native.SYSCONFIG1]);
            result &= device.WriteRegisters(ReportType.SysConfig2, device.Registers[Native.SYSCONFIG2]);
            result &= device.WriteRegisters(ReportType.PowerCfg, device.Registers[Native.POWERCFG]);

         }
         return result;
      }
      
      internal static double ToFrequency(this ushort channel, USBRadioDevice device) {
         double frequency = 0;

         double band = 87.5, spacing = 0.2;

         //Determine the band and spacing
         band = (device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_BAND) != 0 ? 76 : 87.5;

         switch (device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_SPACE) {
            case Native.SYSCONFIG2_SPACE_200KHZ: spacing = 0.2; break;
            case Native.SYSCONFIG2_SPACE_100KHZ: spacing = 0.1; break;
            case Native.SYSCONFIG2_SPACE_50KHZ: spacing = 0.05; break;
         }

         //Calculate the frequency and add .0001 to round up numbers not quite close enough to the frequency
         frequency = (int)(((band + (spacing * channel)) + .0001) * 100.0) / 100.0;

         return frequency;
      }
      
      internal static ushort FromFrequency(this double frequency, USBRadioDevice device) {
         double band = (device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_BAND) != 0 ? 76 : 87.5, spacing = 0.2;
         switch (device.Registers[Native.SYSCONFIG2] & Native.SYSCONFIG2_SPACE) {
            case Native.SYSCONFIG2_SPACE_200KHZ: spacing = 0.2; break;
            case Native.SYSCONFIG2_SPACE_100KHZ: spacing = 0.1; break;
            case Native.SYSCONFIG2_SPACE_50KHZ: spacing = 0.05; break;
         }
         return (ushort)(((frequency - band) / spacing) + .0001);
      }
      
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal static bool WriteRegisters(this USBDevice device, ReportType report, params ushort[] data) {
         var result = false;
         if (!device.IsInvalid) {
            var buffer = new byte[device.FeatureReportLength];
            buffer[0] = (byte)report;               
            if (data.Length <= ((device.FeatureReportLength - 1) / Native.FMRADIO_REGISTER_SIZE)) {
               for (byte i = 0; i < data.Length; i++) {
                  buffer[(i * 2) + 1] = (byte)((data[i] & 0xFF00) >> 8);
                  buffer[(i * 2) + 2] = (byte)(data[i] & 0x00FF);
               }               
            } else if (data.Length <= (device.FeatureReportLength - 1)) {
               for (byte i = 0; i < data.Length; i++) {
                  buffer[i + 1] = (byte)data[i];
               }               
            }
            var hEPBuffer = Marshal.AllocHGlobal(device.FeatureReportLength);
            Marshal.Copy(buffer, 0, hEPBuffer, device.FeatureReportLength);
            result = USBDevice.Native.HidD_SetFeature(device.DangerousGetHandle(), hEPBuffer, (uint)device.FeatureReportLength);
            Marshal.FreeHGlobal(hEPBuffer);
         }
         return result;
      }
      
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal static ushort[] ReadRegisters(this USBDevice device, ReportType report) {
         var result = default(ushort[]);
         if (!device.IsInvalid) {
            var buffer = new byte[device.FeatureReportLength];
            buffer[0] = (byte)report;
            var reportLength = 0;
            switch (report) {
               case ReportType.StatusRssi:
               case ReportType.ReadChan:
                  reportLength = 1;
                  break;
               case ReportType.RDS: reportLength = Native.RDS_REGISTER_NUM; break;
               case ReportType.Entire: reportLength = Native.FMRADIO_REGISTER_NUM; break;
               case ReportType.Scratch: reportLength = Native.SCRATCH_PAGE_SIZE; break;
            }
            result = new ushort[reportLength];
            if (reportLength <= (device.FeatureReportLength - 1) / Native.FMRADIO_REGISTER_SIZE) {
               if (report >= ReportType.DeviceID && report <= ReportType.Entire) {
                  var hEPBuffer = Marshal.AllocHGlobal(buffer.Length);
                  Marshal.Copy(buffer, 0, hEPBuffer, buffer.Length);
                  if (USBDevice.Native.HidD_GetFeature(device.DangerousGetHandle(), hEPBuffer, (uint)buffer.Length)) {
                     Marshal.Copy(hEPBuffer, buffer, 0, buffer.Length);
                     for (byte i = 0; i < result.Length; i++) {
                        result[i] = (ushort)((buffer[(i * 2) + 1] << 8) | buffer[(i * 2) + 2]);
                     }
                  }
                  Marshal.FreeHGlobal(hEPBuffer);
               } else if (report == ReportType.RDS) {
                  buffer = new byte[Native.RDS_REPORT_SIZE];
                  buffer[0] = (byte)ReportType.RDS;
                  uint bytesRead;
                  Native.OVERLAPPED o = new Native.OVERLAPPED();
                  o.hEvent = Native.CreateEventW(IntPtr.Zero, false, false, string.Empty);
                  IntPtr hOverlapped = Marshal.AllocHGlobal(Marshal.SizeOf(o));
                  Marshal.StructureToPtr(o, hOverlapped, true);
                  if (!Native.ReadFile(device.DangerousGetHandle(), buffer, (uint)buffer.Length, out bytesRead, hOverlapped)) {
                     var error = Marshal.GetLastWin32Error();
                     if (error == Native.ERROR_IO_PENDING) {
                        if (Native.WaitForSingleObject(o.hEvent, 3000)) {
                           Native.GetOverlappedResult(device.DangerousGetHandle(), ref o, out bytesRead, false);
                        }
                     }
                  }
                  Marshal.FreeHGlobal(hOverlapped);
                  for (byte i = 0; i < result.Length; i++) {
                     result[i] = (ushort)((buffer[(i * 2) + 1] << 8) | buffer[(i * 2) + 2]);
                  }
               }
            } else if (reportLength <= (device.FeatureReportLength - 1)) {
               var hEPBuffer = Marshal.AllocHGlobal(device.FeatureReportLength);
               Marshal.Copy(buffer, 0, hEPBuffer, buffer.Length);
               if (USBDevice.Native.HidD_GetFeature(device.DangerousGetHandle(), hEPBuffer, device.FeatureReportLength)) {
                  Marshal.Copy(hEPBuffer, buffer, 0, buffer.Length);
                  for (byte i = 0; i < result.Length; i++) {
                     result[i] = (ushort)buffer[i + 1];
                  }
               }
               Marshal.FreeHGlobal(hEPBuffer);
            }
         }
         return result;
      }

      internal static class Native {
         #region consts
         public const byte DEVICEID = 0;
         public const byte CHIPID = 1;
         public const byte POWERCFG = 2;
         public const byte CHANNEL = 3;
         public const byte SYSCONFIG1 = 4;
         public const byte SYSCONFIG2 = 5;
         public const byte SYSCONFIG3 = 6;
         public const byte TEST1 = 7;
         public const byte TEST2 = 8;
         public const byte BOOTCONFIG = 9;
         public const byte STATUSRSSI = 10;
         public const byte READCHAN = 11;
         public const byte RDSA = 12;
         public const byte RDSB = 13;
         public const byte RDSC = 14;
         public const byte RDSD = 15;
         public const byte FMRADIO_REGISTER_NUM = 16;
         public const byte FMRADIO_REGISTER_SIZE = 2;//WORD
         public const byte RDS_REGISTER_NUM = 6; 
         public const byte RDS_REPORT_SIZE = ((RDS_REGISTER_NUM * FMRADIO_REGISTER_SIZE) + 1);         

         public const ushort SYSCONFIG2_BAND = 0x0080;
         public const ushort SYSCONFIG2_SPACE = 0x0030;
         public const ushort SYSCONFIG2_SPACE_200KHZ = 0x0000;
         public const ushort SYSCONFIG2_SPACE_100KHZ = 0x0010;
         public const ushort SYSCONFIG2_SPACE_50KHZ = 0x0020;
         public const ushort SYSCONFIG1_RDS = 0x1000;

         public const ushort DATA_BAND = 0x20;
         public const ushort DATA_BAND_875_108MHZ = 0x00;
         public const ushort DATA_BAND_76_90MHZ = 0x20;

         public const ushort DATA_SPACING = 0x0c;
         public const ushort DATA_SPACING_50KHZ = 0x08;
         public const ushort DATA_SPACING_100KHZ = 0x04;
         public const ushort DATA_SPACING_200KHZ = 0x00;
         
         public const ushort DATA_DEEMPHASIS = 0x02;
         public const ushort DATA_DEEMPHASIS_75 = 0x00;
         public const ushort DATA_DEEMPHASIS_50 = 0x02;
         public const ushort SYSCONFIG1_DE = 0x0800;
         public const ushort SYSCONFIG2_SEEKTH = 0xFF00;
         
         public const ushort PREFERRED_SEEK_THRESHOLD = 25;

         public const byte DATA_MONOSTEREO = 0x10;
         public const byte DATA_MONOSTEREO_STEREO = 0x01;
         public const byte DATA_MONOSTEREO_MONO = 0x10;

         public const ushort POWERCFG_MONO = 0x2000;

         public const byte FLAG_DEDUP = 0x01;
         public const byte FLAG_100Khz = 0x02;
         public const byte FLAG_200Khz = 0x04;
         public const byte FLAG_MUTESTREAM = 0x08;

         public const ushort DEVICEID_PN = 0xF000;
         public const ushort DEVICEID_MFGID = 0x0FFF;
         public const ushort CHIPID_REV = 0xFC00;
         public const ushort CHIPID_DEV = 0x0200;
         public const ushort CHIPID_FIRMWARE = 0x01FF;

         public const ushort SCRATCH_PAGE_SIZE = 63;
         public const ushort SCRATCH_PAGE_USED_SIZE = 22;

         public const ushort CHANNEL_CHAN = 0x03FF;
         public const ushort CHANNEL_TUNE = 0x8000;

         public const ushort STATUSRSSI_STC = 0x4000;
         public const ushort POLL_TIMEOUT_SECONDS = 3;

         public const ushort READCHAN_READCHAN = 0x03FF;
         public const ushort STATUSRSSI_RSSI = 0x00FF;
         public const ushort STATUSRSSI_ST = 0x0100;

         public const ushort POWERCFG_DMUTE = 0x4000;
         public const byte PI_NUM = 30;
         public const byte FLASH_PI_NUM = 12;

         public const ushort POWERCFG_SEEKUP = 0x0200;
         public const ushort POWERCFG_SEEK = 0x0100;

         public const int ERROR_IO_PENDING = 997;
         #endregion

         #region methods
         [DllImport("kernel32.dll", EntryPoint = "ReadFile")]
         [return: MarshalAs(UnmanagedType.Bool)]
         public static extern bool ReadFile(
            IntPtr hFile,
            byte[] lpBuffer,
            uint nNumberOfBytesToRead,
            out uint lpNumberOfBytesRead,
            IntPtr lpOverlapped);

         [DllImport("kernel32.dll", EntryPoint="WaitForSingleObject")]
         public static extern bool WaitForSingleObject(
            [In] IntPtr hHandle, 
            uint dwMilliseconds);

         [DllImport("kernel32.dll", EntryPoint = "GetOverlappedResult")]
         [return: MarshalAs(UnmanagedType.Bool)]
         public static extern bool GetOverlappedResult(
            [In] System.IntPtr hFile, 
            [In] ref OVERLAPPED lpOverlapped, 
            [Out] out uint lpNumberOfBytesTransferred, 
            [MarshalAs(UnmanagedType.Bool)] bool bWait);

         [DllImport("kernel32.dll", EntryPoint = "CreateEventW")]
         public static extern IntPtr CreateEventW(
            [In] IntPtr lpEventAttributes, 
            [MarshalAs(UnmanagedType.Bool)] bool bManualReset, 
            [MarshalAs(UnmanagedType.Bool)] bool bInitialState, 
            [In] [MarshalAs(UnmanagedType.LPWStr)] string lpName);
         #endregion

         #region structs
         [StructLayout(LayoutKind.Sequential)]
         public struct OVERLAPPED {
            public uint Internal;
            public uint InternalHigh;
            public uint Offset;
            public uint OffsetHigh;
            public IntPtr hEvent;
         }

         [StructLayout(LayoutKind.Sequential)]
         public struct SECURITY_ATTRIBUTES {
            public uint nLength;
            public IntPtr lpSecurityDescriptor;
            [MarshalAs(UnmanagedType.Bool)]
            public bool bInheritHandle;
         }
         #endregion
      }
      #endregion
   }

   /// <summary>RDS report type</summary>
   public enum ReportType : byte {
      DeviceID = USBRDSDeviceMethods.Native.DEVICEID + 1,
      ChipID = USBRDSDeviceMethods.Native.CHIPID + 1,
      PowerCfg = USBRDSDeviceMethods.Native.POWERCFG + 1,
      Channel = USBRDSDeviceMethods.Native.CHANNEL + 1,
      SysConfig1 = USBRDSDeviceMethods.Native.SYSCONFIG1 + 1,
      SysConfig2 = USBRDSDeviceMethods.Native.SYSCONFIG2 + 1,
      SysConfig3 = USBRDSDeviceMethods.Native.SYSCONFIG3 + 1,
      Test1 = USBRDSDeviceMethods.Native.TEST1 + 1,
      Test2 = USBRDSDeviceMethods.Native.TEST2 + 1,
      BootConfig = USBRDSDeviceMethods.Native.BOOTCONFIG + 1,
      StatusRssi = USBRDSDeviceMethods.Native.STATUSRSSI + 1,
      ReadChan = USBRDSDeviceMethods.Native.READCHAN + 1,
      RDS_A = USBRDSDeviceMethods.Native.RDSA + 1,
      RDS_B = USBRDSDeviceMethods.Native.RDSB + 1,
      RDS_C = USBRDSDeviceMethods.Native.RDSC + 1,
      RDS_D = USBRDSDeviceMethods.Native.RDSD + 1,
      Led = 19,
      Stream = 19,
      Scratch = 20,
      Entire = (byte)USBRDSDeviceMethods.Native.FMRADIO_REGISTER_NUM + 1,
      RDS = (byte)USBRDSDeviceMethods.Native.FMRADIO_REGISTER_NUM + 2,
   }
}
