﻿/* Copyright (c) Tamir Khason (http://khason.net/). All rights reserved.
 * 
 * Released under MS-PL on CodePlex (http://www.codeplex.com/FM/)
 */

#region using
using System;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;
using System.Security;
using System.Security.Permissions;
using Microsoft.Win32.SafeHandles;
#endregion

namespace USBFM {
   /// <summary>General interfaces for Human Input Device</summary>
   /// <example>
   /// <code>
   /// using (var device = USBDevice.FindDevice(0x0000, 0x1111)) {
   ///  if(!device.IsInvalid) Console.WriteLine("Device {0} was found!",device);
   ///  else Console.WriteLine("Device was not found");
   /// }
   /// </code>
   /// </example>
   [SecurityPermission(SecurityAction.InheritanceDemand, UnmanagedCode = true)]
   [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
   public class USBDevice : SafeHandleZeroOrMinusOneIsInvalid, IDisposable {
      #region props
      /// <summary>Product Id of <see cref="USBDevice"/></summary>
      public uint ProductID { get; private set; }
      /// <summary>Vendor Id of <see cref="USBDevice"/></summary>
      public uint VendorID { get; private set; }
      /// <summary>Firmware version number installed on <see cref="USBDevice"/></summary>
      public uint VersionNumber { get; private set; }
      /// <summary>Name of <see cref="USBDevice"/></summary>
      public string Name { get; private set; }
      /// <summary>Serial number of <see cref="USBDevice"/></summary>
      public string SerialNumber { get; private set; }
      /// <summary>Flag to detect the current state of <see cref="USBDevice"/></summary>
      public override bool IsInvalid { get { return !isValid; } }

      internal ushort FeatureReportLength { get; private set; }
      internal ushort[] Registers { get; set; }
      #endregion

      #region ctor
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      internal USBDevice(uint pid, uint vid) : base(true) { findDevice(pid, vid); }
      #endregion

      #region private
      private Guid _hidGuid = Guid.Empty;
      private IntPtr hHidDeviceInfo;
      private bool isValid = false;
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      private void findDevice(uint pid, uint vid) {
         Native.HidD_GetHidGuid(ref _hidGuid);
         hHidDeviceInfo = Native.SetupDiGetClassDevs(ref _hidGuid, null, IntPtr.Zero, Native.DIGCF_PRESENT | Native.DIGCF_DEVICEINTERFACE);
         if (hHidDeviceInfo.ToInt32() > -1) {
            uint i = 0;
            while (!isValid && i < Native.MAX_USB_DEVICES) {
               var hidDeviceInterfaceData = new Native.SP_DEVICE_INTERFACE_DATA();
               hidDeviceInterfaceData.cbSize = Marshal.SizeOf(hidDeviceInterfaceData);
               if (Native.SetupDiEnumDeviceInterfaces(hHidDeviceInfo, 0, ref _hidGuid, i, ref hidDeviceInterfaceData)) {
                  bool detailResult;
                  uint length, required;
                  Native.SetupDiGetDeviceInterfaceDetail(hHidDeviceInfo, ref hidDeviceInterfaceData, IntPtr.Zero, 0, out length, IntPtr.Zero);
                  var hidDeviceInterfaceDetailData = new Native.PSP_DEVICE_INTERFACE_DETAIL_DATA();
                  hidDeviceInterfaceDetailData.cbSize = 5; //DWORD cbSize (size 4) + Char[0] (size 1) for 32bit only!
                  var hDeviceInterfaceDetailData = Marshal.AllocHGlobal(Marshal.SizeOf(hidDeviceInterfaceDetailData));
                  Marshal.StructureToPtr(hidDeviceInterfaceDetailData, hDeviceInterfaceDetailData, true);
                  detailResult = Native.SetupDiGetDeviceInterfaceDetail(hHidDeviceInfo, ref hidDeviceInterfaceData, hDeviceInterfaceDetailData, length, out required, IntPtr.Zero);
                  Marshal.PtrToStructure(hDeviceInterfaceDetailData, hidDeviceInterfaceDetailData);
                  if (detailResult) {
                     base.handle = Native.CreateFile(hidDeviceInterfaceDetailData.DevicePath, 
                        Native.GENERIC_READ |
                        Native.GENERIC_WRITE,
                        Native.FILE_SHARE_READ |
                        Native.FILE_SHARE_WRITE, 
                        IntPtr.Zero,
                        Native.OPEN_EXISTING,
                        Native.FILE_FLAG_OVERLAPPED, 
                        IntPtr.Zero);
                     if (base.handle.ToInt32() > -1) {
                        Native.HIDD_ATTRIBUTES hidDeviceAttributes;
                        if (Native.HidD_GetAttributes(base.handle, out hidDeviceAttributes)) {
                           if ((hidDeviceAttributes.VendorID == vid) && (hidDeviceAttributes.ProductID == pid)) {
                              isValid = true;
                              ProductID = pid;
                              VendorID = vid;
                              VersionNumber = hidDeviceAttributes.VersionNumber;
                              IntPtr buffer = Marshal.AllocHGlobal(126);//max alloc for string;
                              if (Native.HidD_GetProductString(this.handle, buffer, 126)) Name = Marshal.PtrToStringAuto(buffer);
                              if (Native.HidD_GetSerialNumberString(this.handle, buffer, 126)) SerialNumber = Marshal.PtrToStringAuto(buffer);
                              Marshal.FreeHGlobal(buffer);
                              var capabilities = new Native.HIDP_CAPS();
                              IntPtr hPreparsedData;
                              if (Native.HidD_GetPreparsedData(this.handle, out hPreparsedData)) {
                                 if (Native.HidP_GetCaps(hPreparsedData, out capabilities)) FeatureReportLength = capabilities.FeatureReportByteLength;
                                 Native.HidD_FreePreparsedData(hPreparsedData);
                              }
                              break;
                           }
                        } else {
                           Native.CloseHandle(base.handle);
                        }
                     }
                  }
                  Marshal.FreeHGlobal(hDeviceInterfaceDetailData);
               }
               i++;

            }
         }
      }

      #region p/invoke
      [SuppressUnmanagedCodeSecurity()]
      internal static class Native {
         #region methods
         [DllImport("hid.dll", SetLastError = true)]
         internal static extern void HidD_GetHidGuid(
            ref Guid lpHidGuid);

         [DllImport("hid.dll", SetLastError = true)]
         internal static extern bool HidD_GetAttributes(
            IntPtr hDevice,
            out HIDD_ATTRIBUTES Attributes);

         [DllImport("hid.dll", SetLastError = true)]
         internal static extern bool HidD_GetPreparsedData(
            IntPtr hDevice,
            out IntPtr hData);

         [DllImport("hid.dll", SetLastError = true)]
         internal static extern bool HidD_FreePreparsedData(
            IntPtr hData);

         [DllImport("hid.dll", SetLastError = true)]
         internal static extern bool HidP_GetCaps(
            IntPtr hData,
            out HIDP_CAPS capabilities);

         [DllImport("hid.dll", SetLastError = true, CallingConvention = CallingConvention.StdCall)]
         internal static extern bool HidD_GetFeature(
            IntPtr hDevice,
            IntPtr hReportBuffer,
            uint ReportBufferLength);

         [DllImport("hid.dll", SetLastError = true, CallingConvention = CallingConvention.StdCall)]
         internal static extern bool HidD_SetFeature(
            IntPtr hDevice,
            IntPtr ReportBuffer,
            uint ReportBufferLength);

         [DllImport("hid.dll", SetLastError = true, CallingConvention = CallingConvention.StdCall)]
         internal static extern bool HidD_GetProductString(
            IntPtr hDevice, 
            IntPtr Buffer, 
            uint BufferLength);

         [DllImport("hid.dll", SetLastError = true, CallingConvention = CallingConvention.StdCall)]
         internal static extern bool HidD_GetSerialNumberString(
            IntPtr hDevice,
            IntPtr Buffer,
            uint BufferLength);

         [DllImport("setupapi.dll", SetLastError = true)]
         internal static extern IntPtr SetupDiGetClassDevs(
            ref Guid ClassGuid,
            [MarshalAs(UnmanagedType.LPTStr)] string Enumerator,
            IntPtr hwndParent,
            UInt32 Flags);

         [DllImport("setupapi.dll", SetLastError = true)]
         internal static extern bool SetupDiEnumDeviceInterfaces(
            IntPtr DeviceInfoSet,
            int DeviceInfoData,
            ref  Guid lpHidGuid,
            uint MemberIndex,
            ref  SP_DEVICE_INTERFACE_DATA lpDeviceInterfaceData);

         [DllImport("setupapi.dll", SetLastError = true)]
         internal static extern bool SetupDiGetDeviceInterfaceDetail(
            IntPtr DeviceInfoSet,
            ref SP_DEVICE_INTERFACE_DATA lpDeviceInterfaceData,
            IntPtr hDeviceInterfaceDetailData,
            uint detailSize,
            out uint requiredSize,
            IntPtr hDeviceInfoData);

         [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
         [DllImport("kernel32.dll", CharSet = CharSet.Auto, CallingConvention = CallingConvention.StdCall, SetLastError = true)]
         internal static extern IntPtr CreateFile(
               string lpFileName,
               uint dwDesiredAccess,
               uint dwShareMode,
               IntPtr SecurityAttributes,
               uint dwCreationDisposition,
               uint dwFlagsAndAttributes,
               IntPtr hTemplateFile);

         [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
         [DllImport("kernel32.dll", CharSet = CharSet.Auto, CallingConvention = CallingConvention.StdCall, SetLastError = true)]
         internal static extern bool CloseHandle(IntPtr hHandle);
         #endregion
         #region structs
         [StructLayout(LayoutKind.Sequential)]
         internal struct SP_DEVICE_INTERFACE_DATA {
            public int cbSize;
            public Guid InterfaceClassGuid;
            public int Flags;
            public int Reserved;
         }

         [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
         internal class PSP_DEVICE_INTERFACE_DETAIL_DATA {
            public int cbSize;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 256)]
            public string DevicePath;
         }

         [StructLayout(LayoutKind.Sequential)]
         internal struct HIDD_ATTRIBUTES {
            public int Size; // = sizeof (struct _HIDD_ATTRIBUTES) = 10
            public UInt16 VendorID;
            public UInt16 ProductID;
            public UInt16 VersionNumber;
         }
         [StructLayout(LayoutKind.Sequential)]
         internal struct HIDP_CAPS {
            public UInt16 Usage;
            public UInt16 UsagePage;
            public UInt16 InputReportByteLength;
            public UInt16 OutputReportByteLength;
            public UInt16 FeatureReportByteLength;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 17)]
            public UInt16[] Reserved;
            public UInt16 NumberLinkCollectionNodes;
            public UInt16 NumberInputButtonCaps;
            public UInt16 NumberInputValueCaps;
            public UInt16 NumberInputDataIndices;
            public UInt16 NumberOutputButtonCaps;
            public UInt16 NumberOutputValueCaps;
            public UInt16 NumberOutputDataIndices;
            public UInt16 NumberFeatureButtonCaps;
            public UInt16 NumberFeatureValueCaps;
            public UInt16 NumberFeatureDataIndices;
         }
         #endregion
         #region constants
         internal const uint DIGCF_PRESENT = 0x00000002;
         internal const uint DIGCF_DEVICEINTERFACE = 0x00000010;
         internal const uint GENERIC_READ = 0x80000000;
         internal const uint GENERIC_WRITE = 0x40000000;
         internal const uint FILE_SHARE_READ = 0x00000001;
         internal const uint FILE_SHARE_WRITE = 0x00000002;
         internal const int OPEN_EXISTING = 3;
         internal const int FILE_FLAG_OVERLAPPED = 0x40000000;
         internal const uint MAX_USB_DEVICES = 16;
         #endregion
      }
      #endregion
      #endregion

      #region methods
      /// <summary>Method to seek and find <see cref="USBDevice"/> in the operation system</summary>
      /// <param name="pid">Product ID of <see cref="USBDevice"/> to find</param>
      /// <param name="vid">Vendor ID of <see cref="USBDevice"/> to find</param>
      /// <returns>Fully initialized and ready to work instance of <see cref="USBDevice"/></returns>
      public static USBDevice FindDevice(uint pid, uint vid) {
         var device = new USBDevice(pid,vid);
         var fillRegisters = device.InitRegisters();
         if (!device.IsInvalid && fillRegisters) return device;
         else throw new ArgumentOutOfRangeException(string.Format("Human input device {0} was not found.", pid));
      }

      /// <summary>Converts all significant properties of <see cref="USBDevice"/> to human readble string</summary>
      /// <returns>Human readble string</returns>
      public override string ToString() {
         return string.Format("{0} (Product:{1:x}, Vendor:{2:x}, Version:{3:x}, S/N:{4})", Name, ProductID, VendorID, VersionNumber, SerialNumber);
      }

      /// <summary>Releases currently active handler to <see cref="USBDevice"/></summary>
      /// <returns>True if the operation was sucsessful</returns>
      [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
      protected override bool ReleaseHandle() {
         return Native.CloseHandle(base.handle);
      }

      #region IDisposable Members
      /// <summary>Disposes currently active <see cref="USBDevice"/> and supresses GC</summary>
      public void Dispose() {
         Dispose(true);
         GC.SuppressFinalize(this);

      }
      [SecurityPermission(SecurityAction.Demand, UnmanagedCode = true)]
      void IDisposable.Dispose() {
         if (base.handle != null && !base.IsInvalid) {
            // Free the handle
            base.Dispose();
         }
      }

      #endregion
      #endregion
   }
}
