﻿/* Copyright (c) Tamir Khason (http://khason.net/). All rights reserved.
 * 
 * Released under MS-PL on CodePlex (http://www.codeplex.com/FM/)
 */

#region using
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Native = USBFM.USBRDSDeviceMethods.Native;
#endregion

namespace USBFM {
   /// <summary>Extension methods to work with RDS reports of audio device</summary>
   public static class RDSReportMethods {
      #region private 
      private static Thread _dwRDSThread;
      #endregion
      /// <summary>Static flag signaling whether Radio Data Service interfaces are ready and fully functional</summary>
      public static bool IsReady { get; set; }

      /// <summary>Initializes RDS reporting services</summary>
      /// <param name="device"><see cref="USBRadioDevice"/> for reporting</param>
      public static void InitRDSReports(this USBRadioDevice device) {
         IsReady = true;
         _dwRDSThread = new Thread((ThreadStart)delegate {
            var _freq = default(double);
            while (IsReady) {
               if (device.CurrentFrequency != _freq) device.RDS = new RDSData();
               _freq = device.CurrentFrequency; 
               device.UpdateRDS();
               //Thread.Sleep(TimeSpan.FromMilliseconds(1000 / 200)); //twice max rate
            }
         });
         _dwRDSThread.Start();
      }

      /// <summary>Stops RDS reporting service and releases working thread</summary>
      /// <param name="device"><see cref="USBRadioDevice"/> for reporting</param>
      public static void StopRDSReports(this USBRadioDevice device) {
         IsReady = false;
         if (_dwRDSThread != null && _dwRDSThread.IsAlive) _dwRDSThread.Join();
      }

      #region internal
      internal static void UpdateRDS(this USBRadioDevice device) {
         if (device.RDS.IsInDefaultValue()) device.RDS = new RDSData();
         var registers = device.ReadRegisters(ReportType.RDS);
         if (registers != default(ushort[])) {
            registers.CopyTo(device.Registers, Native.STATUSRSSI);
            device.RDS.SignalStrength = (byte)(device.Registers[Native.STATUSRSSI] & Native.STATUSRSSI_RSSI);
            device.RDS.CurrentStation = ((ushort)(device.Registers[Native.READCHAN] & Native.READCHAN_READCHAN)).ToFrequency(device);
            device.RDS.IsStereo = ((device.Registers[Native.STATUSRSSI] & Native.STATUSRSSI_ST) >> 8) == 0;
            device.UpdateRDSText();
         }
      }

      internal static void UpdateRDSText(this USBRadioDevice device) {
         var errorCount = (device.Registers[Native.STATUSRSSI] & 0x0E00) >> 9;
         var errorFlags = (byte)(device.Registers[Native.SYSCONFIG3] & 0xFF);
         if (errorCount < 4) device.RDS.rdsBlocksValid += (ushort)(4 - errorCount);
         if (errorCount != 0) return;
         if ((device.Registers[Native.STATUSRSSI] & Native.STATUSRSSI_RSSI) > 35) device.RDS.validationLimit = 1;
         else if ((device.Registers[Native.STATUSRSSI] & Native.STATUSRSSI_RSSI) > 31) device.RDS.validationLimit = 2;
         else device.RDS.validationLimit = 4;
         device.RDS.groupType = (RDSGroupType)(device.Registers[Native.RDSB] >> 11);
         device.RDS.updatePI(device.Registers[Native.RDSA]);
         if (((byte)device.RDS.groupType & 0x01) != 0) device.RDS.updatePI(device.Registers[Native.RDSC]);
         device.RDS.updatePTY((byte)((device.Registers[Native.RDSB] >> 5) & 0x1f));
         switch (device.RDS.groupType) {
            case RDSGroupType.TYPE_0A:
               if (device.Registers[Native.RDSA] == device.RDS.internalPI) {
                  device.RDS.HasMS = ((device.Registers[Native.RDSB] & 0x08) == 0x08) ? true : false;
                  device.RDS.HasTA = ((device.Registers[Native.RDSB] & 0x10) == 0x10) ? true : false;
                  device.RDS.HasTP = ((device.Registers[Native.RDSB] & 0x400) == 0x400) ? true : false;

                  if (device.RDS.AF == null) device.RDS.AF = new List<double>();
                  var af = (byte)((device.Registers[Native.RDSC] >> 8) & 0xff);
                  if (af > 0 && af < 205) {
                     var faf = af.ToAFrequency();
                     if (!device.RDS.AF.Contains(faf)) {
                        device.RDS.AF.Add(faf); device.RDS.OnPropertyChanged("AF");
                     }
                  }
                  af = (byte)(device.Registers[Native.RDSC] & 0xff);
                  if (af > 0 && af < 205) {
                     var faf = af.ToAFrequency();
                     if (!device.RDS.AF.Contains(faf)) {
                        device.RDS.AF.Add(faf); device.RDS.OnPropertyChanged("AF");
                     }
                  }
               }
               var addr = (device.Registers[Native.RDSB] & 0x3) * 2;
               device.RDS.updatePS((byte)(addr + 0), (byte)(device.Registers[Native.RDSD] >> 8));
               device.RDS.updatePS((byte)(addr + 1), (byte)(device.Registers[Native.RDSD] & 0xff));
               break;
            case RDSGroupType.TYPE_0B:
               addr = (device.Registers[Native.RDSB] & 0x3) * 2;
               device.RDS.updatePS((byte)(addr + 0), (byte)(device.Registers[Native.RDSD] >> 8));
               device.RDS.updatePS((byte)(addr + 1), (byte)(device.Registers[Native.RDSD] & 0xff));
               device.RDS.updatePI(device.Registers[Native.RDSC]);
               if (device.Registers[Native.RDSA] == device.RDS.internalPI) {
                  device.RDS.HasMS = ((device.Registers[Native.RDSB] & 0x08) == 0x08) ? true : false;
                  device.RDS.HasTA = ((device.Registers[Native.RDSB] & 0x10) == 0x10) ? true : false;
                  device.RDS.HasTP = ((device.Registers[Native.RDSB] & 0x400) == 0x400) ? true : false;
               }
               break;
            case RDSGroupType.TYPE_1A:
               var v_ecc = (byte)(device.Registers[Native.RDSC] & 0xff);
               if (device.RDS.c_ecc > device.RDS.validationLimit) {
                  device.RDS.ecc = v_ecc;
               } else {
                  device.RDS.c_ecc++;
               }
               break;
            case RDSGroupType.TYPE_1B: device.RDS.updatePI(device.Registers[Native.RDSC]); break;
            case RDSGroupType.TYPE_2A:
               addr = (device.Registers[Native.RDSB] & 0xf) * 4;
               var abflag = (device.Registers[Native.RDSB] & 0x10) == 0x10 ? true : false;
               var registers = device.Registers.Skip(Native.RDSC-1).Take(4).ToArray();
               device.RDS.updateRT(abflag, addr, registers, errorFlags);
               break;
            case RDSGroupType.TYPE_2B:
               addr = (device.Registers[Native.RDSB] & 0xf) * 4;
               device.RDS.HasTP = ((device.Registers[Native.RDSB] & 0x400) == 0x400) ? true : false;
               abflag = (device.Registers[Native.RDSB] & 0x10) == 0x10 ? true : false;
               // The last 32 bytes are unused in this format
               //	m_rtTmp0[32]    = 0x0d;
               //	m_rtTmp1[32]    = 0x0d;
               //	m_rtCnt[32]     = RT_VALIDATE_LIMIT;
               registers = device.Registers.Skip(Native.RDSD-1).Take(2).ToArray();
               device.RDS.updateRT(abflag, addr, registers, errorFlags);
               break;
            case RDSGroupType.TYPE_8A:
               device.RDS.TMC = new string(new char[]{ 
               (char)((device.Registers[Native.RDSA] >> 8) & 0xff),
               (char)(device.Registers[Native.RDSA] & 0xff),
               (char)((device.Registers[Native.RDSA] >> 8) & 0xff),
               (char)(device.Registers[Native.RDSA] & 0xff),
               (char)((device.Registers[Native.RDSA] >> 8) & 0xff),
               (char)(device.Registers[Native.RDSA] & 0xff),
               (char)((device.Registers[Native.RDSA] >> 8) & 0xff),
               (char)(device.Registers[Native.RDSA] & 0xff)
                        });
               break;
            case RDSGroupType.TYPE_14B:
               if ((device.Registers[Native.RDSB] & 0x08) == 0x08) {
                  if (device.RDS.TrafficPI == device.Registers[Native.RDSD]) device.RDS.TrafficStart += 50; 
                  else {
                     device.RDS.TrafficPI = device.Registers[Native.RDSD];
                     device.RDS.TrafficStart = 0;
                  }
               } else {
                  if (device.RDS.TrafficPI == device.Registers[Native.RDSD]) device.RDS.TrafficEnd += 50;
                  else {
                     device.RDS.TrafficPI = device.Registers[Native.RDSD];
                     device.RDS.TrafficEnd = 0;
                  }
               }
               break;
            default: int k = 10;break;
         }
      }      
      
      private static double ToAFrequency(this byte af) {
         var basefreq = 87.5;
         var offset = af / 10;
         return Math.Round(basefreq + offset,1,MidpointRounding.AwayFromZero);
      }
      #endregion

   }
}
