﻿/* Copyright (c) Tamir Khason (http://khason.net/). All rights reserved.
 * 
 * Released under MS-PL on CodePlex (http://www.codeplex.com/FM/)
 */

#region using
using System;
using System.Runtime.InteropServices;
using System.Threading;
using Microsoft.DirectX.DirectSound;
#endregion

namespace USBFM {
   /// <summary>
   /// Extension methods to work with Direct Sound
   /// </summary>
   public static class DirectSoundMethods {
      #region private
      private static int _dwNotifySize, _dwOutputBufferSize, _dwCaptureBufferSize;
      private const int NUM_BUFFERS = 16;
      private static Thread _dwCaptureThread;
      private static CaptureBuffer _dwCapBuffer;
      private static SecondaryBuffer _dwDevBuffer;
      #endregion

      /// <summary>Static flag signaling whether Direct Sound interfaces are ready and fully functional</summary>
      public static bool IsReady { get; set; }

      /// <summary>
      /// Initializes Direct Sound echo service
      /// </summary>
      /// <param name="captureDescriptor">Name of device to seek and use within DS. Usually it is something like "Mic" or "Fm". Descriptor is case insensitive.</param>
      public static void InitEchoService(string captureDescriptor) {
         IsReady = true;
         var format = new WaveFormat {
            SamplesPerSecond = 96000,
            BitsPerSample = 16,
            Channels = 2,
            FormatTag = WaveFormatTag.Pcm
         };

         format.BlockAlign = (short)(format.Channels * (format.BitsPerSample / 8));
         format.AverageBytesPerSecond = format.SamplesPerSecond * format.BlockAlign;

         _dwNotifySize = Math.Max(4096, format.AverageBytesPerSecond / 8);
         _dwNotifySize -= _dwNotifySize % format.BlockAlign;
         _dwCaptureBufferSize = NUM_BUFFERS * _dwNotifySize;
         _dwOutputBufferSize = NUM_BUFFERS * _dwNotifySize / 2;

         //init cap buffer
         var cap = default(Capture);
         var cdc = new CaptureDevicesCollection();
         for (int i = 0; i < cdc.Count; i++) {
            if (cdc[i].Description.ToLower().Contains(captureDescriptor.ToLower())) {
               cap = new Capture(cdc[i].DriverGuid);
               break;
            }
         }
         var capDesc = new CaptureBufferDescription {
            Format = format,
            BufferBytes = _dwCaptureBufferSize
         };
         _dwCapBuffer = new CaptureBuffer(capDesc, cap);

         //init play buffer
         var dev = new Device();
         dev.SetCooperativeLevel(Native.GetDesktopWindow(), CooperativeLevel.Priority);


         var devDesc = new BufferDescription {
            BufferBytes = _dwOutputBufferSize,
            Format = format,
            DeferLocation = true,
            GlobalFocus = true
         };
         _dwDevBuffer = new SecondaryBuffer(devDesc, dev);

         //set notifications
         var _resetEvent = new AutoResetEvent(false);
         var _notify = new Notify(_dwCapBuffer);
         //half&half
         var bpn1 = new BufferPositionNotify();
         bpn1.Offset = _dwCapBuffer.Caps.BufferBytes / 2 - 1;
         bpn1.EventNotifyHandle = _resetEvent.SafeWaitHandle.DangerousGetHandle();
         var bpn2 = new BufferPositionNotify();
         bpn2.Offset = _dwCapBuffer.Caps.BufferBytes - 1;
         bpn2.EventNotifyHandle = _resetEvent.SafeWaitHandle.DangerousGetHandle();

         _notify.SetNotificationPositions(new BufferPositionNotify[] { bpn1, bpn2 });

         //start buffers
         int offset = 0;
         _dwCaptureThread = new Thread((ThreadStart)delegate {
            _dwCapBuffer.Start(true);

            while (IsReady) {
               _resetEvent.WaitOne();
               var read = _dwCapBuffer.Read(offset, typeof(byte), LockFlag.None, _dwOutputBufferSize);
               _dwDevBuffer.Write(0, read, LockFlag.EntireBuffer);
               offset = (offset + _dwOutputBufferSize) % _dwCaptureBufferSize;
               _dwDevBuffer.SetCurrentPosition(0);
               _dwDevBuffer.Play(0, BufferPlayFlags.Default);
            }
            _dwCapBuffer.Stop();
         });
         _dwCaptureThread.Start();
      }

      /// <summary>
      /// Stops Direct Sound echo service and releases working thread
      /// </summary>
      public static void StopEchoService() {
         IsReady = false;
         if (_dwCaptureThread != null && _dwCaptureThread.IsAlive) _dwCaptureThread.Join();
      }

      private static class Native {
         [DllImport("user32.dll", SetLastError = false)]
         public static extern IntPtr GetDesktopWindow();
      }
   }
}
