/* ==================================================================================================
 * TPMS plugin
 * File: setup.cs
 * Author: Dipl.-Ing. Wolfgang Otto
 * eMail: Wolfgang@exner-otto.de
 * Date: January 2009
 * 
 * THIS PLUGIN IS A CONVERSION OF THE MARVELESS TOOL OF
 * **************************************************************************************************
 * Roberto Montanari, RoadRunner TMS Plugin															*
 * His VB sources are converted to C# and modified to be used as a CENTRAFUSE plugin				*
 * Thanks for his great work!																		*
 * http://www.mp3car.com/vbulletin/rr-plugins/110330-rrtms-roadrunner-tire-monitoring-sistem.html	*
 * **************************************************************************************************
 * This file is subject to the condition of GNU GENERAL PUBLIC LICENSE								*
 * Therefore you are free to use this software for your own requirements and use the				*
 * source code for any modification.																*
 * The only wish is to inform the author about any kind of development.								*
 * =================================================================================================*/

using System;
using System.Threading;
using System.Xml;
using System.Web;
using System.Windows.Forms;
using System.Collections;
using System.IO;
using System.Diagnostics;
using centrafuse.Plugins;



namespace tpms
{
	/// Setup class inherits from CFSetup  so that it will not show up as a seperate plugin, but a dialog within a plugin.  It uses the standard setup screens from the main application
	
	public class setup : CFDialog
	{		
		// private variables
		private tpms.LrnStage LearnStage = tpms.LrnStage.Off;
		private enum AssignStage {OFF = 0, STARTED = 1, ASSIGNED = 2}
		private AssignStage aStage = AssignStage.OFF;

		private System.Windows.Forms.Timer tmrLearning = new System.Windows.Forms.Timer();
		private System.Windows.Forms.Timer tmrFlasher = new System.Windows.Forms.Timer();	// timer for flashing tires

		private int TimerCounter = 0;
		private int LearnTire;
		private bool LearnTireOn = false;
        private static int totalpages = 3;
        private static int currentpage = 1;
		private static info.Infodisplay id = null;

		/// setup()
		public setup()
		{
			this.CF_pluginName = "SETUP";
			this.CF_pluginPauseAudioFlag = false;
			this.CF_pluginIsGUI = true;
            totalpages = 2;
            currentpage = 1;
		}
			
		public void UpdatePressureTempValues()
		{
			string s,t;
			byte bt;

			switch (tpms.PressureUnit)
			{
				case tpms.Pressure.KPa:
					tpms.PressureSuffix = "KPa";
					break;

				case tpms.Pressure.Bar:
					tpms.PressureSuffix = "Bar";
					break;

				case tpms.Pressure.PSI:
					tpms.PressureSuffix = "PSI";
					break;
			}
			
			switch (tpms.TemperatureUnit)
			{
				case tpms.Temperature.C:
					tpms.TemperatureSuffix = "C";

					break;

				case tpms.Temperature.F:
					tpms.TemperatureSuffix = "F";
					
					break;
			}

			for (int i = 1; i<=5 ; i++)
			{
				s = Convert.ToString(i);
				// get minimum pressure values
				bt = Convert.ToByte(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/PRESSURE_MIN/TIRE" + s));
				this.CF_updateButtonText("PMIN" + s, PressureToString(ref bt));
				tpms.Tire[i-1].MinPressure = bt;

				// get maximum pressure values
				bt = Convert.ToByte(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/PRESSURE_MAX/TIRE" + s));
				this.CF_updateButtonText("PMAX" + s, PressureToString(ref bt));
				tpms.Tire[i-1].MaxPressure = bt;					

				// get maximum temperature values
				bt = Convert.ToByte(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/TEMPERATURE_MAX/TIRE" +s));
				this.CF_updateButtonText("TMAX" + s, TemperatureToString(ref bt));
				tpms.Tire[i-1].MaxTemperature = bt;
					
				// get sensor location entries
				t = this.pluginConfig.readPluginField("/APPCONFIG/SETUP/SENSORLOCATION/TIRE" +s);
				this.CF_updateButtonText("TIRE" + s, t);
				tpms.Tire[i-1].Sensor = Convert.ToByte(t);
			}	
		}
		public string TemperatureToString(ref byte temp)
		{
			string res = string.Empty;
			if (temp == 0xFF) 
			{
				res = "--" + tpms.TemperatureSuffix;
				return res;
			}

			if (temp <= 40)
			{
				res =  String.Format("{0:F0} ", 0) + tpms.TemperatureSuffix;
				return res;
			}

			switch (tpms.TemperatureUnit)
			{
				case tpms.Temperature.C:
					res =  String.Format("{0:F0} ", temp - 40) + tpms.TemperatureSuffix;
					break;
				case tpms.Temperature.F:
					res =  String.Format("{0:F0} ", ((temp - 40) * 9 / 5) + 32) + tpms.TemperatureSuffix;
					break;
			} 
			return res;
		}
		public string PressureToString(ref byte pressure)
		{
			string res = string.Empty;
			//Status Port]      Addres:  0x379H or in VB Express &H379S
			if (pressure == 0xFF) 
			{
				res = "--- " + tpms.PressureSuffix;
				return res;
			}
			if (pressure <= 40)
			{
				res =  String.Format("{0:F1} ", 0) + tpms.PressureSuffix;
				return res;
			}

			switch (tpms.PressureUnit)
			{
				case tpms.Pressure.KPa:
					res =  String.Format("{0:F1} ", pressure * 2.5 - 100) + tpms.PressureSuffix;
					break;
				case tpms.Pressure.Bar:
					res =  String.Format("{0:F1} ", ((pressure * 2.5) - 100) / 100) + tpms.PressureSuffix;
					break;
				case tpms.Pressure.PSI:
					res = String.Format("{0:F0} ", ((pressure * 2.5) - 100) * 0.145) + tpms.PressureSuffix;
					break;
			}
			return res;
		}

		
		#region CFPLUGIN standard functions
		public void loadSettings()
		{
            if (tpms.useDebug) CFTools.writeModuleLog("START", tpms.errLog);
			string p,t;
					
			try
			{
				this.CF_updateText("TITLE",			this.pluginLang.readPluginField("/APPLANG/TITLE"));
				this.CF_updateText("PAGE",			Convert.ToString(currentpage) + "/" + Convert.ToString(totalpages));
			
				this.CF_updateButtonText("OK", LanguageReader.getText("APPLANG/BUTTONS/SAVE"));
				this.CF_updateButtonText("CANCEL", LanguageReader.getText("APPLANG/BUTTONS/CANCEL"));

                //====================================================================================================
				if (currentpage == 2)   // special configuration page, so it is no possibility to use the normal setup
                                        // class construction
                //====================================================================================================
				{
					#region PAGE 2
					this.CF_updateButtonText("LEARN", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/LEARN"));	
		
					p = this.pluginConfig.readPluginField("/APPCONFIG/SETUP/PRESSUREUNIT");
					this.CF_updateButtonText("PUNIT", p);
					t = this.pluginConfig.readPluginField("/APPCONFIG/SETUP/TEMPERATUREUNIT").Trim().ToUpper();
					
					switch (t)
					{
						case "F":
							this.CF_updateButtonText("TUNIT", this.pluginLang.readPluginField("/APPLANG/SETUP/FTEMP"));
							break;
						case "C":
							this.CF_updateButtonText("TUNIT", this.pluginLang.readPluginField("/APPLANG/SETUP/CTEMP"));
							break;
						default:
							this.CF_updateButtonText("TUNIT","---");
							break;
					}

					UpdatePressureTempValues();
		
					this.CF_updateText("LABELTIRE",		this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/TIRE"));
					this.CF_updateText("LABELPMIN",		this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/PMIN"));
					this.CF_updateText("LABELPMAX",		this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/PMAX"));
					this.CF_updateText("LABELPUNIT",	this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/PUNIT"));
					this.CF_updateText("LABELTUNIT",	this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/TUNIT"));
					this.CF_updateText("LABELLEARN",	this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/LEARN"));
					this.CF_updateText("LABELTMAX",		this.pluginLang.readPluginField("/APPLANG/SETUP/LABELS/TMAX"));

					#endregion
				}
                //====================================================================================================
				if (currentpage == 1) // it is page No. 1
                //====================================================================================================
				{
					#region PAGE 1

                    this.CF_updateText("LABEL1",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL1"));
                    this.CF_updateText("LABEL2",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL2"));
                    this.CF_updateText("LABEL3",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL3"));
                    this.CF_updateText("LABEL4",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL4"));
                    this.CF_updateText("LABEL5",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL5"));
                    this.CF_updateText("LABEL6",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL6"));
                    this.CF_updateText("LABEL7",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL7"));
                    this.CF_updateText("LABEL8",        this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL8"));
                  	
					for (int i=5; i<=8; i++)
					{
					    this.CF_updateButtonText("BUTTON" + Convert.ToString(i) + "ON", LanguageReader.getText("APPLANG/SETUP/ON"));
						this.CF_updateButtonText("BUTTON" + Convert.ToString(i) + "OFF", LanguageReader.getText("APPLANG/SETUP/OFF"));
					}

					// update button values
					//---------------------
					try { this.CF_updateButtonText("BUTTON1", this.pluginLang.readPluginField("/APPLANG/DISPLAYNAME"));	}
					catch {	this.CF_updateButtonText("BUTTON1", "TPMS"); }

					try { this.CF_updateButtonText("BUTTON2", this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLMS"));}
					catch { this.CF_updateButtonText("BUTTON2", "5000");}

					try { this.CF_updateButtonText("BUTTON3", this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLFLASHMS"));}
					catch { this.CF_updateButtonText("BUTTON3", "300");}

                    try { this.CF_updateButtonText("BUTTON4", this.pluginLang.readPluginField("/APPLANG/SETUP2/BUTTONS/BUTTON4")); }
                    catch { this.CF_updateButtonText("BUTTON4", "ERROR: BUTTON4 TEXT MISSING"); }

					// button 5 - ACTIVE
					if (tpms.configxml.SelectSingleNode("/APPCONFIG/ACTIVE").InnerText.ToUpper() == "TRUE")
					{
						this.CF_setButtonOn("BUTTON5ON");
						this.CF_setButtonOff("BUTTON5OFF");
					}
					else
					{
						this.CF_setButtonOn("BUTTON5OFF");
						this.CF_setButtonOff("BUTTON5ON");
					}
					// button 6 - USEDEBUG
					if (tpms.configxml.SelectSingleNode("/APPCONFIG/USEDEBUG").InnerText.ToUpper() == "TRUE")
					{
						this.CF_setButtonOn("BUTTON6ON");
						this.CF_setButtonOff("BUTTON6OFF");
					}
					else
					{
						this.CF_setButtonOn("BUTTON6OFF");
						this.CF_setButtonOff("BUTTON6ON");
					}
					
					// button 7 - FORCED DEVICE DETECTION
					if (tpms.configxml.SelectSingleNode("/APPCONFIG/FORCEDETECT").InnerText.ToUpper() == "TRUE")
					{
						this.CF_setButtonOn("BUTTON7ON");
						this.CF_setButtonOff("BUTTON7OFF");
					}
					else
					{
						this.CF_setButtonOn("BUTTON7OFF");
						this.CF_setButtonOff("BUTTON7ON");
					}

                    // button 8 - USE 5 THAN 4 TIRE DISPLAY
                    if (tpms.configxml.SelectSingleNode("/APPCONFIG/USE5TIRES").InnerText.ToUpper() == "TRUE")
                    {
                        this.CF_setButtonOn("BUTTON8ON");
                        this.CF_setButtonOff("BUTTON8OFF");
                    }
                    else
                    {
                        this.CF_setButtonOn("BUTTON8OFF");
                        this.CF_setButtonOff("BUTTON8ON");
                    }

					#endregion
				}

			}
			catch(Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); }
            if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog);
		}

		public override void CF_pluginClose()
		{
			// Close the open handles to the device.
			tmrFlasher.Tick -= new EventHandler(tmrFlasher_Timer);
			tmrLearning.Tick -= new EventHandler(tmrLearning_Timer);
			System.Windows.Forms.Application.DoEvents();
			this.Dispose();		// Disposes the plugin.
		}
		
		public override void CF_localskinsetup()
		{

            if (tpms.useDebug) CFTools.writeModuleLog("START", tpms.errLog); 
            
			string p = string.Empty;
			string s = string.Empty;
			string t = string.Empty;

			try
			{
				this.CF_clearControls();

				try { this.pluginDisplay = Int32.Parse(this.pluginConfig.readPluginField("/APPCONFIG/DISPLAY")); }
				catch { this.pluginDisplay = 1; }
                
				this.CF_initSkin(tpms.pluginFolder +"/skins/" + this.currentSkin + "/skin.xml");
                
				if (currentpage == 2)
				{
					this.CF_loadImages(	tpms.pluginFolder +"/skins/" + this.currentSkin + "/SETUP_OFF.png",
						tpms.pluginFolder +"/skins/" + this.currentSkin + "/SETUP_DOWN.png");
				}
                
				if (currentpage == 1)
				{
					this.CF_loadImages(	tpms.pluginFolder +"/skins/" + this.currentSkin + "/SETUP2_OFF.png",
						tpms.pluginFolder +"/skins/" + this.currentSkin + "/SETUP2_DOWN.png");
				}
                
				// the following elements are used on both pages
				this.CF_createLabel("TITLE");
				this.CF_createLabel("PAGE");
                // Creates buttons.
				this.CF_createButton("OK", LanguageReader.getText("APPLANG/BUTTONS/SAVE"), new MouseEventHandler(Setup_Click));
				this.CF_createButton("CANCEL", LanguageReader.getText("APPLANG/BUTTONS/CANCEL"), new MouseEventHandler(Setup_Click));
				this.CF_createButton("BACK", "", new MouseEventHandler(Setup_Click));			
				this.CF_createButton("FORWARD", "", new MouseEventHandler(Setup_Click));
                // now again page sensitive handling is used
				if (currentpage == 2)
				{
					// Creates labels.
					this.CF_createLabel("LABELTMAX");
					this.CF_createLabel("LABELTIRE");
					this.CF_createLabel("LABELPMIN");
					this.CF_createLabel("LABELPMAX");
					this.CF_createLabel("LABELPUNIT");
					this.CF_createLabel("LABELTUNIT");
					this.CF_createLabel("LABELLEARN");
									
					// Creates buttons.
					this.CF_createButton("LEARN", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/LEARN"), new MouseEventHandler(Setup_Click));			
					this.CF_createButton("TUNIT", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/TUNIT"), new MouseEventHandler(Setup_Click));			
					this.CF_createButton("PUNIT", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/PUNIT"), new MouseEventHandler(Setup_Click));			
					
					for (int i = 1; i <= 5; i++)
					{
						s = Convert.ToString(i);
						this.CF_createButton("PMIN" + s, "", new MouseEventHandler(Setup_Click));
						this.CF_createButton("PMAX" + s, "", new MouseEventHandler(Setup_Click));
						this.CF_createButton("TMAX" + s, "", new MouseEventHandler(Setup_Click));
						this.CF_createButton("TIRE" + s, "", new MouseEventHandler(Setup_Click));
						this.CF_createButton("LOC" + s, "", new MouseEventHandler(Setup_Click));

					}
				}
				
				if (currentpage == 1)
				{
					for (int i = 1; i <= 4; i++)
					{
						s = Convert.ToString(i);
						t = Convert.ToString(i+4);
                        this.CF_createButton("BUTTON" + s, "", new MouseEventHandler(Setup_Click));
						this.CF_createButton("BUTTON" + t + "ON", "", new MouseEventHandler(Setup_Click));
						this.CF_createButton("BUTTON" + t + "OFF", "", new MouseEventHandler(Setup_Click));
						this.CF_createLabel("LABEL" + s);
						this.CF_createLabel("LABEL" + t);
					}
				}
			}

			catch(Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); }
			
			loadSettings();

			this.Invalidate();

            if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog);
		}

		public override void CF_pluginInit()
		{
			string sx = string.Empty;

            if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog); 

			try
			{	
				this.CF_loadConfig(tpms.pluginCfg);
				tpms.configxml.Load(tpms.pluginCfg);

				this.currentSkin = this.CF_getConfigSetting(CF_ConfigSettings.Skin);
				if(!Directory.Exists(tpms.pluginFolder + "/skins/" + this.currentSkin)) this.currentSkin = "Onyx WS Night";

				sx = tpms.pluginFolder + "/languages/" + this.CF_getConfigSetting(CF_ConfigSettings.Language).Trim() + ".xml";
				if (!File.Exists(sx)) sx = tpms.pluginFolder + "/languages/english.xml";
				this.CF_loadLang(sx);

				this.CF_localskinsetup();
				
				tmrFlasher = new System.Windows.Forms.Timer();
				tmrFlasher.Interval = 1000;
				tmrFlasher.Enabled = false;
				tmrFlasher.Tick += new EventHandler(tmrFlasher_Timer);

				tmrLearning = new System.Windows.Forms.Timer();
				tmrLearning.Interval = 250;
				tmrLearning.Enabled = false;
				tmrLearning.Tick += new EventHandler(tmrLearning_Timer);
			}
			catch(Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); }

            if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog); 

		}

		public override void CF_pluginUpdateLanguageSkin(string language, string skin) 
		{
			// This is called by the system when the language has changed.
			this.CF_localskinsetup();
		}

		#endregion

		protected void tmrLearning_Timer(object sender, EventArgs e)
		{
			if (TimerCounter > 0) // if no detection timeout reached
			{
			
				TpmsInputReport inputReport = null;
				
				//--------------------------------------
				// if no tire is selected, switch it off
				//--------------------------------------
				if (LearnTire == 0) LearnStage = tpms.LrnStage.Off;

				/*
				 * In order to facilitate the communication between the device and PC, Roberto Montenari has 
				 * developed a simple communication protocol. Basically the device waits for a request from 
				 * the PC (1 byte) and returns the information required (4 byte).
				 * 
				 * The following table shows the protocol details:
				 * 00H (0) Abort any pairing request (see below) --> Returns: Nothing
				 * 01H (1) Sensor 1 pairing request --> Returns: Nothing
				 * 02H (2) Sensor 2 pairing request --> Returns: Nothing
				 * 03H (3) Sensor 3 pairing request --> Returns: Nothing
				 * 04H (4) Sensor 4 pairing request --> Returns: Nothing
				 * 05H (5) Sensor 5 pairing request --> Returns: Nothing
				 * 
				 * 10H (16) State of a pairing request --> Returns: X,0H,0H,0H
				 * Where X is the number of the sensor to be paired, or 0 if the pairing has been completed.
				 */
				switch (LearnStage)
				{

						#region OFF - do nothing ...
					case tpms.LrnStage.Off:
						tmrLearning.Enabled = false;
						break;
						#endregion OFF

						#region STARTED
					case tpms.LrnStage.Started:
						// If the device hasn't been detected or it timed out on a previous attempt
						// to access it, look for the device.
						if(tpms.device == null) tpms.device = TpmsDevice.FindTpmsDevice();
						if(tpms.device != null) // device was found
						{
							TpmsOutputReport outputReport = new TpmsOutputReport(tpms.device);					
							// Load the buffer with the byte to be sent.
							// terminate any older pairing request
							// 00H (0) Abort any pairing request (see below) --> Returns: Nothing
							outputReport.Buffer[0] = 0x0;
							outputReport.Buffer[1] = 0x0;
							tpms.device.WriteReport(outputReport);

							// now start a new pairing request
							// 01H (1) Sensor 1 pairing request --> Returns: Nothing
							// 02H (2) Sensor 2 pairing request --> Returns: Nothing
							// 03H (3) Sensor 3 pairing request --> Returns: Nothing
							// 04H (4) Sensor 4 pairing request --> Returns: Nothing
							// 05H (5) Sensor 5 pairing request --> Returns: Nothing
							outputReport.Buffer[0] = 0x0;
							outputReport.Buffer[1] = tpms.Tire[LearnTire-1].Sensor;
							tpms.device.WriteReport(outputReport);

							// tpms.device = null; modified version 0.4.1.3

							LearnStage = tpms.LrnStage.Waiting;
							
							if (id == null) id = new info.Infodisplay(this.currentSkin);
							id.ShowInfo(this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/WAIT"));
							
							/*tpms tp = new tpms();
							tp.WriteProfile("tmrLearning_Timer started");					
							tp.Dispose();
							*/
						}
						else
						{	// if no device is found stop the learning process
							AbortLearning(true);
						}

						break;
						#endregion STARTED

					case tpms.LrnStage.Waiting:
					
						//--------------------------------------------------------------------------    
						// If the device hasn't been detected or it timed out on a previous attempt  
						// to access it, look for the device.
						//--------------------------------------------------------------------------

						if(tpms.device == null) tpms.device = TpmsDevice.FindTpmsDevice();
						if(tpms.device != null)
						{
							// Ask the receiver wether pairing is successfull
							// 10H (16) State of a pairing request --> Returns: X,0H,0H,0H
							
							TpmsOutputReport outputReport = new TpmsOutputReport(tpms.device);
							outputReport.Buffer[0] = 0x0;
							outputReport.Buffer[1] = 0x10;
							tpms.device.WriteReport(outputReport);

							// tpms.device = null; modified version 0.4.1.3

							// Read the data from the device.
							inputReport = tpms.device.ReadReport();	
						}
						else
						{
							LearnStage = tpms.LrnStage.Off;	
						}
			
						// Check if the sensor ID as been paired
						/* 10H (16) State of a pairing request --> Returns: X,0H,0H,0H
						 * Where X is the number of the sensor to be paired, or 0 if the pairing has been completed.
						 */
						if (inputReport != null) 
						{
							if (inputReport.Buffer[1] == 0)	
							{
                                if (tpms.useDebug) 
                                    CFTools.writeModuleLog("Pairing of tire " + Convert.ToString(LearnTire) + " sucessfull",tpms.errLog); 
								AbortLearning(false);
							}
						}

						break;
				}
				TimerCounter--;
			}
		}
		private void AbortLearning(bool FORCE)
		{
			LearnStage = tpms.LrnStage.Off;
			
			#region forced break
			if (FORCE == true)
			{
				// If the device hasn't been detected or it timed out on a previous attempt
				// to access it, look for the device.
				if(tpms.device == null) tpms.device = TpmsDevice.FindTpmsDevice();
				if(tpms.device != null)
				{
					// Load the buffer with the byte to be sent.
					TpmsOutputReport outputReport = new TpmsOutputReport(tpms.device);
					outputReport.Buffer[0] = 0x0;
					outputReport.Buffer[1] = 0x0;
					// Write the data to the device
					tpms.device.WriteReport(outputReport);
					// tpms.device = null; modified version 0.4.1.3
				}
				tmrFlasher.Stop();		// stop flashing timer
				tmrLearning.Stop();		// stop learning timer 
				TimerCounter = 20;		// reset the time to the default timeout value
				aStage = AssignStage.OFF;

				this.CF_setButtonOff("LOC" + LearnTire.ToString());
				
				if (id == null) id = new info.Infodisplay(this.currentSkin);
				id.ShowInfo(this.pluginLang.readPluginField("/APPLANG/MSGLEARNABORTED"),2000);
			}
			#endregion forced break

			else
			
			#region normal pairing successfully finished
			{
				// successfully paired tire sensor
				tmrFlasher.Stop();		// stop flashing timer
				tmrLearning.Stop();		// stop learning timer 
				TimerCounter = 20;		// reset the time to the default timeout value
				aStage = AssignStage.ASSIGNED;

				this.CF_setButtonOn("LOC" + LearnTire.ToString());
				
				if (id == null) id = new info.Infodisplay(this.currentSkin);
				id.HideInfo();

				string strTireSuccess = this.pluginLang.readPluginField("/APPLANG/MSGLEARNSUCCESS");
				strTireSuccess = strTireSuccess.Replace("%1",LearnTire.ToString());
				this.CF_systemDisplayDialog(CF_Dialogs.OkBox,strTireSuccess);
			}
			#endregion normal pairing successfully finished

			//Close the open handles to the device.
			if (tpms.device != null) 
			{
				tpms.device.Dispose();
				tpms.device = null;
			}
			this.CF_setButtonOff("LOC" + LearnTire.ToString());
			this.CF_updateButtonText("LEARN", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/LEARN"));
            
			LearnTire = 0;
		}
		protected void tmrFlasher_Timer(object sender, EventArgs e)
		{
			if (LearnTire != 0)
			{
				LearnTireOn = !LearnTireOn;
				if (LearnTireOn)
					this.CF_setButtonOn("LOC" + Convert.ToString(LearnTire));
				else
					this.CF_setButtonOff("LOC" + Convert.ToString(LearnTire));
			}
		}
        
		#region Standard Button Clicks	// standard button events, used the same way in every application 
		private void Setup_Click(object sender, MouseEventArgs e)
		{
			string btnStr = string.Empty;
			string btnNr = string.Empty;
			string strres = string.Empty;

            if (tpms.useDebug) CFTools.writeModuleLog("START", tpms.errLog); 
			
			byte res;
			string OldValue;
			string resultvalue, resulttext, sx;

			CFControls.skinButton tmp = new centrafuse.Plugins.CFControls.skinButton();
			tmp = (CFControls.skinButton)sender;
			btnStr = tmp.Name.ToString().Trim().ToUpper();
						
			switch (btnStr)
			{
				#region FORWARD - goto next setup page
				case "FORWARD":
					if((currentpage + 1) <= totalpages)
					{
						currentpage++;
						CF_localskinsetup();
					}
					break;
				#endregion
					
				#region BACK -  goto previous setup page
				case "BACK":
					if((currentpage - 1) > 0)
					{
						currentpage--;
						CF_localskinsetup();
					}
					break;	
				#endregion
			
				#region OK - Save all modified settings
				case "OK":
					// Launches system YesNo dialog to verify the user wants to overwrite their current settings.  If yes then the local XML document is saved.
					if(this.CF_systemDisplayDialog(CF_Dialogs.YesNo, this.pluginLang.readPluginField("/APPLANG/SETUP/OVERWRITETEXT")) == DialogResult.OK)
					{
						this.DialogResult = DialogResult.OK;
						// Save and reload config settings
						tpms.configxml.Save(tpms.pluginCfg);
						this.CF_loadConfig(tpms.pluginCfg);

						sx = tpms.pluginFolder + "/languages/" + this.CF_getConfigSetting(CF_ConfigSettings.Language).Trim().ToUpper() + ".xml";
						if (!File.Exists(sx)) sx = tpms.pluginFolder + "/languages/english.xml";
						tpms.languagexml.Save(sx);
						this.CF_loadLang(sx);

						if (tmrFlasher.Enabled) tmrFlasher.Stop();
					}
					break;
				#endregion

				#region CANCEL - rollback a changed settings
				case "CANCEL":
					this.DialogResult = DialogResult.Cancel;
					tpms.configxml.Load(tpms.pluginCfg);
					sx = tpms.pluginFolder + "/languages/" + this.CF_getConfigSetting(CF_ConfigSettings.Language).Trim().ToUpper() + ".xml";
					if (!File.Exists(sx)) sx = tpms.pluginFolder + "/languages/english.xml";
					tpms.languagexml.Load(sx);
					if (tmrFlasher.Enabled) tmrFlasher.Stop();
					break;
				#endregion

				#region LEARN A NEW TIRE SENSOR
				case "LEARN":
					if (LearnTire == 0) 
					{
						// First select a tire to pair
						this.CF_systemDisplayDialog(CF_Dialogs.OkBox, this.pluginLang.readPluginField("/APPLANG/SETUP/MSG_NO_TIRE_SELECTED"));
					}
					else
					{
						// There are three possible operation states and two button states
						//	- LEARN, start the learning process, set status to WAIT, set label to ABORT and display a info dialogo with WAITING...
						//  - WAIT, an info dialog will pop up
						//  - ABORT, stop the learning process and set the label back to START
						switch (LearnStage)
						{
							case tpms.LrnStage.Off:
								//==============================================================
								// if current status is OFF, so set label to WAIT and start learning 
								// process
								//==============================================================
								
								// activate the timer to terminate the learn process
								TimerCounter = 120; // set maximum recognition time to 20 ( 120 x 250ms = 30 seconds)
								LearnStage = tpms.LrnStage.Started;
								this.CF_updateButtonText("LEARN", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/WAIT"));
								string strPressure = this.pluginLang.readPluginField("/APPLANG/SETUP/MSGLEARNSTART");
								byte bPressure = (40 + 12);
                                								
								strPressure = strPressure.Replace("%1",PressureToString(ref bPressure)); 
								ShowLargeMessage(strPressure);
								tmrLearning.Enabled = true;
								break;

							case tpms.LrnStage.Started:
							case tpms.LrnStage.Waiting:
								//----------------------------------------------------------------------
								// if current status is WAIT then abort the learning process
								//----------------------------------------------------------------------
								LearnStage = tpms.LrnStage.Off;
								AbortLearning(false);
								this.CF_updateButtonText("LEARN", this.pluginLang.readPluginField("/APPLANG/SETUP/BUTTONS/LEARN"));
								break;
						}
					}
					break;
					#endregion

				#region MINIMUM TIRE PRESSURE - change it
				case "PMIN1":
				case "PMIN2":
				case "PMIN3":
				case "PMIN4":
				case "PMIN5":
					// fist get the value,which should be saved to the config.xml file
					OldValue = tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/PRESSURE_MIN/TIRE" + btnStr[4]).InnerText;
					res = GetNewPressure(OldValue,"PMIN");
					// the convert this value to a normalized string 
					this.CF_updateButtonText(btnStr, PressureToString(ref res));
					tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/PRESSURE_MIN/TIRE" + btnStr[4]).InnerText = Convert.ToString(res);
					break;
					#endregion

				#region MAXIMUM TIRE PRESSURE - change it
				case "PMAX1":
				case "PMAX2":
				case "PMAX3":
				case "PMAX4":
				case "PMAX5":
					OldValue = tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/PRESSURE_MAX/TIRE" + btnStr[4]).InnerText;
					res = GetNewPressure(OldValue,"PMAX");
					this.CF_updateButtonText(btnStr, PressureToString(ref res));
					tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/PRESSURE_MAX/TIRE" + btnStr[4]).InnerText = Convert.ToString(res);
					break;
					#endregion

				#region MAXIMUM TIRE TEMPERATURE
				case "TMAX1":
				case "TMAX2":
				case "TMAX3":
				case "TMAX4":
				case "TMAX5":
					OldValue = tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/TEMPERATURE_MAX/TIRE" + btnStr[4]).InnerText;
					res = GetNewTemperature(OldValue,"TMAX");
					this.CF_updateButtonText(btnStr, TemperatureToString(ref res));
					tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/TEMPERATURE_MAX/TIRE" + btnStr[4]).InnerText = Convert.ToString(res);
					break;
					#endregion

                #region TIRE SENSOR ID RENAMING
                case "LOC1": 
                case "LOC2":
                case "LOC3":
                case "LOC4":
                case "LOC5":
                    if (this.CF_systemDisplayDialog(CF_Dialogs.YesNo, "Mchten Sie die Sensornummer fr diesen Reifen ndern?") == DialogResult.Yes)
                    { 
                        // btnStr enthlt Namen des gedrckten Buttons
                        OpenAssigment();
                    }
                    break;
                #endregion

                #region TIRE SENSOR ASSIGNMENT
                case "TIRE1":
				case "TIRE2":
				case "TIRE3":
				case "TIRE4":
				case "TIRE5":
					//---------------------------------------------------------------------
					// stop a running flasher timer & reset a highlighted tire button
					// choose a new tire to learn and restart flasher timer and learn timer
					//---------------------------------------------------------------------
					tmrFlasher.Stop();
					int NewTire = Convert.ToInt16(btnStr.Remove(0,4));

					if (LearnTire != 0) this.CF_setButtonOff("LOC" + Convert.ToString(LearnTire));
					if (LearnTire != NewTire)
					{
						LearnTire = NewTire;
						if (id == null) id = new info.Infodisplay(this.currentSkin);
						string strMessage = this.pluginLang.readPluginField("/APPLANG/SETUP/MSG_TIRE_SELECTED");
						strMessage = strMessage.Replace("%1", LearnTire.ToString());
						id.ShowInfo(strMessage,2000);
							
						// now select the stage specific handling
						switch (aStage)
						{
							case AssignStage.OFF:
								// start a flasher timer to show, which tire is to be paired.
								tmrFlasher.Start();	
								aStage = AssignStage.STARTED;
								break;

							case AssignStage.STARTED:
								tmrFlasher.Start();	
								break;

							case AssignStage.ASSIGNED:
								tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/SENSORLOCATION/" + btnStr).InnerText = LearnTire.ToString();
								this.CF_setButtonOff("LOC" + LearnTire.ToString());
								aStage = AssignStage.OFF;
								LearnTire = 0;				// set the learn tire back to undefined
								break;
						}
						Invalidate();
					}
					break;
					#endregion
			
				#region SELECT TEMP UNIT
				case "TUNIT":
					switch (tpms.TemperatureUnit)
					{
						case tpms.Temperature.C:
							tpms.TemperatureUnit = tpms.Temperature.F;
							tpms.TemperatureSuffix = "F";
							tpms.TemperatureText = this.pluginLang.readPluginField("/APPLANG/SETUP/FTEMP");
							break;

						case tpms.Temperature.F:
							tpms.TemperatureUnit = tpms.Temperature.C;
							tpms.TemperatureSuffix = "C";
							tpms.TemperatureText = this.pluginLang.readPluginField("/APPLANG/SETUP/CTEMP");
							break;
					}
					tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/TEMPERATUREUNIT").InnerText = tpms.TemperatureUnit.ToString();
					this.CF_updateButtonText("TUNIT", tpms.TemperatureText);
					// update the displayed values...
					UpdatePressureTempValues();
					break;
					#endregion

				#region PRESSURE UNIT
					case "PUNIT":
						switch (tpms.PressureUnit)
						{
							case tpms.Pressure.KPa: 
								tpms.PressureUnit = tpms.Pressure.Bar;
								break;
							case tpms.Pressure.Bar: tpms.PressureUnit = tpms.Pressure.PSI;
								break;
							case tpms.Pressure.PSI: tpms.PressureUnit = tpms.Pressure.KPa;
								break;
						}
						tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/PRESSUREUNIT").InnerText = tpms.PressureUnit.ToString();
						this.CF_updateButtonText("PUNIT", tpms.configxml.SelectSingleNode("/APPCONFIG/SETUP/PRESSUREUNIT").InnerText);
						// update the displayed values...
						UpdatePressureTempValues();
						break;
					#endregion

				#region BUTTON 1 - DISPLAYNAME
						//======================================================================================================
						// Set the displayname of this plugin
						//======================================================================================================
					case "BUTTON1":
						if(this.CF_systemDisplayDialog(CF_Dialogs.OSK, this.pluginLang.readPluginField("/APPLANG/SETUP2/LABELS/LABEL1"), this.CF_getButtonText("BUTTON1"), out resultvalue, out resulttext) == DialogResult.OK)
						{
							this.CF_updateButtonText("BUTTON1", resultvalue);
							tpms.languagexml.SelectSingleNode("/APPLANG/DISPLAYNAME").InnerText = HttpUtility.HtmlEncode(resultvalue);
						}
						break;
						#endregion
	
				#region BUTTON 2 - DETECTION INTERVALL TIME
						//======================================================================================================
						// Set the number of relay connected to the serial interface
						//======================================================================================================
					case "BUTTON2":
						if(this.CF_systemDisplayDialog(CF_Dialogs.NumberPad,
							this.pluginLang.readPluginField("/APPLANG/SETUP2/LABEL2"),
							this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLMS"), 
							out resultvalue, out resulttext) == DialogResult.OK)
						{
							int resVal = Convert.ToInt32( resultvalue);
							if ((Convert.ToInt32( resultvalue) <  1000) || (Convert.ToInt32( resultvalue) > 60000))
								resultvalue = Convert.ToString(10000);
                            else
                                resultvalue = Convert.ToInt32(resultvalue).ToString();

							this.CF_updateButtonText("BUTTON2", resultvalue);
							tpms.configxml.SelectSingleNode("/APPCONFIG/INTERVALLMS").InnerText = resultvalue;
						}
						break;
						#endregion

				#region BUTTON 3 - flash time
					case "BUTTON3":
						if(this.CF_systemDisplayDialog(CF_Dialogs.NumberPad,
							this.pluginLang.readPluginField("/APPLANG/SETUP2/LABEL3"),
							this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLFLASHMS"), 
							out resultvalue, out resulttext) == DialogResult.OK)
						{
							int resVal = Convert.ToInt32( resultvalue);
                            if ((Convert.ToInt32(resultvalue) < 100) || (Convert.ToInt32(resultvalue) > 3000))
                                resultvalue = Convert.ToString(500);
                            else
                                resultvalue = Convert.ToInt32(resultvalue).ToString();

							this.CF_updateButtonText("BUTTON3", resultvalue);
							tpms.configxml.SelectSingleNode("/APPCONFIG/INTERVALLFLASHMS").InnerText = resultvalue;
						}
						break;
				#endregion

				#region BUTTON 4 - open tire sensor assigment dialog 
                    case "BUTTON4":
                        OpenAssigment();    
                        break;
				#endregion

				#region BUTTON 5 - ACTIVATE
					case "BUTTON5ON":
						this.CF_setButtonOn("BUTTON5ON");
						this.CF_setButtonOff("BUTTON5OFF");
						tpms.configxml.SelectSingleNode("/APPCONFIG/ACTIVE").InnerText = "TRUE";
						break;
					case "BUTTON5OFF":
						this.CF_setButtonOff("BUTTON5ON");
						this.CF_setButtonOn("BUTTON5OFF");
						tpms.configxml.SelectSingleNode("/APPCONFIG/ACTIVE").InnerText = "FALSE";
						break;
						#endregion

				#region BUTTON 6 - USEDEBUG
					case "BUTTON6ON":
						this.CF_setButtonOn("BUTTON6ON");
						this.CF_setButtonOff("BUTTON6OFF");
						tpms.configxml.SelectSingleNode("/APPCONFIG/USEDEBUG").InnerText = "TRUE";
						break;
					case "BUTTON6OFF":
						this.CF_setButtonOff("BUTTON6ON");
						this.CF_setButtonOn("BUTTON6OFF");
						tpms.configxml.SelectSingleNode("/APPCONFIG/USEDEBUG").InnerText = "FALSE";
						break;
						#endregion

				#region BUTTON 7 - FORCED DEVICE DETECTION
					case "BUTTON7ON":
						this.CF_setButtonOn("BUTTON7ON");
						this.CF_setButtonOff("BUTTON7OFF");
						tpms.configxml.SelectSingleNode("/APPCONFIG/ACTIVE").InnerText = "TRUE";
						break;
					case "BUTTON7OFF":
						this.CF_setButtonOff("BUTTON7ON");
						this.CF_setButtonOn("BUTTON7OFF");
						tpms.configxml.SelectSingleNode("/APPCONFIG/ACTIVE").InnerText = "FALSE";
						break;
						#endregion

                #region BUTTON 8 - USE 5 THAN 4 TIRES
                    case "BUTTON8ON":
						this.CF_setButtonOn("BUTTON8ON");
						this.CF_setButtonOff("BUTTON8OFF");
                        tpms.configxml.SelectSingleNode("/APPCONFIG/USE5TIRES").InnerText = "TRUE";
						break;

					case "BUTTON8OFF":
						this.CF_setButtonOff("BUTTON8ON");
						this.CF_setButtonOn("BUTTON8OFF");
                        tpms.configxml.SelectSingleNode("/APPCONFIG/USE5TIRES").InnerText = "FALSE";
						break;
                #endregion

			}
            if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog); 
		}

        private void OpenAssigment()
        {
            // =========================================
            // Open dialog to define sensor ID of tires
            // =========================================

            DialogResult returnvalue = DialogResult.Cancel;
            try
            {
                // Creates a new plugin setup instance. If you create a CFDialog or CFSetup you must set 
                // its MainForm property to the main plugins MainForm property.
                sensorpos SP = new sensorpos();
                SP.MainForm = this.MainForm;
                SP.currentSkin = this.currentSkin;
                SP.CF_pluginInit();
                returnvalue = SP.ShowDialog();
                SP.Close();
                SP = null;
            }
            catch (Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace, tpms.errLog); }
	     }

		private byte GetNewPressure(string old, string label)
		{
            if (tpms.useDebug) CFTools.writeModuleLog("START", tpms.errLog); 
			
			double temp=0;
			byte b = 0;

			try	{b = Convert.ToByte(old);}
			catch {	b=0;}

			string lastvalue = string.Empty;
			string resultvalue, resulttext;

			switch (tpms.PressureUnit)
			{
				case tpms.Pressure.KPa:
					lastvalue =  String.Format("{0:F1} ", b * 2.5 - 100).Trim()	;
					break;
				case tpms.Pressure.Bar:
					lastvalue =  String.Format("{0:F1} ", ((b * 2.5) - 100) / 100).Trim();
					break;
				case tpms.Pressure.PSI:
					lastvalue = String.Format("{0:F0} ", ((b * 2.5) - 100) * 0.145).Trim();
					break;
			}
			string txt = this.pluginLang.readPluginField("/APPLANG/SETUP/" + label);
			txt = txt.Replace("%", "[" + tpms.PressureSuffix + "]" );

			// ready for opening the number dialog			
			if(this.CF_systemDisplayDialog(CF_Dialogs.NumberPad, txt,
				lastvalue, out resultvalue, out resulttext) == DialogResult.OK)
			{ 
				resultvalue = resultvalue.Replace(".",",");
				switch (tpms.PressureUnit)
				{
					case tpms.Pressure.KPa:
						temp = (Convert.ToDouble(resultvalue) + 100) / 2.5;
						break;
					case tpms.Pressure.Bar:
						temp = (((Convert.ToDouble(resultvalue) * 100) + 100) / 2.5);
						break;
					case tpms.Pressure.PSI:
						temp = (((Convert.ToDouble(resultvalue) / 0.145) + 100) /2.5);
						break;
				}
				if (temp <= 254) return Convert.ToByte(temp);
				else return 254;
			}
			else
			{
                if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog); 
				return b;
			}
		}

		public void ShowLargeMessage(string txt)
		{
			message Msg = null;
			DialogResult returnvalue = DialogResult.OK;
			try
			{
				// Creates a new plugin setup instance. If you create a CFDialog or CFSetup you must set 
				// its MainForm property to the main plugins MainForm property.
				if (Msg == null) Msg = new message();
				Msg.MainForm = this.MainForm;
				Msg.currentSkin = this.currentSkin;
				Msg.MESSAGETEXT = txt;
				Msg.CF_pluginInit();
				returnvalue = Msg.ShowDialog();
				Msg.Close();
			}
			catch(Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); }
		}

		private byte GetNewTemperature(string old, string label)
		{
            if (tpms.useDebug) CFTools.writeModuleLog("START", tpms.errLog); 
			
			byte b = 0;
			double temp=0;

			try {b = Convert.ToByte(old);}
			catch { b=0; }

			string lastvalue = string.Empty;
			string resultvalue, resulttext;

			switch (tpms.TemperatureUnit)
			{
				case tpms.Temperature.C:
					lastvalue =  String.Format("{0:F0} ", b - 40).Trim();
					break;
				case tpms.Temperature.F:
					lastvalue =  String.Format("{0:F0} ", ((b - 40) * 9 / 5) + 32).Trim();
					break;
			}

			string txt = this.pluginLang.readPluginField("/APPLANG/SETUP/" + label);
			txt = txt.Replace("%", "[" + tpms.TemperatureSuffix + "]");

			// ready for opening the number dialog			
			if(this.CF_systemDisplayDialog(CF_Dialogs.NumberPad, txt, 
				lastvalue, out resultvalue, out resulttext) == DialogResult.OK)
			{
				resultvalue = resultvalue.Replace(".",",");
			
				switch (tpms.TemperatureUnit)
				{
					case tpms.Temperature.C:
						temp = Convert.ToDouble(resultvalue) + 40;
						break;
					case tpms.Temperature.F:
						temp = (Convert.ToDouble(resultvalue) - 32)* 5 / 9 + 40;
						break;
				}
				if (temp <= 254) return Convert.ToByte(temp);
				else return 254;
			}
		else
			{
                if (tpms.useDebug) CFTools.writeModuleLog("END", tpms.errLog); 
				return b;
			}
		}

		#endregion
	}
}
