using System;

namespace tpms
{
	public class TpmsDevice: HIDDevice
	{
		private volatile TpmsInputReport inputReport = null;
        private volatile bool bRemoved = false;

		public TpmsDevice()
		{
		}

		public override InputReport CreateInputReport()
		{
			return new TpmsInputReport(this);
		}

		/// <summary>
		/// Fired when data has been received over USB
		/// </summary>----------------------------------------------------------------------------------------------------------

		/// <param name="oInRep">Input report received</param>
		protected override void HandleDataReceived(InputReport oInRep)
		{
			inputReport = (TpmsInputReport)oInRep;
		}

        protected override void HandleDeviceRemoved() { bRemoved = true; }
        protected override void HandleDeviceArrived() { bRemoved = false; }

		public TpmsInputReport ReadReport()
		{
            if (bRemoved) return null;

			// Wait for an input report to arrive (busy waiting - bad...)
			inputReport = null;
			BeginAsyncRead();

            for (int Count = 1; (Count < 10) && (inputReport == null); Count++)
			    System.Threading.Thread.Sleep(100);

            return inputReport;
		}

		public void WriteReport(TpmsOutputReport outputReport)
		{
            if (bRemoved) return;

			try
			{
				// Reset the input report so the next call to ReadReport has new data
				inputReport = null;

				// Write data
				Write(outputReport);
			}
			catch(HIDDeviceException)
			{
				// 0.4.0.6
				this.Dispose(); // Device may have been removed!
				
			}
		}

		/// <summary>
		/// Finds the TMS USB Receiver.
		/// Caution: VID 0000 and PID 0001 looks suspicious.
		/// </summary>
		/// <returns>A new TpmsDevice or null if not found.</returns>
		public static TpmsDevice FindTpmsDevice()
		{
			// VID and PID for Tpms device are 0x0000 and 0x0001, respectively
			return (TpmsDevice)FindDevice(0x0000, 0x0001, typeof(TpmsDevice));
		}

	}

	public class TpmsInputReport: InputReport
	{
		public TpmsInputReport(HIDDevice dev): base(dev) {}

		// We don't parse anything here at the moment
		public override void ProcessData()
		{
		}
	}

	public class TpmsOutputReport: OutputReport
	{
		public TpmsOutputReport(HIDDevice dev): base(dev) {}
	}
}
