/* ==================================================================================================
 * TPMS plugin
 * File: tpms.cs
 * Author: Dipl.-Ing. Wolfgang Otto
 * eMail: Wolfgang@exner-otto.de
 * Date: November 2007
 * 
 * THIS PLUGIN IS A CONVERSION OF THE MARVELESS TOOL OF
 * **************************************************************************************************
 * Roberto Montanari, RoadRunner TMS Plugin															*
 * His VB sources are converted to C# and modified to be used as a CENTRAFUSE plugin				*
 * Thanks for his great work!																		*
 * http://www.mp3car.com/vbulletin/rr-plugins/110330-rrtms-roadrunner-tire-monitoring-sistem.html	*
 * **************************************************************************************************
 * This file is subject to the condition of GNU GENERAL PUBLIC LICENSE								*
 * Therefore you are free to use this software for your own requirements and use the				*
 * source code for any modification.																*	
 * The only wish is to inform the author about any kind of development.								*
 * =================================================================================================*/

// to use the simulation mode change the complilation directive to SIMULATION
#define NODEVELOP

using System;
using System.Drawing;
using System.Windows.Forms;
using System.Xml;
using System.IO;
using System.Data;
using System.Data.OleDb;
using System.Web;
using System.Collections;
using System.Threading;
using System.Text;
using System.ComponentModel;
using System.Runtime.InteropServices;
using System.Drawing.Drawing2D;
using SpeechLib;
using System.Reflection;
using System.Diagnostics;
using centrafuse.Plugins;


namespace tpms
{	

	public class tpms : CFPlugin
	{
		// Global variables
		public enum Reason {Normal = 0, TemperatureChange = 1, PressureChange = 2}
		public enum LrnStage {Off = 0, Started = 1, Waiting = 2}
		
		public static string PressureSuffix;
		public static string TemperatureSuffix;
		
		public enum Pressure {KPa = 0,	Bar = 1, PSI = 2}
		public static Pressure PressureUnit;

		public enum Temperature {C = 0, F = 1}
		public static Temperature TemperatureUnit;
		public static string TemperatureText;

		protected System.Windows.Forms.Timer checkTimer;
		protected System.Windows.Forms.Timer flashTimer;
		
		public struct TiresData
		{
			public bool Alert;
			public bool IsSilent;
			public byte Sensor;
			public byte Pressure;
			public byte PressureOLD;		// Placeholder of an average pressure information, to detect a quick value change
			public byte Temperature;
			public byte TemperatureOLD;		// Placeholder of an average temperature information
			public byte Battery;
			public byte BatteryOLD;			// Placeholder of an average battery status information
			public Reason TxReason;
			public int MinPressure;
			public int MaxPressure;
			public byte MaxTemperature;
			public string MinPressureMessage;
			public string MaxPressureMessage;
			public string MaxTemperatureMessage;
			
		}
		
		public static TiresData[] Tire = new TiresData[5];

		public bool SilentMode = false;		// global default to give an acoustic warn alert signal for tire alerts
		private bool BlinkOn = false;

		public bool PressureAlert;
		public bool TemperatureAlert;

		#region PAGE and CONFIG handling
		public static int  LENLABEL5_8 = 30;
		public static int  LENLABEL1_4 = 40;
		public static string pluginFolder = "plugins/tpms";
        public static string pluginCfg = "plugins/tpms/config.xml";
        public static string errLog = Application.StartupPath + @"\plugins\tpms\tpms.log";
		#endregion
		
		private Thread OutputTexttoSpeechThread = null;
		private string ALERTMSG = "";
		private bool VOICE_IS_ON = false;
		private bool ForceDetection = false;
        private bool Use5Tires = true;
        public static bool bHeaderWritten = false;          // Date/Time Information header for error log file
 
		private volatile TpmsOutputReport outputReport = null;
		private volatile TpmsInputReport inputReport = null;

		
		#region PROFILING CONFIGURATION VARIABLES
		
		private static string	profName = string.Empty;
		private static bool	profLog = false;
		private static StreamWriter streamProfile;
		
		#endregion

		public setup tpmssetup = null;
		
		// TPMS USB Device
		public static TpmsDevice device = null;
		// Set these to match the values in the device's firmware.
		public const short MyVendorID = 0x0;
		public const short MyProductID = 0x1;
		
		//  XML documents used to load configuration data.
		public static XmlDocument languagexml = new XmlDocument();
		public static XmlDocument configxml = new XmlDocument();

		public static bool useDebug = false;
		public static bool IsAlert = false;

		public string PressureToString(ref byte pressure)
		{      
			string res = string.Empty;

			if (pressure == 0xFF) 
			{
				res = "--- " + PressureSuffix;
				return res;
			}

			if (pressure <= 40)
			{
				res =  String.Format("{0:F1} ", 0) + PressureSuffix;
				return res;
			}
						
			switch (PressureUnit)
			{
				case Pressure.KPa:
					res =  String.Format("{0:F1} ", (pressure * 2.5) - 100) + PressureSuffix;
					break;
				case Pressure.Bar:
					res =  String.Format("{0:F1} ", (pressure * 2.5 - 100) / 100) + PressureSuffix;
					break;
				case Pressure.PSI:
					res = String.Format("{0:F0} ", ((pressure * 2.5) - 100) * 0.145) + PressureSuffix;
					break;
			}
			return res;
		}
		public string TemperatureToString(ref byte temp)
		{
			string res = string.Empty;
			
			if (temp == 0xFF) 
			{
				res = "--" + TemperatureSuffix;
				return res;
			}

			if (temp <= 40)
			{
				res =  String.Format("{0:F0}", 0) + TemperatureSuffix;
				return res;
			}

			switch (TemperatureUnit)
			{
				case Temperature.C:
					res =  String.Format("{0:F0}", temp - 40) + TemperatureSuffix;
					break;
				case Temperature.F:
					res =  String.Format("{0:F0}", ((temp - 40) * 9 / 5) + 32) + TemperatureSuffix;
					break;
			}
            			
			return res;
		}
		public static string LimitLength (string input, int maxLength)
		{
			if (input.Length > maxLength)
			{
				int cBegin, cEnd;
				string sMiddle = "...";
				cBegin = (int)System.Math.Floor(0.5 * maxLength);
				cEnd = maxLength - cBegin - sMiddle.Length;
                return input.Substring(0, cBegin) + sMiddle + input.Substring(input.Length - cEnd, cEnd);
			}
			else
			{
				if (useDebug ==true) CFTools.writeModuleLog("END", tpms.errLog);
				return input;
			}
		}
		public tpms()
		{
			this.CF_pluginPauseAudioFlag = false;
			this.CF_pluginName = "TPMS";
			this.CF_pluginIsGUI = true;
		}
        public bool HookHardwareNotifications(IntPtr callback, bool UseWindowHandle)
        {
            try
            {
                Win32Usb.DeviceBroadcastInterface dbdi = new Win32Usb.DeviceBroadcastInterface();
                dbdi.Size = Marshal.SizeOf(dbdi);
                dbdi.Reserved = 0;
                dbdi.DeviceType = Win32Usb.DBT_DEVTYP_DEVICEINTERFACE;
                if (UseWindowHandle)
                {
                    Win32Usb.RegisterDeviceNotification(callback,
                        dbdi,
                        Win32Usb.DEVICE_NOTIFY_ALL_INTERFACE_CLASSES |
                        Win32Usb.DEVICE_NOTIFY_WINDOW_HANDLE);
                }
                else
                {
                    Win32Usb.RegisterDeviceNotification(callback,
                        dbdi,
                        Win32Usb.DEVICE_NOTIFY_ALL_INTERFACE_CLASSES |
                        Win32Usb.DEVICE_NOTIFY_SERVICE_HANDLE);
                }
                return true;
            }
            catch (Exception ex)
            {
                string err = ex.Message;
                return false;
            }
        }
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32Usb.WM_DEVICECHANGE:
                    {
                        if (useDebug) CFTools.writeModuleLog("ReceiverDevice_OnDeviceChanged", tpms.errLog);
                        checkTimer.Enabled = true;
                        break;
                    }
            }
            base.WndProc(ref m);
        }
        public void WriteProfile(string txt)
		{
			//------------------------------------------------------------
			// The following events are logged to the file
			// 
			// - SayMessage(msg)
			// - SETUP-tmrLearning_Timer
			// - ???
			//------------------------------------------------------------
			if (profName != string.Empty)
			{
				if (streamProfile == null) // not yet opened
				{
					// Determine whether the file exists.
					streamProfile = new StreamWriter(profName, File.Exists(profName), System.Text.Encoding.UTF8);
					streamProfile.WriteLine(this.pluginLang.readPluginField("APPLANG/VERSIONNAME") + " " + Assembly.GetExecutingAssembly().GetName().Version.ToString());
				}
				#region WRITE OUTPUT
				// write information to file
				streamProfile.Write(DateTime.Now.ToShortDateString() + "-" + DateTime.Now.ToLongTimeString()+ " --> ");
				streamProfile.WriteLine(txt);
				streamProfile.Flush();
				#endregion
			}
		}
		
		public void LoadValues()
		{
			if (useDebug ==true) CFTools.writeModuleLog("START", tpms.errLog);
			//----------------------------------------------------------------
			// Read config from .xml file for each tire 1-5 (or better 0 to 4)
			//----------------------------------------------------------------
			for (int n=1; n<=5; n++)
			{
                // This value is to assign a sensor to a mounting car position
                // The user can select which sensor is in which place. 
                // As I see it right now, when you move the wheels around during normal rotation, 
                // user would have to clear the PIC using TPMS doctor and relearn ? or keep the sensors 
                // in the same places which is a pain if you just want to move wheels front to back or corner to corner.
                try
                {
                    Tire[n - 1].Sensor = Convert.ToByte(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/SENSORLOCATION/TIRE" + n.ToString()));
                }
                catch 
                {
                    Tire[n - 1].Sensor = Convert.ToByte(n);
                }
				Tire[n-1].MinPressure = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/PRESSURE_MIN/TIRE" + n.ToString()));
				try {Tire[n-1].MaxPressure = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/PRESSURE_MAX/TIRE" + n.ToString()));}
				catch {Tire[n-1].MaxPressure = 160;}
				try {Tire[n-1].MaxTemperature = Convert.ToByte(this.pluginConfig.readPluginField("/APPCONFIG/SETUP/TEMPERATURE_MAX/TIRE" + n.ToString()));} 
				catch {Tire[n-1].MaxTemperature = 120;}
				Tire[n-1].MinPressureMessage = this.pluginLang.readPluginField("/APPLANG/SETUP/MIN_PRESSURE_MSG/TIRE" + n.ToString());
				Tire[n-1].MaxPressureMessage = this.pluginLang.readPluginField("/APPLANG/SETUP/MAX_PRESSURE_MSG/TIRE" + n.ToString());
				Tire[n-1].MaxTemperatureMessage = this.pluginLang.readPluginField("/APPLANG/SETUP/MAX_TEMP_MSG/TIRE" + n.ToString());
			}
			try 
			{
				PressureUnit = (Pressure) Enum.Parse(typeof(Pressure),this.pluginConfig.readPluginField("/APPCONFIG/SETUP/PRESSUREUNIT"),true);
			}
			catch {PressureUnit = 0;}

			//----------------------------------------------------------------
			// Pressure Unit
			//----------------------------------------------------------------
			switch (PressureUnit)
			{
				case Pressure.KPa: PressureSuffix = "KPa"; break;
				case Pressure.Bar: PressureSuffix = "Bar"; break;
				case Pressure.PSI: PressureSuffix = "PSI"; break;
			}
            
			//----------------------------------------------------------------
			// force device detection before enabling this plugin
			//----------------------------------------------------------------
			try 
			{
				ForceDetection = (this.pluginConfig.readPluginField("/APPCONFIG/FORCEDETECT").Trim().ToUpper() == "TRUE");
			}
			catch(Exception errmsg)	
			{
				CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); 
				ForceDetection = false;
			}
            
			//----------------------------------------------------------------
			// load profiling settings 
			//----------------------------------------------------------------
			try 
			{
				if (this.pluginConfig.readPluginField("/APPCONFIG/PROFILE/USEPROFILE").Trim().ToUpper() == "TRUE")
					profName = Application.StartupPath + "\\plugins\\tpms\\profillog.txt";
				profLog = (this.pluginConfig.readPluginField("/APPCONFIG/PROFILE/PERMANENTLOG").Trim().ToUpper() == "TRUE");
			}
			catch(Exception errmsg)	
			{
				CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); 
				profName = string.Empty;
				profLog = false;
			}
            
			// Temperature unit
			try 
			{
				TemperatureUnit = (Temperature) Enum.Parse(typeof(Temperature),this.pluginConfig.readPluginField("/APPCONFIG/SETUP/TEMPERATUREUNIT"),true);
			}
			catch(Exception errmsg)	
			{
				CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); 
				TemperatureUnit = 0;
			}
            
			switch (TemperatureUnit)
			{
				case Temperature.C:
					TemperatureSuffix = "C";
					TemperatureText = this.pluginLang.readPluginField("/APPLANG/SETUP/CTEMP");
					break;

				case Temperature.F:
					TemperatureSuffix = "F";
					TemperatureText = this.pluginLang.readPluginField("/APPLANG/SETUP/FTEMP");
					break;
			}
            
			try { checkTimer.Interval = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLMS")); } 
			catch {checkTimer.Interval = 5000;} // standard value is 5 seconds
            
			try { flashTimer.Interval = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLFLASHMS")); } 
			catch {flashTimer.Interval = 500;} // standard value is 1/2 seconds
            
            try { Use5Tires = Convert.ToBoolean(this.pluginConfig.readPluginField("/APPCONFIG/USE5TIRES")); }
            catch { }
            
			if (useDebug ==true) CFTools.writeModuleLog("START", tpms.errLog);			
		}
		private void OutputTexttoSpeech()
		{
			//----------------------------------------------------------------
			// Thread for alert messaging output
			// During alert message output the common audio output is set to mute
			// to enable a better recognizing of the voice output
			//----------------------------------------------------------------
            SpVoice spVoice = new SpVoice();
			try
			{
				this.CF_systemCommand(CF_Actions.MUTEAUDIO);
				
				spVoice.Speak(ALERTMSG, new SpeechVoiceSpeakFlags());
				do 
				{
					Application.DoEvents();
				}
				while (!spVoice.WaitUntilDone(10));
				this.CF_systemCommand(CF_Actions.UNMUTEAUDIO);
			}
			catch(Exception)	
			{
				this.CF_systemCommand(CF_Actions.UNMUTEAUDIO);
			}
			finally
			{
				//if (spVoice != null) ((IDisposable)spVoice).Dispose();
				VOICE_IS_ON = false;
			}
		}
		private void SayMessage(string msg)
		{
			// Use TextToSpeech a little bit later to give a vocal message out
			if (!VOICE_IS_ON)
			{
				//WriteProfile(msg); // Write to Profile file
				ALERTMSG = msg; // to give the string to a thread
				if ((ALERTMSG != "") && (ALERTMSG != null))
				{
                    try { checkTimer.Interval = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/ALERTREPEATTIME")); }
                    catch { checkTimer.Interval = 5000; } // standard value is 5 seconds

					VOICE_IS_ON = true;
					OutputTexttoSpeechThread = new Thread(new ThreadStart(OutputTexttoSpeech));
					OutputTexttoSpeechThread.Start();
				}
			}
		}
		private void flashTimer_Tick(object sender, EventArgs e)
		{
            int max = 5;
			//===========================================================
			// let the tire symbol flash with an intervall of 1/3 second
			//===========================================================
			flashTimer.Stop();

            if (Use5Tires == false) 
                max = 4;

            // Detect the current ON/OFF status of each tire. 
            // 
            // Attention: the tire number is detected out of
            // the config paramater <SENSORLOCATION> for each tire.
            // you can assign the tire number to a free sensor location
            // i.e. TIRE1 was formertimes mounted at the front left position, now you have moved it to the rear 
            // left position TIRE1(OLD) = 1, TIRE1(NEW) = 3
                
            for (int n=1; n<=max;n++)
			{
                if ((Tire[n-1].Alert) && (BlinkOn))
					this.CF_setButtonOn("TIRE" + Convert.ToString(Tire[Tire[n-1].Sensor - 1]));
				else
                    this.CF_setButtonOff("TIRE" + Convert.ToString(Tire[Tire[n - 1].Sensor - 1]));
				Invalidate();
			}
			BlinkOn = !BlinkOn;
			
			flashTimer.Start();
		}
		public void ShowLargeMessage(string txt)
		{
            DialogResult returnvalue = DialogResult.Cancel;
			try
			{
				// Creates a new plugin setup instance. If you create a CFDialog or CFSetup you must set 
				// its MainForm property to the main plugins MainForm property.
				message Msg = new message();
				Msg.MainForm = this.MainForm;
				Msg.currentSkin = this.currentSkin;
				Msg.MESSAGETEXT = txt;
				Msg.CF_pluginInit();
				returnvalue = Msg.ShowDialog();
				Msg.Close();
				Msg = null;
			}
			catch(Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); }
		}
		private void Btn_Click(object sender, MouseEventArgs e)
		{
			if (useDebug ==true) CFTools.writeModuleLog("START", tpms.errLog);

			CFControls.skinButton tmp = new centrafuse.Plugins.CFControls.skinButton();
			tmp = (CFControls.skinButton)sender;
			switch (tmp.Name.ToString())
			{
				case "SILENT":
					SilentMode = !SilentMode;
					if (SilentMode)
						this.CF_updateButtonText("SILENT", this.pluginLang.readPluginField("/APPLANG/TPMS/BUTTONS/NOSILENT"));
					else
						this.CF_updateButtonText("SILENT", this.pluginLang.readPluginField("/APPLANG/TPMS/BUTTONS/SILENT"));
					
					for (int i=1; i<=5 ; i++)
					{
						Tire[i-1].IsSilent = (SilentMode && Tire[i-1].Alert);
					}
					break;
			}
			if (useDebug ==true) CFTools.writeModuleLog("EMD", tpms.errLog);
		}		
		protected void checkTimer_Tick(object sender, EventArgs e)
		{
			checkTimer.Stop();
			if (this.pluginConfig.readPluginField("/APPCONFIG/ACTIVE").ToUpper() == "TRUE")
				GetDataFromDevice();
			checkTimer.Start();         
		}
		public string ReturnBatteryImage(int tirenumber)
		{
            byte bv = Tire[tirenumber].Battery;
			int lfd = 0;
			string s = string.Empty;

			/*****************************************************************
			* This Function will be called with requested tire number to get 
			* a specified battery loading image out of the skin path.
			* ***************************************************************/
			
			if (bv == 0)	lfd = 0;
			else			lfd = Convert.ToByte(Math.Round((Convert.ToDecimal(bv) / 100 * 21),0));
			
			//------------------------------
			//error handling new in 0.4.1.2
			if (lfd > 20) lfd = 20;
			if (lfd < 0) lfd = 0;
            //------------------------------
			s = pluginFolder + "/skins/" + this.currentSkin + "/BATTERY/battery_" +lfd.ToString() + ".gif";
			if (!File.Exists(s)) CFTools.writeModuleLog("File: " + s + " not found",tpms.errLog);
			return s;
		}
        private void OpenDevice()
        {
            try
            {
                device = TpmsDevice.FindTpmsDevice();
            }
            finally { }

            if (device != null)
            {
                device.OnDeviceRemoved += new EventHandler(ReceiverDevice_OnDeviceRemoved);
                device.OnDeviceArrived += new EventHandler(ReceiverDevice_OnDeviceArrived);
            }
        }
		private void GetDataFromDevice()
		{
			string s = string.Empty;
            int maxTires = 4;
            if (Use5Tires == true) maxTires = 5;
 
            if (tpms.useDebug == true) CFTools.writeModuleLog("START",tpms.errLog);
			//======================================================================================
			// This routine detects a pressure or temperature problem with one of the wheels (tires)
			//======================================================================================
			// Sends one byte to the device and reads four bytes back.
			
			// Reset alerts
			PressureAlert = false;
			TemperatureAlert = false;

			// If the device hasn't been detected or it timed out on a previous attempt
			// to access it, look for the device.
			
			try
			{
				try	{ if (device == null) OpenDevice(); }
				finally {}
               
                #region Device found

                if (device != null)
				{
					if (flashTimer.Enabled == false) flashTimer.Enabled= true;

					for (int n = 1; n<=maxTires; n++)
					{
						if ((Tire[n-1].Sensor > 0) && (Tire[n-1].Sensor < 6)) // only for assigned tire sensors
						{
							#region GET DATA FROM DEVICE
							// Load the buffer with the byte to be sent.
							if (outputReport == null) 
								outputReport = new TpmsOutputReport(device);

							outputReport.Buffer[0] = 0x0;
							outputReport.Buffer[1] = (byte)(32 + (Tire[n-1].Sensor));

							// Write the data to the device
							// and secure that this buffer is deleted after the use.
							//
							device.WriteReport(outputReport);		
							outputReport = null;
							
							// Read the data from the device.
							inputReport = device.ReadReport();

                            if (inputReport == null)
                            {
                                if (useDebug == true) CFTools.writeModuleLog("END", tpms.errLog);
                                break;
                            }

							//--------------------------------------------------------------------
							// Get pressure
							// to avoid negative values check for buffer[n] > 40 
							// if buffer[n] <= 40 then use the old result value
							//--------------------------------------------------------------------
							
							if(inputReport.Buffer[1] == 0xFF) 
								Tire[n-1].Pressure = 0xFF;								
							else
							{
								if(inputReport.Buffer[1] > 40)	Tire[n-1].Pressure = (byte)inputReport.Buffer[1];						
							}
					
							//--------------------------------------------------------------------
							// Get temperature
							// to avoid negative values check for buffer[n] > 40 
							// if buffer[n] <= 40 then use the old result value
							//--------------------------------------------------------------------
							if(inputReport.Buffer[2] == 0xFF)	
								Tire[n-1].Temperature = 0xFF;
							else
							{
								if(inputReport.Buffer[2] > 40) 		Tire[n-1].Temperature = (byte)inputReport.Buffer[2];						
							}

							//--------------------------------------------------------------------
							// Get battery status
							//--------------------------------------------------------------------
							if(inputReport.Buffer[3] == 0xFF) 	
								Tire[n-1].Battery = 0;
							else
								Tire[n-1].Battery = (byte)(inputReport.Buffer[3] & 127);
							
							//--------------------------------------------------------------------
							// Get the transmission reason
							//--------------------------------------------------------------------
							Tire[n-1].TxReason = (Reason)inputReport.Buffer[4];					
						
							//--------------------------------------------------------------------
							// Alerts handle, set ALERT per default to false for each tire
							//--------------------------------------------------------------------
							Tire[n-1].Alert = false;
							//====================================================================
							// up to this point there is no alert recognized
							// now the check data with limits is started
							//====================================================================
							#region MIN PRESSURE
							//--------------------------------------------------------------------
							// Start checking wheather one of the parameters is out of range
							// Checking MIN PRESSURE
							//--------------------------------------------------------------------
							if ((Tire[n-1].Pressure < 0xFF) && (Tire[n-1].Pressure < Tire[n-1].MinPressure)) 
							{
								// now we have a pressure alert situation
								PressureAlert = true;
								if (!Tire[n-1].Alert) // if there is no alert for this tire[n-1] up to now, make an alert warning
								{ //5
									// Show a warning message and activate the corresponding tire button
                                    this.CF_updateText("ALERTMSG",Tire[Tire[n-1].Sensor - 1].MinPressureMessage);

									//--------------------------------------------------------------------
									// Write profile information to log file output
									//--------------------------------------------------------------------
                                    if (profLog == true)
									{
										WriteProfile("MIN Pressure(raw data) of TIRE[" + Convert.ToString(n-1) + "]->data(1-4): [" + 
											inputReport.Buffer[1].ToString() + "][" +
											inputReport.Buffer[2].ToString() + "][" +
											inputReport.Buffer[3].ToString() + "][" +
											inputReport.Buffer[4].ToString() + "]");
									}
									// if a message string is available then SAY it

                                    
									if (Tire[n-1].MinPressureMessage != "") 
									{
										if ((!SilentMode) || ((SilentMode) && (Tire[n-1].IsSilent == false)))
										{
                                            SayMessage(Tire[Tire[n - 1].Sensor - 1].MinPressureMessage);
										}
									}
									Tire[n-1].Alert = true;
								}
							}
							#endregion MIN PRESSURE
							#region MAX PRESSURE
							//--------------------------------------------------------------------------
							// Checking MAX PRESSURE
							//--------------------------------------------------------------------------
							if ((Tire[n-1].Pressure < 0xFF) && (Tire[n-1].Pressure > Tire[n-1].MaxPressure))
							{ //4
								PressureAlert = true;
								if (!Tire[n-1].Alert)
								{
                                    this.CF_updateText("ALERTMSG", Tire[Tire[n - 1].Sensor - 1].MaxPressureMessage);
									//--------------------------------------------------------------------
									// Write profile information to log file output
									//--------------------------------------------------------------------

									if (profLog == true)
									{
										WriteProfile("MAX Pressure(raw data) of TIRE[" + Convert.ToString(n-1) + "]->data(1-4): [" + 
											inputReport.Buffer[1].ToString() + "][" +
											inputReport.Buffer[2].ToString() + "][" +
											inputReport.Buffer[3].ToString() + "][" +
											inputReport.Buffer[4].ToString() + "]");
									}
                                    if (Tire[Tire[n - 1].Sensor - 1].MaxPressureMessage != "") 
									{
										if ((!SilentMode) || ((SilentMode) && (Tire[n-1].IsSilent == false)))
										{
                                            SayMessage(Tire[Tire[n - 1].Sensor - 1].MaxPressureMessage);
										}
									}
									Tire[n-1].Alert = true;
								}
							}
							#endregion MAX PRESSURE
							#region MAX TEMPERATURE
							//--------------------------------------------------------------------------
							// Checking MAX TEMPERATURE
							//--------------------------------------------------------------------------
							if ((Tire[n-1].Temperature < 0xFF) && (Tire[n-1].Temperature > Tire[n-1].MaxTemperature))
							{

                                TemperatureAlert = true;
								if (!Tire[n-1].Alert)
								{
                                    this.CF_updateText("ALERTMSG", Tire[Tire[n - 1].Sensor - 1].MaxTemperatureMessage);								
									//--------------------------------------------------------------------
									// Write profile information to log file output
									//--------------------------------------------------------------------
									if (profLog == true)
									{
										WriteProfile("MAX Temperature(raw data)of TIRE[" + Convert.ToString(n-1) + "]->data(1-4): [" + 
											inputReport.Buffer[1].ToString() + "][" +
											inputReport.Buffer[2].ToString() + "][" +
											inputReport.Buffer[3].ToString() + "][" +
											inputReport.Buffer[4].ToString() + "]");
									}
                                    if (Tire[Tire[n - 1].Sensor - 1].MaxTemperatureMessage != "") 
									{
										if ((!SilentMode) || ((SilentMode) && (Tire[n-1].IsSilent == false)))
										{
                                            SayMessage(Tire[Tire[n - 1].Sensor - 1].MaxTemperatureMessage);
										}
									}
									Tire[n-1].Alert = true;
								}
							}
							#endregion
							#endregion
						}
						else
							CFTools.writeModuleLog("Tire[" + Convert.ToString(Tire[Tire[n - 1].Sensor - 1]) + "] has invalid sensor ID",tpms.errLog);
						
						//========================================================================
						//# Display to normal pressure and temperature information for each tire #
						//========================================================================
						#region DISPLAY NORMAL TIRE INFORMATION

                        this.CF_updateText("TMSG" + Convert.ToString(Tire[n - 1].Sensor),
                                PressureToString(ref Tire[n - 1].Pressure) + System.Environment.NewLine + TemperatureToString(ref Tire[n - 1].Temperature));

                        // now show the battery status symbol for each tire
                        this.CF_setPictureImage("BATTERY" + Convert.ToString(Tire[n - 1].Sensor), Image.FromFile(ReturnBatteryImage(n - 1)));
                        
                        //========================================================================
						#endregion DISPLAY NORMAL TIRE INFORMATION

					}
										
					#region display alert image
					if ((PressureAlert) || (TemperatureAlert))
					{
						//--------------------------------------------------------------------------
						// display the PRESSURE alert image
						//--------------------------------------------------------------------------
						if (PressureAlert)
						{
							s = pluginFolder + "/skins/" + this.currentSkin + "/tms_pressure.png";
							if (!File.Exists(s)) CFTools.writeModuleLog("File: " + s + " not found",tpms.errLog);
							else
								this.CF_setPictureImage("ALERT", Image.FromFile(s));
						}

						//--------------------------------------------------------------------------
						// display the TEMPERATURE alert image
						//--------------------------------------------------------------------------
						if (TemperatureAlert)
						{
							s = pluginFolder + "/skins/" + this.currentSkin + "/tms_temperature.png";
							if (!File.Exists(s)) CFTools.writeModuleLog("File: " + s + " not found",tpms.errLog);
							else
								this.CF_setPictureImage("ALERT", Image.FromFile(s));
						}

						// disable the timer controlled pop up of warning dialog to ensure the further skin handling of CF
						if (!SilentMode) this.CF_systemCommand(CF_Actions.PLUGIN, "TPMS");
						//
					}
					else
					{
						s = pluginFolder + "/skins/" + this.currentSkin + "/tms_ok.png";
						if (!File.Exists(s)) CFTools.writeModuleLog("File: " + s + " not found",tpms.errLog);
						else this.CF_setPictureImage("ALERT", Image.FromFile(s));
						this.CF_updateText("ALERTMSG", this.pluginLang.readPluginField("/APPLANG/MSGOK"));

                        try { checkTimer.Interval = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLMS")); }
                        catch { checkTimer.Interval = 5000; } // standard value is 5 seconds
					}

					#endregion
                }
                #endregion Device found

                else
				{
					#region nothing found
					//======================================================================================================
					// Device hasn't been detected.
					//======================================================================================================
                    for (int n=1; n<=5;n++)
					{
						if (Tire[n-1].Sensor > 0)
						{
							Tire[n-1].Pressure = 0xFF;;
							Tire[n-1].Temperature = 0xFF;;
							Tire[n-1].Battery = 0;
                            this.CF_setPictureImage("BATTERY" + Convert.ToString(n), Image.FromFile(pluginFolder + "/skins/" + this.currentSkin + "/BATTERY/battery_NO.gif"));
                            Tire[n-1].TxReason = Reason.Normal;
							Tire[n-1].Alert = false;
							Tire[n-1].IsSilent = false;
						}
					}
					NoUsb();
					#endregion
				} 
			}
			catch(Exception errmsg)	
			{
				CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); 
			}
		}
		private void NoUsb()
		{
            int max = 5;
            if (Use5Tires == false) max = 4;

			this.CF_setPictureImage("ALERT", Image.FromFile(pluginFolder + "/Skins/" + this.currentSkin + "/NoUsb.png"));
			this.CF_updateText("ALERTMSG",this.pluginLang.readPluginField("/APPLANG/MSGNOUSB"));
			
            for (int i=1; i <= max; i++)
			{
                this.CF_setButtonOff("TIRE" + Convert.ToString(i));
                this.CF_updateText("TMSG" + Convert.ToString(i), "---" + System.Environment.NewLine + "---");
                this.CF_setPictureImage("BATTERY" + Convert.ToString(i), Image.FromFile(pluginFolder + "/skins/" + this.currentSkin + "/BATTERY/battery_NO.gif"));
			}
		}
        private void CheckErrorHeader()
        {
            if (bHeaderWritten == false)
            {
               
                CFTools.writeLog("Test", tpms.errLog);
                bHeaderWritten = true;
            }
        }
		private void TPMS_Loaded(object sender, EventArgs e)
		{
 			//===========================================================
			// Event that fires when the aplication is loaded.
			//===========================================================
            if (useDebug == true)
            {
                CheckErrorHeader();
                CFTools.writeModuleLog("START", tpms.errLog);
            }
						
			try
			{
				//----------------------------------------------------------------------------------
				// checkTimer: Timer for detecting a pressure or temperature fault by on of the tires
				//----------------------------------------------------------------------------------
				try { checkTimer.Interval = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLMS")); } 
				catch {checkTimer.Interval = 5000;} // standard value is 5 seconds
				checkTimer.Tick += new EventHandler(checkTimer_Tick);

				try { flashTimer.Interval = Convert.ToInt16(this.pluginConfig.readPluginField("/APPCONFIG/INTERVALLFLASHMS")); } 
				catch {flashTimer.Interval = 500;} // standard value is 1/2 seconds
				flashTimer.Tick += new EventHandler(flashTimer_Tick);
				 
				//*******************************************
				// Now search for the TPMS USB receiver device
				//*******************************************
                OpenDevice();
				//*******************************************

				//
				// It is necessary to start the check timer also if no device is found 
				// to ensure that by inserted a device this could be detected
				//
				checkTimer.Enabled = true;
				if (device != null)
				{
					WriteProfile("TPMS plug-in started with USB Device");
                    flashTimer.Enabled = true;
				}
				else
				{
					// display messagebox device not found, plugin won't work
					WriteProfile("TPMS plug-in started (no USB Device)");
					NoUsb();
					if (configxml.SelectSingleNode("/APPCONFIG/ACTIVE").InnerText.ToUpper() == "TRUE")
						this.CF_systemDisplayDialog(CF_Dialogs.OkBox,this.pluginLang.readPluginField("/APPLANG/MSGNODEVICE"));
				}
			}
			catch(Exception errmsg) { CFTools.writeModuleLog("TPMS_Loaded: " + errmsg.Message, errmsg.StackTrace,tpms.errLog); }
			if (useDebug ==true) CFTools.writeModuleLog("END", tpms.errLog);
		}
		private void ReceiverDevice_OnDeviceRemoved(object sender, EventArgs e)
		{
            if (InvokeRequired)
            {
                Invoke(new EventHandler(ReceiverDevice_OnDeviceRemoved), new object[] { sender, e });
            }
            else
            {
                flashTimer.Enabled = false;
                if (tpms.useDebug) CFTools.writeModuleLog("ReceiverDevice_OnDeviceRemoved", tpms.errLog);
                WriteProfile("TPMS USB device REMOVED");
                device = null;
                NoUsb();
            }
		}
        private void ReceiverDevice_OnDeviceArrived(object sender, EventArgs e)
        {
            if (useDebug == true) { CFTools.writeModuleLog("START", tpms.errLog); }
            if (InvokeRequired)
            {
                Invoke(new EventHandler(ReceiverDevice_OnDeviceArrived), new object[] { sender, e });
            }
            else
            {
                if (useDebug) CFTools.writeModuleLog("ReceiverDevice_OnDeviceArrived", tpms.errLog);
                checkTimer.Enabled = true;
            }
            if (useDebug == true) { CFTools.writeModuleLog("END", tpms.errLog); }
        }

		//==========================================================================================
		#region CFPlugin methods
		//==========================================================================================
		public override void CF_pluginShow()
		{
			if (this.pluginConfig.readPluginField("/APPCONFIG/ACTIVE").Trim().ToUpper() == "TRUE")
			{	// Display the plugin only if it is activated
				//  This is called by the system when a button with this plugin action has been clicked.
				if (ForceDetection == true) 
				{
					this.Visible = (device != null);
				}
				else
				{
					this.Visible = true;
				}
			}
		}
		public override void CF_pluginClose()
		{
			// stop all timers
			flashTimer.Stop();
			checkTimer.Stop();

			checkTimer.Tick -= new EventHandler(checkTimer_Tick);
			flashTimer.Tick -= new EventHandler(flashTimer_Tick);
            
            // Don't forget to free this event procedure
            this.CF_Event_powerModeChanged -= new Microsoft.Win32.PowerModeChangedEventHandler(TPMS_CF_Event_powerModeChanged);

			// Close an open voice output a.s.a.p
			OutputTexttoSpeechThread.Abort();
			OutputTexttoSpeechThread = null;

			// Close the open handles to the device.
			device.Dispose();
            device = null;

			if (streamProfile != null)
			{
				streamProfile.Close();
				streamProfile.Flush();
				streamProfile = null;
			}
			this.Dispose();		// Disposes the plugin.
		}
		public override void CF_pluginUpdateLanguageSkin(string language, string skin) 
		{
			// This is called by the system when the language has changed.
            loadSettings();
            this.CF_localskinsetup();
		}
		public override DialogResult CF_pluginShowSetup()
		{
			// This is called by the system when the plugin setup is clicked.
			// Return DialogResult.OK for the main application to update from plugin changes.
			DialogResult returnvalue = DialogResult.Cancel;
			try
			{
				// Creates a new plugin setup instance. If you create a CFDialog or CFSetup you must set 
				// its MainForm property to the main plugins MainForm property.
				tpmssetup = new setup();
				tpmssetup.MainForm = this.MainForm;
				tpmssetup.CF_pluginInit();

				returnvalue = tpmssetup.ShowDialog();
				if(returnvalue == DialogResult.OK)
				{
					// Reloads plugin configuration.
					loadSettings();
					LoadValues();
                    // reload skin configuration because of the possibilty, 4 than 5 tires are now selected
                    CF_localskinsetup();
				}
				tpmssetup.Close();
				tpmssetup = null;
			}
			catch(Exception errmsg) { CFTools.writeModuleLog(errmsg.Message, errmsg.StackTrace,tpms.errLog); }
			return returnvalue;
		}
		public override void CF_pluginPause()
		{
			// This is called by the system when it pauses all audio.
		}
		public override void CF_pluginResume()
		{
			// This is called by the system when it resumes all audio.
		}
		public override void CF_pluginCommand(string command, string param1, string param2)
		{
			// Used for plugin to plugin communication.  Parameters can be passed into CF_Main_systemCommands 
			// with CF_Actions.PLUGIN, plugin name, plugin command, and a command parameter.
		}
		public override string CF_pluginData(string command, string param)
		{
        	// Used for retrieving information from plugins. You can run CF_getPluginData with a plugin name, 
			// command, and parameter to retrieve information from other plugins running on the system.
			string retvalue = "";
			byte tnr = 1;
            string strPressureDimension = string.Empty;
            string strTemperatureDimension = string.Empty;

			// param must be the converted tire number (1 to 5)
            if (param.IndexOf(".") != -1)
            {
                byte.TryParse(param.Split('.')[0].Trim(),out tnr);
            }
            else
            {
                try { tnr = Convert.ToByte(param); }
                catch { tnr = 0; };
            }

            //================================================================================================
            // This routine only looks for the absolute tire sensor number, not the current mounting position!
            //================================================================================================

			if (tnr > 0)
			{
				switch (command.Trim().ToUpper())
                {
                    #region GETPRESSURE
                    case "GETPRESSURE":
                        if (param.IndexOf(".") != -1)  strPressureDimension = param.Split('.')[1].ToUpper().Trim();
                                                
                        switch (strPressureDimension)
                        {
                            case "BAR":
                                retvalue = String.Format("{0:F1} ", (Tire[tnr - 1].Pressure * 2.5 - 100) / 100) + "Bar";
                                break;
                            case "PSI":
                                retvalue = String.Format("{0:F0} ", ((Tire[tnr - 1].Pressure * 2.5) - 100) * 0.145) + "PSI";
                                break;
                            case "KPA":
                                retvalue = String.Format("{0:F1} ", (Tire[tnr - 1].Pressure * 2.5) - 100) + "KPa";
                                break;
                            default:
                                retvalue = Convert.ToString(Tire[tnr - 1].Pressure);
                                break;
                        }
						break;
                    #endregion
                    #region GETTEMPERATURE
                    case "GETTEMPERATURE":
                        if (param.IndexOf(".") != -1)  strTemperatureDimension = param.Split('.')[1].ToUpper().Trim();
                     
                        switch (strTemperatureDimension)
                        {
                            case "C":
                                if (Tire[tnr - 1].Temperature == 0xFF) retvalue = "--" + "C";
				                if (Tire[tnr - 1].Temperature <= 40) 
                                    retvalue = String.Format("{0:F0}", 0) + "C";
                                else
								    retvalue =  String.Format("{0:F0}", Tire[tnr - 1].Pressure - 40) + "C";

					            break;
				
                            case "F":
                                if (Tire[tnr - 1].Temperature == 0xFF) retvalue = "--" + "F";
                                if (Tire[tnr - 1].Temperature <= 40) 
                                    retvalue = String.Format("{0:F0}", 0) + "F";
                                else
                                    retvalue = String.Format("{0:F0}", ((Tire[tnr - 1].Temperature - 40) * 9 / 5) + 32) + "F";

					            break;

                            default:
                                retvalue = Convert.ToString(Tire[tnr - 1].Temperature);
                                break;
                        }
						break;
                    #endregion GETTEMPERATURE

                    case "GETBATTERY":
						retvalue = Convert.ToString(Tire[tnr-1].Battery);
						break;

					case "GETREASON":
						retvalue = Convert.ToString(Tire[tnr-1].TxReason);
						break;

					case "ISALERT":
						retvalue = Convert.ToString(Tire[tnr-1].Alert);
						break;
				}
			}
			return retvalue;
		}
		public void loadSettings()
		{
            string strLng = this.CF_getConfigSetting(CF_ConfigSettings.Language);

			// Reloads plugin settings.
			//================================================================
			#region CONFIGURATION LOADING ...
			//================================================================
			try
			{
				this.CF_loadConfig(pluginCfg);
				configxml.Load(pluginCfg);
			}

#if DEVELOP
            catch(Exception errmsg) { CFTools.writeError("Error in CONFIG.XML found", errmsg.Message); }
#else
			catch(Exception errmsg) { CFTools.writeModuleLog("Error in CONFIG.XML found", errmsg.Message,tpms.errLog ); }
#endif
			#endregion
		
			if (Assembly.GetExecutingAssembly().GetName().Version.ToString().CompareTo(this.pluginConfig.readPluginField("/APPCONFIG/VERSION")) != 0)
			{
				configxml.SelectSingleNode("/APPCONFIG/VERSION").InnerText = Assembly.GetExecutingAssembly().GetName().Version.ToString();
				configxml.Save(pluginCfg);
				this.CF_loadConfig(pluginCfg);
			}
			try { useDebug = Boolean.Parse(this.pluginConfig.readPluginField("/APPCONFIG/USEDEBUG"));} 
			catch {useDebug = false;}
			if (useDebug ==true) CFTools.writeModuleLog("START", errLog);	
			

			//================================================================
			#region LANGUAGE SYNCHRONISATION
			//================================================================
			try
			{
				if (!File.Exists(pluginFolder + "/languages/" + strLng + ".xml")) strLng = "English";
				this.CF_loadLang(pluginFolder + "/languages/" + strLng + ".xml");
				languagexml.Load(pluginFolder + "/languages/" + strLng + ".xml");
			}
			catch (Exception errmsg) {CFTools.writeModuleLog("Error in language file found", errmsg.Message,tpms.errLog);}
            
			#endregion language synchronisation
			//================================================================
			#region SKIN SYNCHRONISATION
			//================================================================
			try
			{
				// if someone is using a skin with a non existing plugin specific skin file, use Onyx WS Night
				//
				if(!Directory.Exists(Application.StartupPath + "/" + pluginFolder +"/skins/" + this.CF_getConfigSetting(CF_ConfigSettings.Skin)))
					this.currentSkin = "Onyx WS Night";
                else
                    this.currentSkin = this.CF_getConfigSetting(CF_ConfigSettings.Skin);
				this.CF_pluginDisplayName = this.pluginLang.readPluginField("/APPLANG/DISPLAYNAME");
			}
			catch (Exception errmsg) { CFTools.writeModuleLog("TPMS - Error in skin loading found", errmsg.Message,tpms.errLog); }
			#endregion

			if (useDebug ==true) CFTools.writeModuleLog("END", errLog);			
		}
		public override void CF_pluginInit()
		{
        	if (useDebug ==true) CFTools.writeModuleLog("START", tpms.errLog);	
			try
			{
				configxml = new XmlDocument();
				languagexml = new XmlDocument();

				this.loadSettings();
				//----------------------------------------------------------------
				#region SET DEFAULT VALUES
				// Actions that must execute when the program starts.
				//----------------------------------------------------------------

				for (int n=1; n<=5; n++)
				{
					Tire[n-1].Pressure = 0xFF;
					Tire[n-1].Temperature = 0xFF;
					Tire[n-1].Battery = 0;
					Tire[n-1].TxReason = 0;
					Tire[n-1].Alert = false;
					Tire[n-1].IsSilent = false;
				}

				checkTimer = new System.Windows.Forms.Timer();
				flashTimer = new System.Windows.Forms.Timer();

				LoadValues();
				#endregion

				//----------------------------------------------------------------
				// configure skin layout
				//----------------------------------------------------------------
				this.CF_localskinsetup();

                //---------------------------------------------------------------------------
                // add event handlers for keyboard and power mode change
                // many thanks to DEJAN for this idea
                //---------------------------------------------------------------------------
                this.CF_Event_powerModeChanged += new Microsoft.Win32.PowerModeChangedEventHandler(TPMS_CF_Event_powerModeChanged);


			}
            catch (Exception errmsg) { CFTools.writeModuleLog("TPMS - " + errmsg.Message, errmsg.StackTrace, tpms.errLog); }
			finally
			{
				// create event to detect application is loaded
				this.CF_Event_applicationLoaded += new EventHandler(TPMS_Loaded);
			}
			if (tpms.useDebug ==true) CFTools.writeModuleLog("END", tpms.errLog);
		}
		public override void CF_localskinsetup()
		{
            int maxTires = 4;
            if (Use5Tires) maxTires = 5;

			//  This is called to setup the skin.  This will usually be called  in CF_pluginInit.  It will also called by the system when
			//  the resolution has been changed.
            if (tpms.useDebug == true) CFTools.writeModuleLog("CF_localskinsetup - START",tpms.errLog);
			try
			{
				this.CF_clearControls();

                //  Initializes the skin with the specified skin file path, if it exists.  Otherwise load the default skin.
				this.CF_initSkin(pluginFolder + "/skins/" + this.currentSkin + "/skin.xml");
                
				//  Loads the skin images.
                if (Use5Tires)
                {
                    this.CF_loadImages(pluginFolder + "/skins/" + this.currentSkin + "/tpms5_off.png",
                                       pluginFolder + "/skins/" + this.currentSkin + "/tpms5_down.png");
                }
                else 
                {
                    this.CF_loadImages(pluginFolder + "/skins/" + this.currentSkin + "/tpms_off.png",
                                       pluginFolder + "/skins/" + this.currentSkin + "/tpms_down.png");
                }
                
				// Define the reduced size of this plugin
				this.Bounds = this.CF_createRect(
					Int32.Parse(pluginSkinReader.readPluginField("/SKIN/TPMS/X")),
					Int32.Parse(pluginSkinReader.readPluginField("/SKIN/TPMS/Y")),
					Int32.Parse(pluginSkinReader.readPluginField("/SKIN/TPMS/WIDTH")),
					Int32.Parse(pluginSkinReader.readPluginField("/SKIN/TPMS/HEIGHT")));
                
				//  Creates plugin buttons
				this.CF_createButton("SILENT", this.pluginLang.readPluginField("/APPLANG/TPMS/BUTTONS/SILENT"), new MouseEventHandler(Btn_Click));


				for (int i = 1; i <= maxTires; i++)
				{
					this.CF_createButton("TIRE" + Convert.ToString(i), "", new MouseEventHandler(Btn_Click));
					this.CF_createPictureBox("BATTERY" + Convert.ToString(i));
					this.CF_createLabel("TMSG" + Convert.ToString(i));
				}
                
				//  Creates plugin warning symbol
				this.CF_createPictureBox("ALERT");
                
				//  Creates plugin labels.
				this.CF_createLabel("ALERTMSG");
				this.CF_createLabel("TITLE");
				this.CF_updateText("TITLE", this.pluginLang.readPluginField("/APPLANG/DISPLAYNAME"));
				this.CF_createLabel("VERSION");
				this.CF_updateText("VERSION", this.pluginLang.readPluginField("APPLANG/VERSIONNAME") + Assembly.GetExecutingAssembly().GetName().Version.ToString());									
			}
			catch(Exception errmsg) { CFTools.writeModuleLog("TPMS - <CF_localskinsetup>: " + errmsg.Message, errmsg.StackTrace,tpms.errLog); }

            if (tpms.useDebug == true) CFTools.writeModuleLog("CF_localskinsetup - END",tpms.errLog);
		}
		#endregion

        private void TPMS_CF_Event_powerModeChanged(object sender, Microsoft.Win32.PowerModeChangedEventArgs e)
        {
            if (e.Mode.ToString() == "Suspend")
            {
                // Die Ausfhrung des Betriebssystems wird unterbrochen
                if (useDebug == true) { CFTools.writeModuleLog("CF_Event_powerModeChanged - Mode: Suspend", tpms.errLog); }
                checkTimer.Stop();
            }
            else if (e.Mode.ToString() == "Resume")
            {
                // Das Betriebssystem steht unmittelbar davor, aus dem Ruhezustand fortgesetzt zu werden. 
                if (useDebug == true) { CFTools.writeModuleLog("CF_Event_powerModeChanged - Mode: Resume", tpms.errLog); }
                loadSettings();
                LoadValues();
                if (useDebug == true) {if (device == null) CFTools.writeModuleLog("TPMS device IS NULL", tpms.errLog);}
                checkTimer.Start();
            }
        }
	
	}
}

	

	

	
