using System;
using System.Runtime.InteropServices;
using System.IO;
using System.Windows.Forms;



namespace tpms
{
	public class ModHIDCom
	{
		//******************************************************************************
		// User-defined types for API calls, listed alphabetically
		//******************************************************************************
		public struct HIDD_ATTRIBUTES
		{
			public uint Size;
			public ushort VendorID;
			public ushort ProductID;
			public ushort VersionNumber;
		}

		public struct GUID
		{
			public uint Data1;
			public ushort Data2;
			public ushort Data3;
			[MarshalAs(UnmanagedType.ByValArray, SizeConst=8)]
			public byte[] Data4;

			public static GUID Empty
			{
				get
				{
					GUID g = new GUID();
					g.Data1 = 0;
					g.Data2 = 0;
					g.Data3 = 0;
					g.Data4 = new byte[8];
					for(int i = 0; i < 8; ++i) g.Data4[i] = 0;
					return g;
				}
			}
		}

		// Windows 98 DDK documentation is incomplete.
		// Use the structure defined in hidpi.h
		public struct HIDP_CAPS
		{
			public short Usage;
			public short UsagePage;
			public ushort InputReportByteLength;
			public ushort OutputReportByteLength;
			public ushort FeatureReportByteLength;
			[MarshalAs(UnmanagedType.ByValArray, SizeConst=17)]
			public ushort[] Reserved;
			public ushort NumberLinkCollectionNodes;
			public ushort NumberInputButtonCaps;
			public ushort NumberInputValueCaps;
			public ushort NumberInputDataIndices;
			public ushort NumberOutputButtonCaps;
			public ushort NumberOutputValueCaps;
			public ushort NumberOutputDataIndices;
			public ushort NumberFeatureButtonCaps;
			public ushort NumberFeatureValueCaps;
			public ushort NumberFeatureDataIndices;
			
		}
		    
		[StructLayout(LayoutKind.Sequential)]
		public unsafe struct OVERLAPPED
		{
			public uint* Internal;
			public uint* InternalHigh;
			public uint Offset;
			public uint OffsetHigh;
			public IntPtr hEvent;
		}
		
		[StructLayout(LayoutKind.Sequential)]
		public struct SECURITY_ATTRIBUTES
		{
			public int nLength;
			public IntPtr lpSecurityDescriptor;
			public int bInheritHandle;
		}

		protected struct DeviceInterfaceData
		{
			public int cbSize;
			public Guid InterfaceClassGuid;
			public int Flags;
			public int Reserved;
		}

		public HIDP_CAPS Capabilities;
		public uint DetailData;
		public byte DetailDataBuffer;
		public HIDD_ATTRIBUTES DeviceAttributes;
		public string DevicePathName;
		public IntPtr DeviceInfoSet;
		public IntPtr EventObject;
		public IntPtr HIDHandle;
		public IntPtr ReadHandle;
		public OVERLAPPED HIDOverlapped;
		public bool LastDevice;
		protected static DeviceInterfaceData MyDeviceInterfaceData = new DeviceInterfaceData();
		public IntPtr PreparsedData;
		
		public int Result;
		public SECURITY_ATTRIBUTES Security;

		public static bool HidDeviceDetected;
		public static byte[] HidOutputReportData = new byte[7];
		public static byte[] HidInputReportData;

		// Set these to match the values in the device's firmware.
		public const short MyVendorID = 0x0;
		public const short MyProductID = 0x1;

		//******************************************************************************
		// API constants, listed alphabetically
		//******************************************************************************
		// From setupapi.h
		const short		DIGCF_PRESENT			= 0x2;
		const short		DIGCF_DEVICEINTERFACE	= 0x10;
		const int		FILE_FLAG_OVERLAPPED	= 0x40000000;
		const uint		FILE_SHARE_READ			= 0x1;
		const uint		FILE_SHARE_WRITE		= 0x2;
		const uint		GENERIC_READ			= 0x80000000;
		const uint		GENERIC_WRITE			= 0x40000000;
		const uint		OPEN_EXISTING			= 3;
		const int		WAIT_TIMEOUT			= 0x102;
		const short		WAIT_OBJECT_0			= 0;

		// ******************************************************************************
		// API functions KERNEL32.DLL
		// ******************************************************************************
		
		[DllImport("kernel32.dll", SetLastError = true)]
		private static extern int CancelIo(IntPtr hFile);

		[DllImport("kernel32.dll", SetLastError = true)]
		private static extern int CloseHandle(IntPtr hObject);

		[DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Auto)]
		private static extern IntPtr CreateEvent(ref SECURITY_ATTRIBUTES SecurityAttributes,
			int bManualReset, int bInitialState, string lpName);

		[DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Auto)]
		private static extern IntPtr CreateFile(string lpFileName, uint dwDesiredAccess, 
			uint dwShareMode, ref SECURITY_ATTRIBUTES lpSecurityAttributes, uint dwCreationDisposition,
			uint dwFlagsAndAttributes, IntPtr hTemplateFile);

		[DllImport("kernel32.dll", SetLastError = true)]
		private static extern unsafe int ReadFile(IntPtr hFile, void* lpBuffer, uint nNumberOfBytesToRead,
			uint* lpNumberOfBytesRead, ref OVERLAPPED lpOverlapped);

		[DllImport("kernel32.dll", SetLastError = true)]
		private static extern int WaitForSingleObject(IntPtr hHandle, uint dwMilliseconds);

		[DllImport("kernel32.dll", SetLastError = true)]
		private static extern unsafe int WriteFile(IntPtr hFile, void* lpBuffer, uint nNumberOfBytesToWrite,
			uint* lpNumberOfBytesWritten, IntPtr lpOverlapped /*ref OVERLAPPED lpOverlapped*/);

		[DllImport("kernel32.dll", SetLastError = true)]
		private static extern bool ResetEvent(IntPtr hEvent);


		// ******************************************************************************
		// API functions HID.DLL
		// ******************************************************************************
		[DllImport("hid.dll")]
		static public extern int HidD_FlushQueue(IntPtr HidDeviceObject);
		[DllImport("hid.dll")]
		static public extern int HidD_FreePreparsedData(IntPtr PreparsedData);
		[DllImport("hid.dll")]
		static public extern int HidD_GetAttributes(IntPtr HidDeviceObject, ref HIDD_ATTRIBUTES Attributes);
		
		//Step1: Obtaining the Device Interface Guid
		[DllImport("hid.dll",	SetLastError = true)]	protected static extern void HidD_GetHidGuid(out Guid HidGuid);
		//static public extern int HidP_GetValueCaps(int ReportType, ref byte ValueCaps, ref short ValueCapsLength, IntPtr PreparsedData);
		[DllImport("setupapi.dll", SetLastError = true)] protected static extern int SetupDiDestroyDeviceInfoList(IntPtr DeviceInfoSet);
		// requesting a pointer to a device Information Set
		[DllImport("setupapi.dll", SetLastError = true)] protected static extern IntPtr SetupDiGetClassDevs(ref Guid ClassGuid, [MarshalAs(UnmanagedType.LPStr)] string Enumerator,	IntPtr hwndParent, uint Flags);



		[DllImport("hid.dll")]
		static public extern int HidD_GetPreparsedData(IntPtr HidDeviceObject, ref IntPtr PreparsedData);
		[DllImport("hid.dll")]
		static public extern int HidP_GetCaps(IntPtr PreparsedData, ref HIDP_CAPS Capabilities);
		[DllImport("hid.dll")]
		static public extern unsafe int HidD_GetFeature(IntPtr HidDeviceObject, void* lpReportBuffer, uint ReportBufferLength);
		[DllImport("hid.dll")]
		static public extern unsafe int HidD_GetInputReport(IntPtr HidDeviceObject, void* lpReportBuffer, uint ReportBufferLength);
		[DllImport("hid.dll")]
		static public extern unsafe int HidD_GetNumInputBuffers(IntPtr HidDeviceObject, uint* NumberBuffers);
		[DllImport("hid.dll")]
		static public extern unsafe int HidD_SetFeature(IntPtr HidDeviceObject, void* lpReportBuffer, uint ReportBufferLength);
		[DllImport("hid.dll")]
		static public extern int HidD_SetNumInputBuffers(IntPtr HidDeviceObject, uint NumberBuffers);
		[DllImport("hid.dll")]
		static public extern unsafe int HidD_SetOutputReport(IntPtr HidDeviceObject, void* lpReportBuffer, uint ReportBufferLength);
		//[DllImport("hid.dll")]
		

		[DllImport("setupapi.dll", SetLastError = true)]
		private static extern int SetupDiEnumDeviceInterfaces(IntPtr DeviceInfoSet,
			IntPtr DeviceInfoData, ref GUID InterfaceClassGuid, uint MemberIndex,
			ref DeviceInterfaceData DeviceInterfaceData);

		//Step3: requesting a structure containing the Device Path Name
		[DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Auto)]
		private static extern unsafe int SetupDiGetDeviceInterfaceDetail(IntPtr DeviceInfoSet,
			ref DeviceInterfaceData oInterfaceData, void* oInterfaceDetailData,
			uint DeviceInterfacedetailDatasize, uint* RequiredSize, IntPtr oDeviceInfoData);

		[DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Auto)]
		public static extern unsafe int SetupDiClassGuidsFromName(string ClassN, ref GUID guids,
			uint ClassNameSize, uint* ReqSize);
			
		/*****************************************************************************
		*  CODE AREA
		******************************************************************************/

		private void GetDeviceCapabilities()
		{
			// Get the pointer to the buffer containing information about the device's capabilities.
			// Preparsed Data is a pointer to a routine-allocated buffer.
			HidD_GetPreparsedData(HIDHandle, ref PreparsedData);
			// Find out the device's capabilities.
			HidP_GetCaps(PreparsedData, ref Capabilities);
			
			LogIt("GetDeviceCapabilities");
		}

		private void PrepareForOverlappedTransfer()
		{		
			if(EventObject == IntPtr.Zero) EventObject = CreateEvent(ref Security, 0, 0, "");
			// Set the members of the overlapped structure.
			HIDOverlapped.Offset = 0;
			HIDOverlapped.OffsetHigh = 0;
			HIDOverlapped.hEvent = EventObject;
			
			LogIt("PrepareForOverlappedTransfer");
		}

		public void HidShutdown()
		{
			//Close the open handles to the device.
			CloseHandle(HIDHandle);
			CloseHandle(ReadHandle);

			LogIt("HidShutdown");
		}

		public unsafe void HidWriteReport()
		{
			//Send data to the device.
			uint numWritten = 0;
			uint* pNumWritten = &numWritten;
			byte[] SendBuffer = new byte[Capabilities.OutputReportByteLength - 1];
		
			// The first byte is the Report ID
			SendBuffer[0] = 0;
		
			// The next bytes are data
			for(int Count = 1; Count < (Capabilities.OutputReportByteLength - 1); Count++)
				SendBuffer[Count] = HidOutputReportData[Count - 1];
	
			//Sends The report to the device.
			fixed(byte* buffer = SendBuffer)
			{
				WriteFile(HIDHandle, buffer, (uint)Capabilities.OutputReportByteLength, pNumWritten, IntPtr.Zero);
			}

			LogIt("HidWriteReport");
		}

		public unsafe void HidReadReport()
		{
			// Read data from the device.
			//object Count;
			uint numRead = 0;
			uint* pNumRead = &numRead;

			// Allocate a buffer for the report.
			// Byte 0 is the report ID.
			byte[] ReadBuffer = new byte[Capabilities.InputReportByteLength - 1];
			
			// Do an overlapped ReadFile.
			// The function returns immediately, even if the data hasn't been received yet.
			// Get the report in ReadBuffer.
			fixed(byte* buffer = ReadBuffer)
			{
				ReadFile(ReadHandle, buffer, (uint)Capabilities.InputReportByteLength, pNumRead, ref HIDOverlapped);
		
				//Returns when ReadFile has received the requested amount of data or on timeout (1000 mS).
				Result = WaitForSingleObject(EventObject, 1000);
		
				//Find out if ReadFile completed or timeout.
				switch(Result)
				{
					case WAIT_OBJECT_0:
						//ReadFile has completed.
						Array.Copy(ReadBuffer, HidInputReportData, numRead);
						break;
					case WAIT_TIMEOUT:
						// Timeout! Cancel the operation.
						Array.Copy(ReadBuffer, HidInputReportData, numRead);
						// Cancels the ReadFile
						CancelIo(ReadHandle);
						// The timeout may have been because the device was removed, so close any open handles and
						// set HidDeviceDetected=False to cause the application to look for the device on the next attempt.
						CloseHandle(HIDHandle);
						CloseHandle(ReadHandle);
						HidDeviceDetected = false;
						break;
					default:
						Array.Copy(ReadBuffer, HidInputReportData, numRead);
						HidDeviceDetected = false;
						break;
				}
		
				// Sets the event object in the overlapped structure to non-signaled.
				ResetEvent(EventObject);
			}

			LogIt("HidReadReport");
		}

		/*
		public static HIDDevice FindDevice(int nVid, int nPid, Type oType)
		{
			string strPath = string.Empty;
			string strSearch = string.Format("vid_{0:x4}&pid_{1:x4}", nVid, nPid); // first, build the path search string
			Guid gHid;
			HidD_GetHidGuid(out gHid);	// next, get the GUID from Windows that it uses to represent the HID USB interface
			IntPtr hInfoSet = SetupDiGetClassDevs(ref gHid, null, IntPtr.Zero, DIGCF_DEVICEINTERFACE | DIGCF_PRESENT);	// this gets a list of all HID devices currently connected to the computer (InfoSet)
			try
			{
				DeviceInterfaceData oInterface = new DeviceInterfaceData();	// build up a device interface data block
				oInterface.Size = Marshal.SizeOf(oInterface);
				// Now iterate through the InfoSet memory block assigned within Windows in the call to SetupDiGetClassDevs
				// to get device details for each device connected
				int nIndex = 0;
				while (SetupDiEnumDeviceInterfaces(hInfoSet, 0, ref gHid, (uint)nIndex, ref oInterface))	// this gets the device interface information for a device at index 'nIndex' in the memory block
				{
					string strDevicePath = GetDevicePath(hInfoSet, ref oInterface);	// get the device path (see helper method 'GetDevicePath')
					if (strDevicePath.IndexOf(strSearch) >= 0)	// do a string search, if we find the VID/PID string then we found our device!
					{
						HIDDevice oNewDevice = (HIDDevice)Activator.CreateInstance(oType);	// create an instance of the class for this device
						oNewDevice.Initialise(strDevicePath);	// initialise it with the device path
						return oNewDevice;	// and return it
					}
					nIndex++;	// if we get here, we didn't find our device. So move on to the next one.
				}
			}
			finally
			{
				// Before we go, we have to free up the InfoSet memory reserved by SetupDiGetClassDevs
				SetupDiDestroyDeviceInfoList(hInfoSet);
			}
			return null;	// oops, didn't find our device
		}
		*/

		public unsafe bool FindTheHid()
		{ 
			// Makes a series of API calls to locate the desired HID-class device.
			// Returns True if the device is detected, False if not detected.
			bool res = false; // default return value
			
			string strPath = string.Empty;
			string strSearch = string.Format("vid_{0:x4}&pid_{1:x4}", MyVendorID, MyProductID); // first, build the path search string
			string strDevicePath = string.Empty;

			Guid gHid;
			
			//Get the GUID for all system HIDs.
			// next, get the Guid from Windows that it uses to represent the HID USB interface
			HidD_GetHidGuid(out gHid);	


			// this gets a list of all HID devices currently connected to the computer (InfoSet)
			IntPtr hInfoSet = SetupDiGetClassDevs(ref gHid, null, IntPtr.Zero, DIGCF_DEVICEINTERFACE | DIGCF_PRESENT);	

			// Values for SECURITY_ATTRIBUTES structure:
			Security.lpSecurityDescriptor = IntPtr.Zero;
			Security.bInheritHandle = 1;
			Security.nLength = System.Runtime.InteropServices.Marshal.SizeOf(Security);
			
			HidDeviceDetected = false;

			try
			{
				DeviceInterfaceData oInterface = new DeviceInterfaceData();	// build up a device interface data block
				oInterface.cbSize = Marshal.SizeOf(oInterface);
				// Now iterate through the InfoSet memory block assigned within Windows in the call to SetupDiGetClassDevs
				// to get device details for each device connected
				int nIndex = 0;
				while (SetupDiEnumDeviceInterfaces(hInfoSet, 0, ref gHid, (uint)nIndex, ref oInterface))	// this gets the device interface information for a device at index 'nIndex' in the memory block
				{
					strDevicePath = GetDevicePath(hInfoSet, ref oInterface);	// get the device path (see helper method 'GetDevicePath')
					if (strDevicePath.IndexOf(strSearch) >= 0)	// do a string search, if we find the VID/PID string then we found our device!
					{
						// now we have found the TPMS device
						HidDeviceDetected = true;
						// Get the handle that enables reading and writing to the device.
						HIDHandle = CreateFile(strDevicePath, (GENERIC_READ | GENERIC_WRITE), (FILE_SHARE_READ | FILE_SHARE_WRITE), ref Security, OPEN_EXISTING, 0, IntPtr.Zero);
					}
				}


				if (!HidDeviceDetected) // sorry, it was not found 
				{
					// Free the memory reserved for the DeviceInfoSet returned by SetupDiGetClassDevs.
					res = CloseHandle(HIDHandle) != 0;
				}
				else
				{
					// Hurra, we found it
					//---------------------------------------------------------------------------------
					// TMS USB Receiver 
					// MyVendorID = 0x0
					// MyProductID = 0x1
					//---------------------------------------------------------------------------------
					res = true;
					LogIt("Device found: " + strDevicePath);
					// Learn the capabilities of the device
					GetDeviceCapabilities();
					// Get another handle for the overlapped ReadFiles
					ReadHandle = CreateFile(strDevicePath, (GENERIC_READ | GENERIC_WRITE), (FILE_SHARE_READ | FILE_SHARE_WRITE), ref Security, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, IntPtr.Zero);
					PrepareForOverlappedTransfer();
				}
			}
			finally
			{
				// Before we go, we have to free up the InfoSet memory reserved by SetupDiGetClassDevs
				SetupDiDestroyDeviceInfoList(hInfoSet);
			}
			return res;
		}
			
		public void LogIt(string txt)
		{
			string filename = Application.StartupPath + "\\plugins\\tpms\\TPMS_log.txt";
			StreamWriter foutfile;
			
			if (File.Exists(filename) == true) foutfile = File.AppendText(filename);
			else foutfile = File.CreateText(filename);
			
			#region Save to files
			if (foutfile != null) 
			{
				foutfile.Write(DateTime.Now.ToString() + ": ");
				foutfile.WriteLine(txt);
			}
			#endregion
					
			// Close the object if it has been created.
			if (foutfile != null) 
			{	
				foutfile.Flush();
				foutfile.Close();
			}
		}
	}
}



