Attribute VB_Name = "ModHIDCom"
Option Explicit

Dim Capabilities As HIDP_CAPS
Dim DetailData As Long
Dim DetailDataBuffer() As Byte
Dim DeviceAttributes As HIDD_ATTRIBUTES
Dim DevicePathName As String
Dim DeviceInfoSet As Long
Dim EventObject As Long
Dim HIDHandle As Long
Dim HIDOverlapped As OVERLAPPED
Dim LastDevice As Boolean
Dim MyDeviceInterfaceDetailData As SP_DEVICE_INTERFACE_DETAIL_DATA
Dim MyDeviceInterfaceData As SP_DEVICE_INTERFACE_DATA
Dim Needed As Long
Dim PreparsedData As Long
Dim ReadHandle As Long
Dim Result As Long
Dim Security As SECURITY_ATTRIBUTES

Public HidDeviceDetected As Boolean
Public HidOutputReportData(1) As Byte
Public HidInputReportData() As Byte

'Set these to match the values in the device's firmware.
Const MyVendorID = &H0
Const MyProductID = &H1

'******************************************************************************
'API constants, listed alphabetically
'******************************************************************************
'From setupapi.h
Const DIGCF_PRESENT = &H2
Const DIGCF_DEVICEINTERFACE = &H10
Const FILE_FLAG_OVERLAPPED = &H40000000
Const FILE_SHARE_READ = &H1
Const FILE_SHARE_WRITE = &H2
Const GENERIC_READ = &H80000000
Const GENERIC_WRITE = &H40000000

Const OPEN_EXISTING = 3
Const WAIT_TIMEOUT = &H102&
Const WAIT_OBJECT_0 = 0

'******************************************************************************
'User-defined types for API calls, listed alphabetically
'******************************************************************************

Type GUID
    Data1 As Long
    Data2 As Integer
    Data3 As Integer
    Data4(7) As Byte
End Type

Type HIDD_ATTRIBUTES
    Size As Long
    VendorID As Integer
    ProductID As Integer
    VersionNumber As Integer
End Type

'Windows 98 DDK documentation is incomplete.
'Use the structure defined in hidpi.h
Type HIDP_CAPS
    Usage As Integer
    UsagePage As Integer
    InputReportByteLength As Integer
    OutputReportByteLength As Integer
    FeatureReportByteLength As Integer
    Reserved(16) As Integer
    NumberLinkCollectionNodes As Integer
    NumberInputButtonCaps As Integer
    NumberInputValueCaps As Integer
    NumberInputDataIndices As Integer
    NumberOutputButtonCaps As Integer
    NumberOutputValueCaps As Integer
    NumberOutputDataIndices As Integer
    NumberFeatureButtonCaps As Integer
    NumberFeatureValueCaps As Integer
    NumberFeatureDataIndices As Integer
End Type

Type OVERLAPPED
    Internal As Long
    InternalHigh As Long
    Offset As Long
    OffsetHigh As Long
    hEvent As Long
End Type

Type SECURITY_ATTRIBUTES
    nLength As Long
    lpSecurityDescriptor As Long
    bInheritHandle As Long
End Type

Type SP_DEVICE_INTERFACE_DATA
   cbSize As Long
   InterfaceClassGuid As GUID
   Flags As Long
   Reserved As Long
End Type

Type SP_DEVICE_INTERFACE_DETAIL_DATA
    cbSize As Long
    DevicePath As Byte
End Type

'******************************************************************************
'API functions
'******************************************************************************

'******************************************************************************
'CancelIo
'Cancels the ReadFile
'Requires the device handle.
'Returns non-zero on success.
'******************************************************************************
Private Declare Function CancelIo _
    Lib "kernel32" _
    (ByVal hFile As Long) _
As Long

Private Declare Function CloseHandle _
    Lib "kernel32" _
    (ByVal hObject As Long) _
As Long

'******************************************************************************
'CreateEvent
'Creates an event object for the overlapped structure used with ReadFile.
'Requires a security attributes structure or null,
'Manual Reset = True (ResetEvent resets the manual reset object to nonsignaled),
'Initial state = True (signaled),
'and event object name (optional)
'Returns a handle to the event object.
'******************************************************************************
Private Declare Function CreateEvent _
    Lib "kernel32" _
    Alias "CreateEventA" _
    (ByRef lpSecurityAttributes As SECURITY_ATTRIBUTES, _
    ByVal bManualReset As Long, _
    ByVal bInitialState As Long, _
    ByVal lpName As String) _
As Long

'******************************************************************************
'CreateFile
'Returns: a handle that enables reading and writing to the device.
'Requires:
'The DevicePathName returned by SetupDiGetDeviceInterfaceDetail.
'******************************************************************************
Private Declare Function CreateFile _
    Lib "kernel32" _
    Alias "CreateFileA" _
    (ByVal lpFileName As String, _
    ByVal dwDesiredAccess As Long, _
    ByVal dwShareMode As Long, _
    ByRef lpSecurityAttributes As SECURITY_ATTRIBUTES, _
    ByVal dwCreationDisposition As Long, _
    ByVal dwFlagsAndAttributes As Long, _
    ByVal hTemplateFile As Long) _
As Long

'******************************************************************************
'HidD_GetAttributes
'Requests information from the device.
'Requires: The handle returned by CreateFile.
'Returns: an HIDD_ATTRIBUTES structure containing
'the Vendor ID, Product ID, and Product Version Number.
'Use this information to determine if the detected device
'is the one we're looking for.
'******************************************************************************
Private Declare Function HidD_GetAttributes _
    Lib "hid.dll" _
    (ByVal HidDeviceObject As Long, _
    ByRef Attributes As HIDD_ATTRIBUTES) _
As Long

'******************************************************************************
'HidD_GetHidGuid
'Get the GUID for all system HIDs.
'Returns: the GUID in HidGuid.
'The routine doesn't return a value in Result but the routine
'is declared as a function for consistency with the other API calls.
'******************************************************************************
Private Declare Function HidD_GetHidGuid _
    Lib "hid.dll" _
    (ByRef HidGuid As GUID) _
As Long

'******************************************************************************
'HidD_GetPreparsedData
'Returns: a pointer to a buffer containing information about the device's capabilities.
'Requires: A handle returned by CreateFile.
'There's no need to access the buffer directly,
'but HidP_GetCaps and other API functions require a pointer to the buffer.
'******************************************************************************
Private Declare Function HidD_GetPreparsedData _
    Lib "hid.dll" _
    (ByVal HidDeviceObject As Long, _
    ByRef PreparsedData As Long) _
As Long

'******************************************************************************
'HidP_GetCaps
'Find out the device's capabilities.
'For standard devices such as joysticks, you can find out the specific
'capabilities of the device.
'For a custom device, the software will probably know what the device is capable of,
'so this call only verifies the information.
'Requires: The pointer to a buffer containing the information.
'The pointer is returned by HidD_GetPreparsedData.
'Returns: a Capabilites structure containing the information.
'******************************************************************************
Private Declare Function HidP_GetCaps _
    Lib "hid.dll" _
    (ByVal PreparsedData As Long, _
    ByRef Capabilities As HIDP_CAPS) _
As Long

'******************************************************************************
'ReadFile
'Returns: the report in ReadBuffer.
'Requires: a device handle returned by CreateFile
'(for overlapped I/O, CreateFile must be called with FILE_FLAG_OVERLAPPED),
'the Input report length in bytes returned by HidP_GetCaps,
'and an overlapped structure whose hEvent member is set to an event object.
'******************************************************************************
Private Declare Function ReadFile _
    Lib "kernel32" _
    (ByVal hFile As Long, _
    ByRef lpBuffer As Byte, _
    ByVal nNumberOfBytesToRead As Long, _
    ByRef lpNumberOfBytesRead As Long, _
    ByRef lpOverlapped As OVERLAPPED) _
As Long

'******************************************************************************
'ResetEvent
'Sets the event object in the overlapped structure to non-signaled.
'Requires a handle to the event object.
'Returns non-zero on success.
'******************************************************************************
Private Declare Function ResetEvent _
    Lib "kernel32" _
    (ByVal hEvent As Long) _
As Long

Private Declare Function RtlMoveMemory _
    Lib "kernel32" _
    (dest As Any, _
    src As Any, _
    ByVal Count As Long) _
As Long

Private Declare Function SetupDiDestroyDeviceInfoList _
    Lib "setupapi.dll" _
    (ByVal DeviceInfoSet As Long) _
As Long

'******************************************************************************
'SetupDiEnumDeviceInterfaces
'On return, MyDeviceInterfaceData contains the handle to a
'SP_DEVICE_INTERFACE_DATA structure for a detected device.
'Requires:
'The DeviceInfoSet returned in SetupDiGetClassDevs.
'The HidGuid returned in GetHidGuid.
'An index to specify a device.
'******************************************************************************
Private Declare Function SetupDiEnumDeviceInterfaces _
    Lib "setupapi.dll" _
    (ByVal DeviceInfoSet As Long, _
    ByVal DeviceInfoData As Long, _
    ByRef InterfaceClassGuid As GUID, _
    ByVal MemberIndex As Long, _
    ByRef DeviceInterfaceData As SP_DEVICE_INTERFACE_DATA) _
As Long

'******************************************************************************
'SetupDiGetClassDevs
'Returns: a handle to a device information set for all installed devices.
'Requires: the HidGuid returned in GetHidGuid.
'******************************************************************************
Private Declare Function SetupDiGetClassDevs _
    Lib "setupapi.dll" _
    Alias "SetupDiGetClassDevsA" _
    (ByRef ClassGuid As GUID, _
    ByVal Enumerator As String, _
    ByVal hwndParent As Long, _
    ByVal Flags As Long) _
As Long

'******************************************************************************
'SetupDiGetDeviceInterfaceDetail
'Returns: an SP_DEVICE_INTERFACE_DETAIL_DATA structure
'containing information about a device.
'To retrieve the information, call this function twice.
'The first time returns the size of the structure in Needed.
'The second time returns a pointer to the data in DeviceInfoSet.
'Requires:
'A DeviceInfoSet returned by SetupDiGetClassDevs and
'an SP_DEVICE_INTERFACE_DATA structure returned by SetupDiEnumDeviceInterfaces.
'*******************************************************************************
Private Declare Function SetupDiGetDeviceInterfaceDetail _
   Lib "setupapi.dll" _
   Alias "SetupDiGetDeviceInterfaceDetailA" _
   (ByVal DeviceInfoSet As Long, _
   ByRef DeviceInterfaceData As SP_DEVICE_INTERFACE_DATA, _
   ByVal DeviceInterfaceDetailData As Long, _
   ByVal DeviceInterfaceDetailDataSize As Long, _
   ByRef RequiredSize As Long, _
   ByVal DeviceInfoData As Long) _
As Long
    
'******************************************************************************
'WaitForSingleObject
'Used with overlapped ReadFile.
'Returns when ReadFile has received the requested amount of data or on timeout.
'Requires an event object created with CreateEvent
'and a timeout value in milliseconds.
'******************************************************************************
Private Declare Function WaitForSingleObject _
    Lib "kernel32" _
    (ByVal hHandle As Long, _
    ByVal dwMilliseconds As Long) _
As Long
    
'******************************************************************************
'WriteFile
'Sends a report to the device.
'Returns: success or failure.
'Requires: the handle returned by CreateFile and
'The output report byte length returned by HidP_GetCaps
'******************************************************************************
Private Declare Function WriteFile _
    Lib "kernel32" _
    (ByVal hFile As Long, _
    ByRef lpBuffer As Byte, _
    ByVal nNumberOfBytesToWrite As Long, _
    ByRef lpNumberOfBytesWritten As Long, _
    ByVal lpOverlapped As Long) _
As Long


Public Function FindTheHid() As Boolean
'Makes a series of API calls to locate the desired HID-class device.
'Returns True if the device is detected, False if not detected.

Dim Count As Integer
Dim GUIDString As String
Dim HidGuid As GUID
Dim MemberIndex As Long

    LastDevice = False
    HidDeviceDetected = False
    
    'Values for SECURITY_ATTRIBUTES structure:
    Security.lpSecurityDescriptor = 0
    Security.bInheritHandle = True
    Security.nLength = Len(Security)
    
    'Get the GUID for all system HIDs.
    Call HidD_GetHidGuid(HidGuid)
    
    'Get the handle to a device information set for all installed devices.
    DeviceInfoSet = SetupDiGetClassDevs(HidGuid, vbNullString, 0, (DIGCF_PRESENT Or DIGCF_DEVICEINTERFACE))
    
    'Begin with 0 and increment until no more devices are detected.
    MemberIndex = 0
    Do
        'The cbSize element of the MyDeviceInterfaceData structure must be set to
        'the structure's size in bytes. The size is 28 bytes.
        
        'Get the MyDeviceInterfaceData contains the handle to a SP_DEVICE_INTERFACE_DATA structure for a detected device
        MyDeviceInterfaceData.cbSize = LenB(MyDeviceInterfaceData)
        Result = SetupDiEnumDeviceInterfaces(DeviceInfoSet, 0, HidGuid, MemberIndex, MyDeviceInterfaceData)
        If Result = 0 Then LastDevice = True
        
        'If a device exists, get the information returned.
        If Result <> 0 Then
    
            'Get the SP_DEVICE_INTERFACE_DETAIL_DATA structure containing information about a device
            Call SetupDiGetDeviceInterfaceDetail(DeviceInfoSet, MyDeviceInterfaceData, 0, 0, Needed, 0)
            DetailData = Needed
    
            'Store the structure's size.
            MyDeviceInterfaceDetailData.cbSize = Len(MyDeviceInterfaceDetailData)
    
            'Use a byte array to allocate memory for the MyDeviceInterfaceDetailData structure
            ReDim DetailDataBuffer(Needed)
    
            'Store cbSize in the first four bytes of the array.
            Call RtlMoveMemory(DetailDataBuffer(0), MyDeviceInterfaceDetailData, 4)
    
            'Call SetupDiGetDeviceInterfaceDetail again.
            'This time, pass the address of the first element of DetailDataBuffer
            'and the returned required buffer size in DetailData.
            Call SetupDiGetDeviceInterfaceDetail(DeviceInfoSet, MyDeviceInterfaceData, VarPtr(DetailDataBuffer(0)), DetailData, Needed, 0)
    
            'Convert the byte array to a string.
            DevicePathName = CStr(DetailDataBuffer())
    
            'Convert to Unicode.
            DevicePathName = StrConv(DevicePathName, vbUnicode)
    
            'Strip cbSize (4 bytes) from the beginning.
            DevicePathName = Right$(DevicePathName, Len(DevicePathName) - 4)
    
            'Get the handle that enables reading and writing to the device.
            HIDHandle = CreateFile(DevicePathName, GENERIC_READ Or GENERIC_WRITE, (FILE_SHARE_READ Or FILE_SHARE_WRITE), Security, OPEN_EXISTING, 0&, 0)
    
            'Now we can find out if it's the device we're looking for.
    
            'Set the Size property to the number of bytes in the structure.
            DeviceAttributes.Size = LenB(DeviceAttributes)
            
            'Reset the VendorID and the ProductID in the structure.
            DeviceAttributes.VendorID = -1
            DeviceAttributes.ProductID = -1
            
            'Get the HIDD_ATTRIBUTES structure containing the Vendor ID, Product ID, and Product Version Number
            Call HidD_GetAttributes(HIDHandle, DeviceAttributes)
    
            'Find out if the device matches the one we're looking for.
            If (DeviceAttributes.VendorID = MyVendorID) And (DeviceAttributes.ProductID = MyProductID) Then
                    'It's the desired device.
                    HidDeviceDetected = True
            Else
                    HidDeviceDetected = False
                    'If it's not the one we want, close its handle.
                    Result = CloseHandle(HIDHandle)
            End If
        End If
        
    'Keep looking until we find the device or there are no more left to examine.
    MemberIndex = MemberIndex + 1
    Loop Until (LastDevice = True) Or (HidDeviceDetected = True)
    
    'Free the memory reserved for the DeviceInfoSet returned by SetupDiGetClassDevs.
    Call SetupDiDestroyDeviceInfoList(DeviceInfoSet)
    
    If HidDeviceDetected = True Then
        FindTheHid = True
        
        'Learn the capabilities of the device
         Call GetDeviceCapabilities
        
        'Get another handle for the overlapped ReadFiles
        ReadHandle = CreateFile(DevicePathName, (GENERIC_READ Or GENERIC_WRITE), (FILE_SHARE_READ Or FILE_SHARE_WRITE), Security, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0)
        Call PrepareForOverlappedTransfer
    Else
        'Device not found!
    End If

End Function

Private Sub GetDeviceCapabilities()

    'Get the pointer to the buffer containing information about the device's capabilities.
    'Preparsed Data is a pointer to a routine-allocated buffer.
    Call HidD_GetPreparsedData(HIDHandle, PreparsedData)
    
    'Find out the device's capabilities.
    Call HidP_GetCaps(PreparsedData, Capabilities)

End Sub

Private Sub PrepareForOverlappedTransfer()

    If EventObject = 0 Then
        'Creates the event object for the overlapped structure used with ReadFile.
        EventObject = CreateEvent(Security, True, True, "")
    End If
        
    'Set the members of the overlapped structure.
    HIDOverlapped.Offset = 0
    HIDOverlapped.OffsetHigh = 0
    HIDOverlapped.hEvent = EventObject

End Sub

Public Sub HidWriteReport()
'Send data to the device.

Dim Count As Integer
Dim NumberOfBytesWritten As Long
Dim SendBuffer() As Byte

    'Prevention - Cleans the buffer
    Call CancelIo(ReadHandle)
    CloseHandle (ReadHandle)
    ReadHandle = CreateFile(DevicePathName, (GENERIC_READ Or GENERIC_WRITE), (FILE_SHARE_READ Or FILE_SHARE_WRITE), Security, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0)

    'The SendBuffer array begins at 0, so subtract 1 from the number of bytes.
    ReDim SendBuffer(Capabilities.OutputReportByteLength - 1)
    
    'The first byte is the Report ID
    SendBuffer(0) = 0
    
    'The next bytes are data
    For Count = 1 To Capabilities.OutputReportByteLength - 1
        SendBuffer(Count) = HidOutputReportData(Count - 1)
    Next Count
    
    NumberOfBytesWritten = 0
    
    'Sends The report to the device.
    Call WriteFile(HIDHandle, SendBuffer(0), CLng(Capabilities.OutputReportByteLength), NumberOfBytesWritten, 0)

End Sub

Public Sub HidReadReport()
'Read data from the device.

Dim NumberOfBytesRead As Long
Dim ReadBuffer() As Byte

    'The ReadBuffer array begins at 0, so subtract 1 from the number of bytes.
    ReDim ReadBuffer(Capabilities.InputReportByteLength - 1)
        
    'Do an overlapped ReadFile.
    'The function returns immediately, even if the data hasn't been received yet.
            
    'Get the report in ReadBuffer.
    Call ReadFile(ReadHandle, ReadBuffer(0), CLng(Capabilities.InputReportByteLength), NumberOfBytesRead, HIDOverlapped)
    
    'Returns when ReadFile has received the requested amount of data or on timeout (1000 mS).
    Result = WaitForSingleObject(EventObject, 1000)
    
    'Find out if ReadFile completed or timeout.
    Select Case Result
        Case WAIT_OBJECT_0
            'ReadFile has completed.
            HidInputReportData() = ReadBuffer()
        
        Case WAIT_TIMEOUT
            'Timeout! Cancel the operation.
            HidInputReportData() = ReadBuffer()
            'Cancels the ReadFile
            Call CancelIo(ReadHandle)
            'The timeout may have been because the device was removed, so close any open handles and
            'set HidDeviceDetected=False to cause the application to look for the device on the next attempt.
            CloseHandle (HIDHandle)
            CloseHandle (ReadHandle)
            HidDeviceDetected = False
            Exit Sub

        Case Else
            HidInputReportData() = ReadBuffer()
            HidDeviceDetected = False
            Exit Sub
    End Select

    Call ResetEvent(EventObject)
    
End Sub



Public Sub HidShutdown()
'Close the open handles to the device.
    
    Call CloseHandle(HIDHandle)
    Call CloseHandle(ReadHandle)

End Sub

